/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.docdb.model;

import java.beans.Transient;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the input to <a>ModifyDBCluster</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ModifyDbClusterRequest extends DocDbRequest implements
        ToCopyableBuilder<ModifyDbClusterRequest.Builder, ModifyDbClusterRequest> {
    private static final SdkField<String> DB_CLUSTER_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DBClusterIdentifier").getter(getter(ModifyDbClusterRequest::dbClusterIdentifier))
            .setter(setter(Builder::dbClusterIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterIdentifier").build())
            .build();

    private static final SdkField<String> NEW_DB_CLUSTER_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NewDBClusterIdentifier").getter(getter(ModifyDbClusterRequest::newDBClusterIdentifier))
            .setter(setter(Builder::newDBClusterIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NewDBClusterIdentifier").build())
            .build();

    private static final SdkField<Boolean> APPLY_IMMEDIATELY_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ApplyImmediately").getter(getter(ModifyDbClusterRequest::applyImmediately))
            .setter(setter(Builder::applyImmediately))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplyImmediately").build()).build();

    private static final SdkField<Integer> BACKUP_RETENTION_PERIOD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("BackupRetentionPeriod").getter(getter(ModifyDbClusterRequest::backupRetentionPeriod))
            .setter(setter(Builder::backupRetentionPeriod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupRetentionPeriod").build())
            .build();

    private static final SdkField<String> DB_CLUSTER_PARAMETER_GROUP_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DBClusterParameterGroupName")
            .getter(getter(ModifyDbClusterRequest::dbClusterParameterGroupName))
            .setter(setter(Builder::dbClusterParameterGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DBClusterParameterGroupName")
                    .build()).build();

    private static final SdkField<List<String>> VPC_SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("VpcSecurityGroupIds")
            .getter(getter(ModifyDbClusterRequest::vpcSecurityGroupIds))
            .setter(setter(Builder::vpcSecurityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcSecurityGroupIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("VpcSecurityGroupId")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("VpcSecurityGroupId").build()).build()).build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Port")
            .getter(getter(ModifyDbClusterRequest::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<String> MASTER_USER_PASSWORD_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MasterUserPassword").getter(getter(ModifyDbClusterRequest::masterUserPassword))
            .setter(setter(Builder::masterUserPassword))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MasterUserPassword").build())
            .build();

    private static final SdkField<String> PREFERRED_BACKUP_WINDOW_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PreferredBackupWindow").getter(getter(ModifyDbClusterRequest::preferredBackupWindow))
            .setter(setter(Builder::preferredBackupWindow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredBackupWindow").build())
            .build();

    private static final SdkField<String> PREFERRED_MAINTENANCE_WINDOW_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PreferredMaintenanceWindow")
            .getter(getter(ModifyDbClusterRequest::preferredMaintenanceWindow))
            .setter(setter(Builder::preferredMaintenanceWindow))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredMaintenanceWindow").build())
            .build();

    private static final SdkField<CloudwatchLogsExportConfiguration> CLOUDWATCH_LOGS_EXPORT_CONFIGURATION_FIELD = SdkField
            .<CloudwatchLogsExportConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("CloudwatchLogsExportConfiguration")
            .getter(getter(ModifyDbClusterRequest::cloudwatchLogsExportConfiguration))
            .setter(setter(Builder::cloudwatchLogsExportConfiguration))
            .constructor(CloudwatchLogsExportConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudwatchLogsExportConfiguration")
                    .build()).build();

    private static final SdkField<String> ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EngineVersion").getter(getter(ModifyDbClusterRequest::engineVersion))
            .setter(setter(Builder::engineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineVersion").build()).build();

    private static final SdkField<Boolean> DELETION_PROTECTION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DeletionProtection").getter(getter(ModifyDbClusterRequest::deletionProtection))
            .setter(setter(Builder::deletionProtection))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeletionProtection").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DB_CLUSTER_IDENTIFIER_FIELD,
            NEW_DB_CLUSTER_IDENTIFIER_FIELD, APPLY_IMMEDIATELY_FIELD, BACKUP_RETENTION_PERIOD_FIELD,
            DB_CLUSTER_PARAMETER_GROUP_NAME_FIELD, VPC_SECURITY_GROUP_IDS_FIELD, PORT_FIELD, MASTER_USER_PASSWORD_FIELD,
            PREFERRED_BACKUP_WINDOW_FIELD, PREFERRED_MAINTENANCE_WINDOW_FIELD, CLOUDWATCH_LOGS_EXPORT_CONFIGURATION_FIELD,
            ENGINE_VERSION_FIELD, DELETION_PROTECTION_FIELD));

    private final String dbClusterIdentifier;

    private final String newDBClusterIdentifier;

    private final Boolean applyImmediately;

    private final Integer backupRetentionPeriod;

    private final String dbClusterParameterGroupName;

    private final List<String> vpcSecurityGroupIds;

    private final Integer port;

    private final String masterUserPassword;

    private final String preferredBackupWindow;

    private final String preferredMaintenanceWindow;

    private final CloudwatchLogsExportConfiguration cloudwatchLogsExportConfiguration;

    private final String engineVersion;

    private final Boolean deletionProtection;

    private ModifyDbClusterRequest(BuilderImpl builder) {
        super(builder);
        this.dbClusterIdentifier = builder.dbClusterIdentifier;
        this.newDBClusterIdentifier = builder.newDBClusterIdentifier;
        this.applyImmediately = builder.applyImmediately;
        this.backupRetentionPeriod = builder.backupRetentionPeriod;
        this.dbClusterParameterGroupName = builder.dbClusterParameterGroupName;
        this.vpcSecurityGroupIds = builder.vpcSecurityGroupIds;
        this.port = builder.port;
        this.masterUserPassword = builder.masterUserPassword;
        this.preferredBackupWindow = builder.preferredBackupWindow;
        this.preferredMaintenanceWindow = builder.preferredMaintenanceWindow;
        this.cloudwatchLogsExportConfiguration = builder.cloudwatchLogsExportConfiguration;
        this.engineVersion = builder.engineVersion;
        this.deletionProtection = builder.deletionProtection;
    }

    /**
     * <p>
     * The cluster identifier for the cluster that is being modified. This parameter is not case sensitive.
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must match the identifier of an existing <code>DBCluster</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The cluster identifier for the cluster that is being modified. This parameter is not case sensitive.</p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Must match the identifier of an existing <code>DBCluster</code>.
     *         </p>
     *         </li>
     */
    public final String dbClusterIdentifier() {
        return dbClusterIdentifier;
    }

    /**
     * <p>
     * The new cluster identifier for the cluster when renaming a cluster. This value is stored as a lowercase string.
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must contain from 1 to 63 letters, numbers, or hyphens.
     * </p>
     * </li>
     * <li>
     * <p>
     * The first character must be a letter.
     * </p>
     * </li>
     * <li>
     * <p>
     * Cannot end with a hyphen or contain two consecutive hyphens.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Example: <code>my-cluster2</code>
     * </p>
     * 
     * @return The new cluster identifier for the cluster when renaming a cluster. This value is stored as a lowercase
     *         string.</p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Must contain from 1 to 63 letters, numbers, or hyphens.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The first character must be a letter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cannot end with a hyphen or contain two consecutive hyphens.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Example: <code>my-cluster2</code>
     */
    public final String newDBClusterIdentifier() {
        return newDBClusterIdentifier;
    }

    /**
     * <p>
     * A value that specifies whether the changes in this request and any pending changes are asynchronously applied as
     * soon as possible, regardless of the <code>PreferredMaintenanceWindow</code> setting for the cluster. If this
     * parameter is set to <code>false</code>, changes to the cluster are applied during the next maintenance window.
     * </p>
     * <p>
     * The <code>ApplyImmediately</code> parameter affects only the <code>NewDBClusterIdentifier</code> and
     * <code>MasterUserPassword</code> values. If you set this parameter value to <code>false</code>, the changes to the
     * <code>NewDBClusterIdentifier</code> and <code>MasterUserPassword</code> values are applied during the next
     * maintenance window. All other changes are applied immediately, regardless of the value of the
     * <code>ApplyImmediately</code> parameter.
     * </p>
     * <p>
     * Default: <code>false</code>
     * </p>
     * 
     * @return A value that specifies whether the changes in this request and any pending changes are asynchronously
     *         applied as soon as possible, regardless of the <code>PreferredMaintenanceWindow</code> setting for the
     *         cluster. If this parameter is set to <code>false</code>, changes to the cluster are applied during the
     *         next maintenance window.</p>
     *         <p>
     *         The <code>ApplyImmediately</code> parameter affects only the <code>NewDBClusterIdentifier</code> and
     *         <code>MasterUserPassword</code> values. If you set this parameter value to <code>false</code>, the
     *         changes to the <code>NewDBClusterIdentifier</code> and <code>MasterUserPassword</code> values are applied
     *         during the next maintenance window. All other changes are applied immediately, regardless of the value of
     *         the <code>ApplyImmediately</code> parameter.
     *         </p>
     *         <p>
     *         Default: <code>false</code>
     */
    public final Boolean applyImmediately() {
        return applyImmediately;
    }

    /**
     * <p>
     * The number of days for which automated backups are retained. You must specify a minimum value of 1.
     * </p>
     * <p>
     * Default: 1
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must be a value from 1 to 35.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The number of days for which automated backups are retained. You must specify a minimum value of 1.</p>
     *         <p>
     *         Default: 1
     *         </p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Must be a value from 1 to 35.
     *         </p>
     *         </li>
     */
    public final Integer backupRetentionPeriod() {
        return backupRetentionPeriod;
    }

    /**
     * <p>
     * The name of the cluster parameter group to use for the cluster.
     * </p>
     * 
     * @return The name of the cluster parameter group to use for the cluster.
     */
    public final String dbClusterParameterGroupName() {
        return dbClusterParameterGroupName;
    }

    /**
     * For responses, this returns true if the service returned a value for the VpcSecurityGroupIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVpcSecurityGroupIds() {
        return vpcSecurityGroupIds != null && !(vpcSecurityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of virtual private cloud (VPC) security groups that the cluster will belong to.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVpcSecurityGroupIds} method.
     * </p>
     * 
     * @return A list of virtual private cloud (VPC) security groups that the cluster will belong to.
     */
    public final List<String> vpcSecurityGroupIds() {
        return vpcSecurityGroupIds;
    }

    /**
     * <p>
     * The port number on which the cluster accepts connections.
     * </p>
     * <p>
     * Constraints: Must be a value from <code>1150</code> to <code>65535</code>.
     * </p>
     * <p>
     * Default: The same port as the original cluster.
     * </p>
     * 
     * @return The port number on which the cluster accepts connections.</p>
     *         <p>
     *         Constraints: Must be a value from <code>1150</code> to <code>65535</code>.
     *         </p>
     *         <p>
     *         Default: The same port as the original cluster.
     */
    public final Integer port() {
        return port;
    }

    /**
     * <p>
     * The password for the master database user. This password can contain any printable ASCII character except forward
     * slash (/), double quote ("), or the "at" symbol (@).
     * </p>
     * <p>
     * Constraints: Must contain from 8 to 100 characters.
     * </p>
     * 
     * @return The password for the master database user. This password can contain any printable ASCII character except
     *         forward slash (/), double quote ("), or the "at" symbol (@).</p>
     *         <p>
     *         Constraints: Must contain from 8 to 100 characters.
     */
    public final String masterUserPassword() {
        return masterUserPassword;
    }

    /**
     * <p>
     * The daily time range during which automated backups are created if automated backups are enabled, using the
     * <code>BackupRetentionPeriod</code> parameter.
     * </p>
     * <p>
     * The default is a 30-minute window selected at random from an 8-hour block of time for each Region.
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must be in the format <code>hh24:mi-hh24:mi</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Must be in Universal Coordinated Time (UTC).
     * </p>
     * </li>
     * <li>
     * <p>
     * Must not conflict with the preferred maintenance window.
     * </p>
     * </li>
     * <li>
     * <p>
     * Must be at least 30 minutes.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The daily time range during which automated backups are created if automated backups are enabled, using
     *         the <code>BackupRetentionPeriod</code> parameter. </p>
     *         <p>
     *         The default is a 30-minute window selected at random from an 8-hour block of time for each Region.
     *         </p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Must be in the format <code>hh24:mi-hh24:mi</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must be in Universal Coordinated Time (UTC).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must not conflict with the preferred maintenance window.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must be at least 30 minutes.
     *         </p>
     *         </li>
     */
    public final String preferredBackupWindow() {
        return preferredBackupWindow;
    }

    /**
     * <p>
     * The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).
     * </p>
     * <p>
     * Format: <code>ddd:hh24:mi-ddd:hh24:mi</code>
     * </p>
     * <p>
     * The default is a 30-minute window selected at random from an 8-hour block of time for each Region, occurring on a
     * random day of the week.
     * </p>
     * <p>
     * Valid days: Mon, Tue, Wed, Thu, Fri, Sat, Sun
     * </p>
     * <p>
     * Constraints: Minimum 30-minute window.
     * </p>
     * 
     * @return The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).</p>
     *         <p>
     *         Format: <code>ddd:hh24:mi-ddd:hh24:mi</code>
     *         </p>
     *         <p>
     *         The default is a 30-minute window selected at random from an 8-hour block of time for each Region,
     *         occurring on a random day of the week.
     *         </p>
     *         <p>
     *         Valid days: Mon, Tue, Wed, Thu, Fri, Sat, Sun
     *         </p>
     *         <p>
     *         Constraints: Minimum 30-minute window.
     */
    public final String preferredMaintenanceWindow() {
        return preferredMaintenanceWindow;
    }

    /**
     * <p>
     * The configuration setting for the log types to be enabled for export to Amazon CloudWatch Logs for a specific
     * instance or cluster. The <code>EnableLogTypes</code> and <code>DisableLogTypes</code> arrays determine which logs
     * are exported (or not exported) to CloudWatch Logs.
     * </p>
     * 
     * @return The configuration setting for the log types to be enabled for export to Amazon CloudWatch Logs for a
     *         specific instance or cluster. The <code>EnableLogTypes</code> and <code>DisableLogTypes</code> arrays
     *         determine which logs are exported (or not exported) to CloudWatch Logs.
     */
    public final CloudwatchLogsExportConfiguration cloudwatchLogsExportConfiguration() {
        return cloudwatchLogsExportConfiguration;
    }

    /**
     * <p>
     * The version number of the database engine to which you want to upgrade. Modifying engine version is not supported
     * on Amazon DocumentDB.
     * </p>
     * 
     * @return The version number of the database engine to which you want to upgrade. Modifying engine version is not
     *         supported on Amazon DocumentDB.
     */
    public final String engineVersion() {
        return engineVersion;
    }

    /**
     * <p>
     * Specifies whether this cluster can be deleted. If <code>DeletionProtection</code> is enabled, the cluster cannot
     * be deleted unless it is modified and <code>DeletionProtection</code> is disabled. <code>DeletionProtection</code>
     * protects clusters from being accidentally deleted.
     * </p>
     * 
     * @return Specifies whether this cluster can be deleted. If <code>DeletionProtection</code> is enabled, the cluster
     *         cannot be deleted unless it is modified and <code>DeletionProtection</code> is disabled.
     *         <code>DeletionProtection</code> protects clusters from being accidentally deleted.
     */
    public final Boolean deletionProtection() {
        return deletionProtection;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(newDBClusterIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(applyImmediately());
        hashCode = 31 * hashCode + Objects.hashCode(backupRetentionPeriod());
        hashCode = 31 * hashCode + Objects.hashCode(dbClusterParameterGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(hasVpcSecurityGroupIds() ? vpcSecurityGroupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(masterUserPassword());
        hashCode = 31 * hashCode + Objects.hashCode(preferredBackupWindow());
        hashCode = 31 * hashCode + Objects.hashCode(preferredMaintenanceWindow());
        hashCode = 31 * hashCode + Objects.hashCode(cloudwatchLogsExportConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(engineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(deletionProtection());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ModifyDbClusterRequest)) {
            return false;
        }
        ModifyDbClusterRequest other = (ModifyDbClusterRequest) obj;
        return Objects.equals(dbClusterIdentifier(), other.dbClusterIdentifier())
                && Objects.equals(newDBClusterIdentifier(), other.newDBClusterIdentifier())
                && Objects.equals(applyImmediately(), other.applyImmediately())
                && Objects.equals(backupRetentionPeriod(), other.backupRetentionPeriod())
                && Objects.equals(dbClusterParameterGroupName(), other.dbClusterParameterGroupName())
                && hasVpcSecurityGroupIds() == other.hasVpcSecurityGroupIds()
                && Objects.equals(vpcSecurityGroupIds(), other.vpcSecurityGroupIds()) && Objects.equals(port(), other.port())
                && Objects.equals(masterUserPassword(), other.masterUserPassword())
                && Objects.equals(preferredBackupWindow(), other.preferredBackupWindow())
                && Objects.equals(preferredMaintenanceWindow(), other.preferredMaintenanceWindow())
                && Objects.equals(cloudwatchLogsExportConfiguration(), other.cloudwatchLogsExportConfiguration())
                && Objects.equals(engineVersion(), other.engineVersion())
                && Objects.equals(deletionProtection(), other.deletionProtection());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ModifyDbClusterRequest").add("DBClusterIdentifier", dbClusterIdentifier())
                .add("NewDBClusterIdentifier", newDBClusterIdentifier()).add("ApplyImmediately", applyImmediately())
                .add("BackupRetentionPeriod", backupRetentionPeriod())
                .add("DBClusterParameterGroupName", dbClusterParameterGroupName())
                .add("VpcSecurityGroupIds", hasVpcSecurityGroupIds() ? vpcSecurityGroupIds() : null).add("Port", port())
                .add("MasterUserPassword", masterUserPassword()).add("PreferredBackupWindow", preferredBackupWindow())
                .add("PreferredMaintenanceWindow", preferredMaintenanceWindow())
                .add("CloudwatchLogsExportConfiguration", cloudwatchLogsExportConfiguration())
                .add("EngineVersion", engineVersion()).add("DeletionProtection", deletionProtection()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DBClusterIdentifier":
            return Optional.ofNullable(clazz.cast(dbClusterIdentifier()));
        case "NewDBClusterIdentifier":
            return Optional.ofNullable(clazz.cast(newDBClusterIdentifier()));
        case "ApplyImmediately":
            return Optional.ofNullable(clazz.cast(applyImmediately()));
        case "BackupRetentionPeriod":
            return Optional.ofNullable(clazz.cast(backupRetentionPeriod()));
        case "DBClusterParameterGroupName":
            return Optional.ofNullable(clazz.cast(dbClusterParameterGroupName()));
        case "VpcSecurityGroupIds":
            return Optional.ofNullable(clazz.cast(vpcSecurityGroupIds()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "MasterUserPassword":
            return Optional.ofNullable(clazz.cast(masterUserPassword()));
        case "PreferredBackupWindow":
            return Optional.ofNullable(clazz.cast(preferredBackupWindow()));
        case "PreferredMaintenanceWindow":
            return Optional.ofNullable(clazz.cast(preferredMaintenanceWindow()));
        case "CloudwatchLogsExportConfiguration":
            return Optional.ofNullable(clazz.cast(cloudwatchLogsExportConfiguration()));
        case "EngineVersion":
            return Optional.ofNullable(clazz.cast(engineVersion()));
        case "DeletionProtection":
            return Optional.ofNullable(clazz.cast(deletionProtection()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ModifyDbClusterRequest, T> g) {
        return obj -> g.apply((ModifyDbClusterRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DocDbRequest.Builder, SdkPojo, CopyableBuilder<Builder, ModifyDbClusterRequest> {
        /**
         * <p>
         * The cluster identifier for the cluster that is being modified. This parameter is not case sensitive.
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must match the identifier of an existing <code>DBCluster</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dbClusterIdentifier
         *        The cluster identifier for the cluster that is being modified. This parameter is not case
         *        sensitive.</p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must match the identifier of an existing <code>DBCluster</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterIdentifier(String dbClusterIdentifier);

        /**
         * <p>
         * The new cluster identifier for the cluster when renaming a cluster. This value is stored as a lowercase
         * string.
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must contain from 1 to 63 letters, numbers, or hyphens.
         * </p>
         * </li>
         * <li>
         * <p>
         * The first character must be a letter.
         * </p>
         * </li>
         * <li>
         * <p>
         * Cannot end with a hyphen or contain two consecutive hyphens.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Example: <code>my-cluster2</code>
         * </p>
         * 
         * @param newDBClusterIdentifier
         *        The new cluster identifier for the cluster when renaming a cluster. This value is stored as a
         *        lowercase string.</p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must contain from 1 to 63 letters, numbers, or hyphens.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The first character must be a letter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cannot end with a hyphen or contain two consecutive hyphens.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Example: <code>my-cluster2</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder newDBClusterIdentifier(String newDBClusterIdentifier);

        /**
         * <p>
         * A value that specifies whether the changes in this request and any pending changes are asynchronously applied
         * as soon as possible, regardless of the <code>PreferredMaintenanceWindow</code> setting for the cluster. If
         * this parameter is set to <code>false</code>, changes to the cluster are applied during the next maintenance
         * window.
         * </p>
         * <p>
         * The <code>ApplyImmediately</code> parameter affects only the <code>NewDBClusterIdentifier</code> and
         * <code>MasterUserPassword</code> values. If you set this parameter value to <code>false</code>, the changes to
         * the <code>NewDBClusterIdentifier</code> and <code>MasterUserPassword</code> values are applied during the
         * next maintenance window. All other changes are applied immediately, regardless of the value of the
         * <code>ApplyImmediately</code> parameter.
         * </p>
         * <p>
         * Default: <code>false</code>
         * </p>
         * 
         * @param applyImmediately
         *        A value that specifies whether the changes in this request and any pending changes are asynchronously
         *        applied as soon as possible, regardless of the <code>PreferredMaintenanceWindow</code> setting for the
         *        cluster. If this parameter is set to <code>false</code>, changes to the cluster are applied during the
         *        next maintenance window.</p>
         *        <p>
         *        The <code>ApplyImmediately</code> parameter affects only the <code>NewDBClusterIdentifier</code> and
         *        <code>MasterUserPassword</code> values. If you set this parameter value to <code>false</code>, the
         *        changes to the <code>NewDBClusterIdentifier</code> and <code>MasterUserPassword</code> values are
         *        applied during the next maintenance window. All other changes are applied immediately, regardless of
         *        the value of the <code>ApplyImmediately</code> parameter.
         *        </p>
         *        <p>
         *        Default: <code>false</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applyImmediately(Boolean applyImmediately);

        /**
         * <p>
         * The number of days for which automated backups are retained. You must specify a minimum value of 1.
         * </p>
         * <p>
         * Default: 1
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must be a value from 1 to 35.
         * </p>
         * </li>
         * </ul>
         * 
         * @param backupRetentionPeriod
         *        The number of days for which automated backups are retained. You must specify a minimum value of
         *        1.</p>
         *        <p>
         *        Default: 1
         *        </p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must be a value from 1 to 35.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupRetentionPeriod(Integer backupRetentionPeriod);

        /**
         * <p>
         * The name of the cluster parameter group to use for the cluster.
         * </p>
         * 
         * @param dbClusterParameterGroupName
         *        The name of the cluster parameter group to use for the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbClusterParameterGroupName(String dbClusterParameterGroupName);

        /**
         * <p>
         * A list of virtual private cloud (VPC) security groups that the cluster will belong to.
         * </p>
         * 
         * @param vpcSecurityGroupIds
         *        A list of virtual private cloud (VPC) security groups that the cluster will belong to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSecurityGroupIds(Collection<String> vpcSecurityGroupIds);

        /**
         * <p>
         * A list of virtual private cloud (VPC) security groups that the cluster will belong to.
         * </p>
         * 
         * @param vpcSecurityGroupIds
         *        A list of virtual private cloud (VPC) security groups that the cluster will belong to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcSecurityGroupIds(String... vpcSecurityGroupIds);

        /**
         * <p>
         * The port number on which the cluster accepts connections.
         * </p>
         * <p>
         * Constraints: Must be a value from <code>1150</code> to <code>65535</code>.
         * </p>
         * <p>
         * Default: The same port as the original cluster.
         * </p>
         * 
         * @param port
         *        The port number on which the cluster accepts connections.</p>
         *        <p>
         *        Constraints: Must be a value from <code>1150</code> to <code>65535</code>.
         *        </p>
         *        <p>
         *        Default: The same port as the original cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * The password for the master database user. This password can contain any printable ASCII character except
         * forward slash (/), double quote ("), or the "at" symbol (@).
         * </p>
         * <p>
         * Constraints: Must contain from 8 to 100 characters.
         * </p>
         * 
         * @param masterUserPassword
         *        The password for the master database user. This password can contain any printable ASCII character
         *        except forward slash (/), double quote ("), or the "at" symbol (@).</p>
         *        <p>
         *        Constraints: Must contain from 8 to 100 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterUserPassword(String masterUserPassword);

        /**
         * <p>
         * The daily time range during which automated backups are created if automated backups are enabled, using the
         * <code>BackupRetentionPeriod</code> parameter.
         * </p>
         * <p>
         * The default is a 30-minute window selected at random from an 8-hour block of time for each Region.
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must be in the format <code>hh24:mi-hh24:mi</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Must be in Universal Coordinated Time (UTC).
         * </p>
         * </li>
         * <li>
         * <p>
         * Must not conflict with the preferred maintenance window.
         * </p>
         * </li>
         * <li>
         * <p>
         * Must be at least 30 minutes.
         * </p>
         * </li>
         * </ul>
         * 
         * @param preferredBackupWindow
         *        The daily time range during which automated backups are created if automated backups are enabled,
         *        using the <code>BackupRetentionPeriod</code> parameter. </p>
         *        <p>
         *        The default is a 30-minute window selected at random from an 8-hour block of time for each Region.
         *        </p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must be in the format <code>hh24:mi-hh24:mi</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must be in Universal Coordinated Time (UTC).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must not conflict with the preferred maintenance window.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must be at least 30 minutes.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredBackupWindow(String preferredBackupWindow);

        /**
         * <p>
         * The weekly time range during which system maintenance can occur, in Universal Coordinated Time (UTC).
         * </p>
         * <p>
         * Format: <code>ddd:hh24:mi-ddd:hh24:mi</code>
         * </p>
         * <p>
         * The default is a 30-minute window selected at random from an 8-hour block of time for each Region, occurring
         * on a random day of the week.
         * </p>
         * <p>
         * Valid days: Mon, Tue, Wed, Thu, Fri, Sat, Sun
         * </p>
         * <p>
         * Constraints: Minimum 30-minute window.
         * </p>
         * 
         * @param preferredMaintenanceWindow
         *        The weekly time range during which system maintenance can occur, in Universal Coordinated Time
         *        (UTC).</p>
         *        <p>
         *        Format: <code>ddd:hh24:mi-ddd:hh24:mi</code>
         *        </p>
         *        <p>
         *        The default is a 30-minute window selected at random from an 8-hour block of time for each Region,
         *        occurring on a random day of the week.
         *        </p>
         *        <p>
         *        Valid days: Mon, Tue, Wed, Thu, Fri, Sat, Sun
         *        </p>
         *        <p>
         *        Constraints: Minimum 30-minute window.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredMaintenanceWindow(String preferredMaintenanceWindow);

        /**
         * <p>
         * The configuration setting for the log types to be enabled for export to Amazon CloudWatch Logs for a specific
         * instance or cluster. The <code>EnableLogTypes</code> and <code>DisableLogTypes</code> arrays determine which
         * logs are exported (or not exported) to CloudWatch Logs.
         * </p>
         * 
         * @param cloudwatchLogsExportConfiguration
         *        The configuration setting for the log types to be enabled for export to Amazon CloudWatch Logs for a
         *        specific instance or cluster. The <code>EnableLogTypes</code> and <code>DisableLogTypes</code> arrays
         *        determine which logs are exported (or not exported) to CloudWatch Logs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudwatchLogsExportConfiguration(CloudwatchLogsExportConfiguration cloudwatchLogsExportConfiguration);

        /**
         * <p>
         * The configuration setting for the log types to be enabled for export to Amazon CloudWatch Logs for a specific
         * instance or cluster. The <code>EnableLogTypes</code> and <code>DisableLogTypes</code> arrays determine which
         * logs are exported (or not exported) to CloudWatch Logs.
         * </p>
         * This is a convenience that creates an instance of the {@link CloudwatchLogsExportConfiguration.Builder}
         * avoiding the need to create one manually via {@link CloudwatchLogsExportConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link CloudwatchLogsExportConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #cloudwatchLogsExportConfiguration(CloudwatchLogsExportConfiguration)}.
         * 
         * @param cloudwatchLogsExportConfiguration
         *        a consumer that will call methods on {@link CloudwatchLogsExportConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cloudwatchLogsExportConfiguration(CloudwatchLogsExportConfiguration)
         */
        default Builder cloudwatchLogsExportConfiguration(
                Consumer<CloudwatchLogsExportConfiguration.Builder> cloudwatchLogsExportConfiguration) {
            return cloudwatchLogsExportConfiguration(CloudwatchLogsExportConfiguration.builder()
                    .applyMutation(cloudwatchLogsExportConfiguration).build());
        }

        /**
         * <p>
         * The version number of the database engine to which you want to upgrade. Modifying engine version is not
         * supported on Amazon DocumentDB.
         * </p>
         * 
         * @param engineVersion
         *        The version number of the database engine to which you want to upgrade. Modifying engine version is
         *        not supported on Amazon DocumentDB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineVersion(String engineVersion);

        /**
         * <p>
         * Specifies whether this cluster can be deleted. If <code>DeletionProtection</code> is enabled, the cluster
         * cannot be deleted unless it is modified and <code>DeletionProtection</code> is disabled.
         * <code>DeletionProtection</code> protects clusters from being accidentally deleted.
         * </p>
         * 
         * @param deletionProtection
         *        Specifies whether this cluster can be deleted. If <code>DeletionProtection</code> is enabled, the
         *        cluster cannot be deleted unless it is modified and <code>DeletionProtection</code> is disabled.
         *        <code>DeletionProtection</code> protects clusters from being accidentally deleted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletionProtection(Boolean deletionProtection);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DocDbRequest.BuilderImpl implements Builder {
        private String dbClusterIdentifier;

        private String newDBClusterIdentifier;

        private Boolean applyImmediately;

        private Integer backupRetentionPeriod;

        private String dbClusterParameterGroupName;

        private List<String> vpcSecurityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private Integer port;

        private String masterUserPassword;

        private String preferredBackupWindow;

        private String preferredMaintenanceWindow;

        private CloudwatchLogsExportConfiguration cloudwatchLogsExportConfiguration;

        private String engineVersion;

        private Boolean deletionProtection;

        private BuilderImpl() {
        }

        private BuilderImpl(ModifyDbClusterRequest model) {
            super(model);
            dbClusterIdentifier(model.dbClusterIdentifier);
            newDBClusterIdentifier(model.newDBClusterIdentifier);
            applyImmediately(model.applyImmediately);
            backupRetentionPeriod(model.backupRetentionPeriod);
            dbClusterParameterGroupName(model.dbClusterParameterGroupName);
            vpcSecurityGroupIds(model.vpcSecurityGroupIds);
            port(model.port);
            masterUserPassword(model.masterUserPassword);
            preferredBackupWindow(model.preferredBackupWindow);
            preferredMaintenanceWindow(model.preferredMaintenanceWindow);
            cloudwatchLogsExportConfiguration(model.cloudwatchLogsExportConfiguration);
            engineVersion(model.engineVersion);
            deletionProtection(model.deletionProtection);
        }

        public final String getDbClusterIdentifier() {
            return dbClusterIdentifier;
        }

        public final void setDbClusterIdentifier(String dbClusterIdentifier) {
            this.dbClusterIdentifier = dbClusterIdentifier;
        }

        @Override
        @Transient
        public final Builder dbClusterIdentifier(String dbClusterIdentifier) {
            this.dbClusterIdentifier = dbClusterIdentifier;
            return this;
        }

        public final String getNewDBClusterIdentifier() {
            return newDBClusterIdentifier;
        }

        public final void setNewDBClusterIdentifier(String newDBClusterIdentifier) {
            this.newDBClusterIdentifier = newDBClusterIdentifier;
        }

        @Override
        @Transient
        public final Builder newDBClusterIdentifier(String newDBClusterIdentifier) {
            this.newDBClusterIdentifier = newDBClusterIdentifier;
            return this;
        }

        public final Boolean getApplyImmediately() {
            return applyImmediately;
        }

        public final void setApplyImmediately(Boolean applyImmediately) {
            this.applyImmediately = applyImmediately;
        }

        @Override
        @Transient
        public final Builder applyImmediately(Boolean applyImmediately) {
            this.applyImmediately = applyImmediately;
            return this;
        }

        public final Integer getBackupRetentionPeriod() {
            return backupRetentionPeriod;
        }

        public final void setBackupRetentionPeriod(Integer backupRetentionPeriod) {
            this.backupRetentionPeriod = backupRetentionPeriod;
        }

        @Override
        @Transient
        public final Builder backupRetentionPeriod(Integer backupRetentionPeriod) {
            this.backupRetentionPeriod = backupRetentionPeriod;
            return this;
        }

        public final String getDbClusterParameterGroupName() {
            return dbClusterParameterGroupName;
        }

        public final void setDbClusterParameterGroupName(String dbClusterParameterGroupName) {
            this.dbClusterParameterGroupName = dbClusterParameterGroupName;
        }

        @Override
        @Transient
        public final Builder dbClusterParameterGroupName(String dbClusterParameterGroupName) {
            this.dbClusterParameterGroupName = dbClusterParameterGroupName;
            return this;
        }

        public final Collection<String> getVpcSecurityGroupIds() {
            if (vpcSecurityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return vpcSecurityGroupIds;
        }

        public final void setVpcSecurityGroupIds(Collection<String> vpcSecurityGroupIds) {
            this.vpcSecurityGroupIds = VpcSecurityGroupIdListCopier.copy(vpcSecurityGroupIds);
        }

        @Override
        @Transient
        public final Builder vpcSecurityGroupIds(Collection<String> vpcSecurityGroupIds) {
            this.vpcSecurityGroupIds = VpcSecurityGroupIdListCopier.copy(vpcSecurityGroupIds);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder vpcSecurityGroupIds(String... vpcSecurityGroupIds) {
            vpcSecurityGroupIds(Arrays.asList(vpcSecurityGroupIds));
            return this;
        }

        public final Integer getPort() {
            return port;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        @Override
        @Transient
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final String getMasterUserPassword() {
            return masterUserPassword;
        }

        public final void setMasterUserPassword(String masterUserPassword) {
            this.masterUserPassword = masterUserPassword;
        }

        @Override
        @Transient
        public final Builder masterUserPassword(String masterUserPassword) {
            this.masterUserPassword = masterUserPassword;
            return this;
        }

        public final String getPreferredBackupWindow() {
            return preferredBackupWindow;
        }

        public final void setPreferredBackupWindow(String preferredBackupWindow) {
            this.preferredBackupWindow = preferredBackupWindow;
        }

        @Override
        @Transient
        public final Builder preferredBackupWindow(String preferredBackupWindow) {
            this.preferredBackupWindow = preferredBackupWindow;
            return this;
        }

        public final String getPreferredMaintenanceWindow() {
            return preferredMaintenanceWindow;
        }

        public final void setPreferredMaintenanceWindow(String preferredMaintenanceWindow) {
            this.preferredMaintenanceWindow = preferredMaintenanceWindow;
        }

        @Override
        @Transient
        public final Builder preferredMaintenanceWindow(String preferredMaintenanceWindow) {
            this.preferredMaintenanceWindow = preferredMaintenanceWindow;
            return this;
        }

        public final CloudwatchLogsExportConfiguration.Builder getCloudwatchLogsExportConfiguration() {
            return cloudwatchLogsExportConfiguration != null ? cloudwatchLogsExportConfiguration.toBuilder() : null;
        }

        public final void setCloudwatchLogsExportConfiguration(
                CloudwatchLogsExportConfiguration.BuilderImpl cloudwatchLogsExportConfiguration) {
            this.cloudwatchLogsExportConfiguration = cloudwatchLogsExportConfiguration != null ? cloudwatchLogsExportConfiguration
                    .build() : null;
        }

        @Override
        @Transient
        public final Builder cloudwatchLogsExportConfiguration(CloudwatchLogsExportConfiguration cloudwatchLogsExportConfiguration) {
            this.cloudwatchLogsExportConfiguration = cloudwatchLogsExportConfiguration;
            return this;
        }

        public final String getEngineVersion() {
            return engineVersion;
        }

        public final void setEngineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
        }

        @Override
        @Transient
        public final Builder engineVersion(String engineVersion) {
            this.engineVersion = engineVersion;
            return this;
        }

        public final Boolean getDeletionProtection() {
            return deletionProtection;
        }

        public final void setDeletionProtection(Boolean deletionProtection) {
            this.deletionProtection = deletionProtection;
        }

        @Override
        @Transient
        public final Builder deletionProtection(Boolean deletionProtection) {
            this.deletionProtection = deletionProtection;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ModifyDbClusterRequest build() {
            return new ModifyDbClusterRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
