/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CopySnapshotRequest extends Ec2Request implements
        ToCopyableBuilder<CopySnapshotRequest.Builder, CopySnapshotRequest> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(CopySnapshotRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("Description").build()).build();

    private static final SdkField<String> DESTINATION_OUTPOST_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DestinationOutpostArn")
            .getter(getter(CopySnapshotRequest::destinationOutpostArn))
            .setter(setter(Builder::destinationOutpostArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationOutpostArn")
                    .unmarshallLocationName("DestinationOutpostArn").build()).build();

    private static final SdkField<String> DESTINATION_REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DestinationRegion")
            .getter(getter(CopySnapshotRequest::destinationRegion))
            .setter(setter(Builder::destinationRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationRegion")
                    .unmarshallLocationName("destinationRegion").build()).build();

    private static final SdkField<Boolean> ENCRYPTED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Encrypted")
            .getter(getter(CopySnapshotRequest::encrypted))
            .setter(setter(Builder::encrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Encrypted")
                    .unmarshallLocationName("encrypted").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId")
            .getter(getter(CopySnapshotRequest::kmsKeyId))
            .setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId")
                    .unmarshallLocationName("kmsKeyId").build()).build();

    private static final SdkField<String> PRESIGNED_URL_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PresignedUrl")
            .getter(getter(CopySnapshotRequest::presignedUrl))
            .setter(setter(Builder::presignedUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PresignedUrl")
                    .unmarshallLocationName("presignedUrl").build()).build();

    private static final SdkField<String> SOURCE_REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SourceRegion")
            .getter(getter(CopySnapshotRequest::sourceRegion))
            .setter(setter(Builder::sourceRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceRegion")
                    .unmarshallLocationName("SourceRegion").build()).build();

    private static final SdkField<String> SOURCE_SNAPSHOT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SourceSnapshotId")
            .getter(getter(CopySnapshotRequest::sourceSnapshotId))
            .setter(setter(Builder::sourceSnapshotId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceSnapshotId")
                    .unmarshallLocationName("SourceSnapshotId").build()).build();

    private static final SdkField<List<TagSpecification>> TAG_SPECIFICATIONS_FIELD = SdkField
            .<List<TagSpecification>> builder(MarshallingType.LIST)
            .memberName("TagSpecifications")
            .getter(getter(CopySnapshotRequest::tagSpecifications))
            .setter(setter(Builder::tagSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSpecification")
                    .unmarshallLocationName("TagSpecification").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<TagSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Boolean> DRY_RUN_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DryRun")
            .getter(getter(CopySnapshotRequest::dryRun))
            .setter(setter(Builder::dryRun))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DryRun")
                    .unmarshallLocationName("dryRun").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            DESTINATION_OUTPOST_ARN_FIELD, DESTINATION_REGION_FIELD, ENCRYPTED_FIELD, KMS_KEY_ID_FIELD, PRESIGNED_URL_FIELD,
            SOURCE_REGION_FIELD, SOURCE_SNAPSHOT_ID_FIELD, TAG_SPECIFICATIONS_FIELD, DRY_RUN_FIELD));

    private final String description;

    private final String destinationOutpostArn;

    private final String destinationRegion;

    private final Boolean encrypted;

    private final String kmsKeyId;

    private final String presignedUrl;

    private final String sourceRegion;

    private final String sourceSnapshotId;

    private final List<TagSpecification> tagSpecifications;

    private final Boolean dryRun;

    private CopySnapshotRequest(BuilderImpl builder) {
        super(builder);
        this.description = builder.description;
        this.destinationOutpostArn = builder.destinationOutpostArn;
        this.destinationRegion = builder.destinationRegion;
        this.encrypted = builder.encrypted;
        this.kmsKeyId = builder.kmsKeyId;
        this.presignedUrl = builder.presignedUrl;
        this.sourceRegion = builder.sourceRegion;
        this.sourceSnapshotId = builder.sourceSnapshotId;
        this.tagSpecifications = builder.tagSpecifications;
        this.dryRun = builder.dryRun;
    }

    /**
     * <p>
     * A description for the EBS snapshot.
     * </p>
     * 
     * @return A description for the EBS snapshot.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Outpost to which to copy the snapshot. Only specify this parameter when
     * copying a snapshot from an Amazon Web Services Region to an Outpost. The snapshot must be in the Region for the
     * destination Outpost. You cannot copy a snapshot from an Outpost to a Region, from one Outpost to another, or
     * within the same Outpost.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html#copy-snapshots"> Copy snapshots
     * from an Amazon Web Services Region to an Outpost</a> in the <i>Amazon EBS User Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Outpost to which to copy the snapshot. Only specify this parameter
     *         when copying a snapshot from an Amazon Web Services Region to an Outpost. The snapshot must be in the
     *         Region for the destination Outpost. You cannot copy a snapshot from an Outpost to a Region, from one
     *         Outpost to another, or within the same Outpost.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html#copy-snapshots"> Copy
     *         snapshots from an Amazon Web Services Region to an Outpost</a> in the <i>Amazon EBS User Guide</i>.
     */
    public final String destinationOutpostArn() {
        return destinationOutpostArn;
    }

    /**
     * <p>
     * The destination Region to use in the <code>PresignedUrl</code> parameter of a snapshot copy operation. This
     * parameter is only valid for specifying the destination Region in a <code>PresignedUrl</code> parameter, where it
     * is required.
     * </p>
     * <p>
     * The snapshot copy is sent to the regional endpoint that you sent the HTTP request to (for example,
     * <code>ec2.us-east-1.amazonaws.com</code>). With the CLI, this is specified using the <code>--region</code>
     * parameter or the default Region in your Amazon Web Services configuration file.
     * </p>
     * 
     * @return The destination Region to use in the <code>PresignedUrl</code> parameter of a snapshot copy operation.
     *         This parameter is only valid for specifying the destination Region in a <code>PresignedUrl</code>
     *         parameter, where it is required.</p>
     *         <p>
     *         The snapshot copy is sent to the regional endpoint that you sent the HTTP request to (for example,
     *         <code>ec2.us-east-1.amazonaws.com</code>). With the CLI, this is specified using the
     *         <code>--region</code> parameter or the default Region in your Amazon Web Services configuration file.
     */
    public final String destinationRegion() {
        return destinationRegion;
    }

    /**
     * <p>
     * To encrypt a copy of an unencrypted snapshot if encryption by default is not enabled, enable encryption using
     * this parameter. Otherwise, omit this parameter. Encrypted snapshots are encrypted, even if you omit this
     * parameter and encryption by default is not enabled. You cannot set this parameter to false. For more information,
     * see <a href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-encryption.html">Amazon EBS encryption</a> in
     * the <i>Amazon EBS User Guide</i>.
     * </p>
     * 
     * @return To encrypt a copy of an unencrypted snapshot if encryption by default is not enabled, enable encryption
     *         using this parameter. Otherwise, omit this parameter. Encrypted snapshots are encrypted, even if you omit
     *         this parameter and encryption by default is not enabled. You cannot set this parameter to false. For more
     *         information, see <a href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-encryption.html">Amazon
     *         EBS encryption</a> in the <i>Amazon EBS User Guide</i>.
     */
    public final Boolean encrypted() {
        return encrypted;
    }

    /**
     * <p>
     * The identifier of the KMS key to use for Amazon EBS encryption. If this parameter is not specified, your KMS key
     * for Amazon EBS is used. If <code>KmsKeyId</code> is specified, the encrypted state must be <code>true</code>.
     * </p>
     * <p>
     * You can specify the KMS key using any of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID. For example, 1234abcd-12ab-34cd-56ef-1234567890ab.
     * </p>
     * </li>
     * <li>
     * <p>
     * Key alias. For example, alias/ExampleAlias.
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN. For example, arn:aws:kms:us-east-1:012345678910:key/1234abcd-12ab-34cd-56ef-1234567890ab.
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias ARN. For example, arn:aws:kms:us-east-1:012345678910:alias/ExampleAlias.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Amazon Web Services authenticates the KMS key asynchronously. Therefore, if you specify an ID, alias, or ARN that
     * is not valid, the action can appear to complete, but eventually fails.
     * </p>
     * 
     * @return The identifier of the KMS key to use for Amazon EBS encryption. If this parameter is not specified, your
     *         KMS key for Amazon EBS is used. If <code>KmsKeyId</code> is specified, the encrypted state must be
     *         <code>true</code>.</p>
     *         <p>
     *         You can specify the KMS key using any of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Key ID. For example, 1234abcd-12ab-34cd-56ef-1234567890ab.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key alias. For example, alias/ExampleAlias.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key ARN. For example, arn:aws:kms:us-east-1:012345678910:key/1234abcd-12ab-34cd-56ef-1234567890ab.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias ARN. For example, arn:aws:kms:us-east-1:012345678910:alias/ExampleAlias.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Amazon Web Services authenticates the KMS key asynchronously. Therefore, if you specify an ID, alias, or
     *         ARN that is not valid, the action can appear to complete, but eventually fails.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * When you copy an encrypted source snapshot using the Amazon EC2 Query API, you must supply a pre-signed URL. This
     * parameter is optional for unencrypted snapshots. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Query-Requests.html">Query requests</a>.
     * </p>
     * <p>
     * The <code>PresignedUrl</code> should use the snapshot source endpoint, the <code>CopySnapshot</code> action, and
     * include the <code>SourceRegion</code>, <code>SourceSnapshotId</code>, and <code>DestinationRegion</code>
     * parameters. The <code>PresignedUrl</code> must be signed using Amazon Web Services Signature Version 4. Because
     * EBS snapshots are stored in Amazon S3, the signing algorithm for this parameter uses the same logic that is
     * described in <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/sigv4-query-string-auth.html">
     * Authenticating Requests: Using Query Parameters (Amazon Web Services Signature Version 4)</a> in the <i>Amazon S3
     * API Reference</i>. An invalid or improperly signed <code>PresignedUrl</code> will cause the copy operation to
     * fail asynchronously, and the snapshot will move to an <code>error</code> state.
     * </p>
     * 
     * @return When you copy an encrypted source snapshot using the Amazon EC2 Query API, you must supply a pre-signed
     *         URL. This parameter is optional for unencrypted snapshots. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Query-Requests.html">Query requests</a>.</p>
     *         <p>
     *         The <code>PresignedUrl</code> should use the snapshot source endpoint, the <code>CopySnapshot</code>
     *         action, and include the <code>SourceRegion</code>, <code>SourceSnapshotId</code>, and
     *         <code>DestinationRegion</code> parameters. The <code>PresignedUrl</code> must be signed using Amazon Web
     *         Services Signature Version 4. Because EBS snapshots are stored in Amazon S3, the signing algorithm for
     *         this parameter uses the same logic that is described in <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/API/sigv4-query-string-auth.html"> Authenticating
     *         Requests: Using Query Parameters (Amazon Web Services Signature Version 4)</a> in the <i>Amazon S3 API
     *         Reference</i>. An invalid or improperly signed <code>PresignedUrl</code> will cause the copy operation to
     *         fail asynchronously, and the snapshot will move to an <code>error</code> state.
     */
    public final String presignedUrl() {
        return presignedUrl;
    }

    /**
     * <p>
     * The ID of the Region that contains the snapshot to be copied.
     * </p>
     * 
     * @return The ID of the Region that contains the snapshot to be copied.
     */
    public final String sourceRegion() {
        return sourceRegion;
    }

    /**
     * <p>
     * The ID of the EBS snapshot to copy.
     * </p>
     * 
     * @return The ID of the EBS snapshot to copy.
     */
    public final String sourceSnapshotId() {
        return sourceSnapshotId;
    }

    /**
     * For responses, this returns true if the service returned a value for the TagSpecifications property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTagSpecifications() {
        return tagSpecifications != null && !(tagSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to apply to the new snapshot.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTagSpecifications} method.
     * </p>
     * 
     * @return The tags to apply to the new snapshot.
     */
    public final List<TagSpecification> tagSpecifications() {
        return tagSpecifications;
    }

    /**
     * <p>
     * Checks whether you have the required permissions for the action, without actually making the request, and
     * provides an error response. If you have the required permissions, the error response is
     * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     * </p>
     * 
     * @return Checks whether you have the required permissions for the action, without actually making the request, and
     *         provides an error response. If you have the required permissions, the error response is
     *         <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     */
    public final Boolean dryRun() {
        return dryRun;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(destinationOutpostArn());
        hashCode = 31 * hashCode + Objects.hashCode(destinationRegion());
        hashCode = 31 * hashCode + Objects.hashCode(encrypted());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(presignedUrl());
        hashCode = 31 * hashCode + Objects.hashCode(sourceRegion());
        hashCode = 31 * hashCode + Objects.hashCode(sourceSnapshotId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTagSpecifications() ? tagSpecifications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dryRun());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CopySnapshotRequest)) {
            return false;
        }
        CopySnapshotRequest other = (CopySnapshotRequest) obj;
        return Objects.equals(description(), other.description())
                && Objects.equals(destinationOutpostArn(), other.destinationOutpostArn())
                && Objects.equals(destinationRegion(), other.destinationRegion())
                && Objects.equals(encrypted(), other.encrypted()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(presignedUrl(), other.presignedUrl()) && Objects.equals(sourceRegion(), other.sourceRegion())
                && Objects.equals(sourceSnapshotId(), other.sourceSnapshotId())
                && hasTagSpecifications() == other.hasTagSpecifications()
                && Objects.equals(tagSpecifications(), other.tagSpecifications()) && Objects.equals(dryRun(), other.dryRun());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CopySnapshotRequest").add("Description", description())
                .add("DestinationOutpostArn", destinationOutpostArn()).add("DestinationRegion", destinationRegion())
                .add("Encrypted", encrypted()).add("KmsKeyId", kmsKeyId())
                .add("PresignedUrl", presignedUrl() == null ? null : "*** Sensitive Data Redacted ***")
                .add("SourceRegion", sourceRegion()).add("SourceSnapshotId", sourceSnapshotId())
                .add("TagSpecifications", hasTagSpecifications() ? tagSpecifications() : null).add("DryRun", dryRun()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "DestinationOutpostArn":
            return Optional.ofNullable(clazz.cast(destinationOutpostArn()));
        case "DestinationRegion":
            return Optional.ofNullable(clazz.cast(destinationRegion()));
        case "Encrypted":
            return Optional.ofNullable(clazz.cast(encrypted()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "PresignedUrl":
            return Optional.ofNullable(clazz.cast(presignedUrl()));
        case "SourceRegion":
            return Optional.ofNullable(clazz.cast(sourceRegion()));
        case "SourceSnapshotId":
            return Optional.ofNullable(clazz.cast(sourceSnapshotId()));
        case "TagSpecifications":
            return Optional.ofNullable(clazz.cast(tagSpecifications()));
        case "DryRun":
            return Optional.ofNullable(clazz.cast(dryRun()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CopySnapshotRequest, T> g) {
        return obj -> g.apply((CopySnapshotRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, CopySnapshotRequest> {
        /**
         * <p>
         * A description for the EBS snapshot.
         * </p>
         * 
         * @param description
         *        A description for the EBS snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Outpost to which to copy the snapshot. Only specify this parameter when
         * copying a snapshot from an Amazon Web Services Region to an Outpost. The snapshot must be in the Region for
         * the destination Outpost. You cannot copy a snapshot from an Outpost to a Region, from one Outpost to another,
         * or within the same Outpost.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html#copy-snapshots"> Copy
         * snapshots from an Amazon Web Services Region to an Outpost</a> in the <i>Amazon EBS User Guide</i>.
         * </p>
         * 
         * @param destinationOutpostArn
         *        The Amazon Resource Name (ARN) of the Outpost to which to copy the snapshot. Only specify this
         *        parameter when copying a snapshot from an Amazon Web Services Region to an Outpost. The snapshot must
         *        be in the Region for the destination Outpost. You cannot copy a snapshot from an Outpost to a Region,
         *        from one Outpost to another, or within the same Outpost.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html#copy-snapshots"> Copy
         *        snapshots from an Amazon Web Services Region to an Outpost</a> in the <i>Amazon EBS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationOutpostArn(String destinationOutpostArn);

        /**
         * <p>
         * The destination Region to use in the <code>PresignedUrl</code> parameter of a snapshot copy operation. This
         * parameter is only valid for specifying the destination Region in a <code>PresignedUrl</code> parameter, where
         * it is required.
         * </p>
         * <p>
         * The snapshot copy is sent to the regional endpoint that you sent the HTTP request to (for example,
         * <code>ec2.us-east-1.amazonaws.com</code>). With the CLI, this is specified using the <code>--region</code>
         * parameter or the default Region in your Amazon Web Services configuration file.
         * </p>
         * 
         * @param destinationRegion
         *        The destination Region to use in the <code>PresignedUrl</code> parameter of a snapshot copy operation.
         *        This parameter is only valid for specifying the destination Region in a <code>PresignedUrl</code>
         *        parameter, where it is required.</p>
         *        <p>
         *        The snapshot copy is sent to the regional endpoint that you sent the HTTP request to (for example,
         *        <code>ec2.us-east-1.amazonaws.com</code>). With the CLI, this is specified using the
         *        <code>--region</code> parameter or the default Region in your Amazon Web Services configuration file.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationRegion(String destinationRegion);

        /**
         * <p>
         * To encrypt a copy of an unencrypted snapshot if encryption by default is not enabled, enable encryption using
         * this parameter. Otherwise, omit this parameter. Encrypted snapshots are encrypted, even if you omit this
         * parameter and encryption by default is not enabled. You cannot set this parameter to false. For more
         * information, see <a href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-encryption.html">Amazon EBS
         * encryption</a> in the <i>Amazon EBS User Guide</i>.
         * </p>
         * 
         * @param encrypted
         *        To encrypt a copy of an unencrypted snapshot if encryption by default is not enabled, enable
         *        encryption using this parameter. Otherwise, omit this parameter. Encrypted snapshots are encrypted,
         *        even if you omit this parameter and encryption by default is not enabled. You cannot set this
         *        parameter to false. For more information, see <a
         *        href="https://docs.aws.amazon.com/ebs/latest/userguide/ebs-encryption.html">Amazon EBS encryption</a>
         *        in the <i>Amazon EBS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encrypted(Boolean encrypted);

        /**
         * <p>
         * The identifier of the KMS key to use for Amazon EBS encryption. If this parameter is not specified, your KMS
         * key for Amazon EBS is used. If <code>KmsKeyId</code> is specified, the encrypted state must be
         * <code>true</code>.
         * </p>
         * <p>
         * You can specify the KMS key using any of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Key ID. For example, 1234abcd-12ab-34cd-56ef-1234567890ab.
         * </p>
         * </li>
         * <li>
         * <p>
         * Key alias. For example, alias/ExampleAlias.
         * </p>
         * </li>
         * <li>
         * <p>
         * Key ARN. For example, arn:aws:kms:us-east-1:012345678910:key/1234abcd-12ab-34cd-56ef-1234567890ab.
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias ARN. For example, arn:aws:kms:us-east-1:012345678910:alias/ExampleAlias.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Amazon Web Services authenticates the KMS key asynchronously. Therefore, if you specify an ID, alias, or ARN
         * that is not valid, the action can appear to complete, but eventually fails.
         * </p>
         * 
         * @param kmsKeyId
         *        The identifier of the KMS key to use for Amazon EBS encryption. If this parameter is not specified,
         *        your KMS key for Amazon EBS is used. If <code>KmsKeyId</code> is specified, the encrypted state must
         *        be <code>true</code>.</p>
         *        <p>
         *        You can specify the KMS key using any of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Key ID. For example, 1234abcd-12ab-34cd-56ef-1234567890ab.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Key alias. For example, alias/ExampleAlias.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Key ARN. For example, arn:aws:kms:us-east-1:012345678910:key/1234abcd-12ab-34cd-56ef-1234567890ab.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias ARN. For example, arn:aws:kms:us-east-1:012345678910:alias/ExampleAlias.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Amazon Web Services authenticates the KMS key asynchronously. Therefore, if you specify an ID, alias,
         *        or ARN that is not valid, the action can appear to complete, but eventually fails.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * When you copy an encrypted source snapshot using the Amazon EC2 Query API, you must supply a pre-signed URL.
         * This parameter is optional for unencrypted snapshots. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Query-Requests.html">Query requests</a>.
         * </p>
         * <p>
         * The <code>PresignedUrl</code> should use the snapshot source endpoint, the <code>CopySnapshot</code> action,
         * and include the <code>SourceRegion</code>, <code>SourceSnapshotId</code>, and <code>DestinationRegion</code>
         * parameters. The <code>PresignedUrl</code> must be signed using Amazon Web Services Signature Version 4.
         * Because EBS snapshots are stored in Amazon S3, the signing algorithm for this parameter uses the same logic
         * that is described in <a href="https://docs.aws.amazon.com/AmazonS3/latest/API/sigv4-query-string-auth.html">
         * Authenticating Requests: Using Query Parameters (Amazon Web Services Signature Version 4)</a> in the
         * <i>Amazon S3 API Reference</i>. An invalid or improperly signed <code>PresignedUrl</code> will cause the copy
         * operation to fail asynchronously, and the snapshot will move to an <code>error</code> state.
         * </p>
         * 
         * @param presignedUrl
         *        When you copy an encrypted source snapshot using the Amazon EC2 Query API, you must supply a
         *        pre-signed URL. This parameter is optional for unencrypted snapshots. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Query-Requests.html">Query
         *        requests</a>.</p>
         *        <p>
         *        The <code>PresignedUrl</code> should use the snapshot source endpoint, the <code>CopySnapshot</code>
         *        action, and include the <code>SourceRegion</code>, <code>SourceSnapshotId</code>, and
         *        <code>DestinationRegion</code> parameters. The <code>PresignedUrl</code> must be signed using Amazon
         *        Web Services Signature Version 4. Because EBS snapshots are stored in Amazon S3, the signing algorithm
         *        for this parameter uses the same logic that is described in <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/API/sigv4-query-string-auth.html"> Authenticating
         *        Requests: Using Query Parameters (Amazon Web Services Signature Version 4)</a> in the <i>Amazon S3 API
         *        Reference</i>. An invalid or improperly signed <code>PresignedUrl</code> will cause the copy operation
         *        to fail asynchronously, and the snapshot will move to an <code>error</code> state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder presignedUrl(String presignedUrl);

        /**
         * <p>
         * The ID of the Region that contains the snapshot to be copied.
         * </p>
         * 
         * @param sourceRegion
         *        The ID of the Region that contains the snapshot to be copied.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceRegion(String sourceRegion);

        /**
         * <p>
         * The ID of the EBS snapshot to copy.
         * </p>
         * 
         * @param sourceSnapshotId
         *        The ID of the EBS snapshot to copy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceSnapshotId(String sourceSnapshotId);

        /**
         * <p>
         * The tags to apply to the new snapshot.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to apply to the new snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(Collection<TagSpecification> tagSpecifications);

        /**
         * <p>
         * The tags to apply to the new snapshot.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to apply to the new snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(TagSpecification... tagSpecifications);

        /**
         * <p>
         * The tags to apply to the new snapshot.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.TagSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder#build()} is called immediately and
         * its result is passed to {@link #tagSpecifications(List<TagSpecification>)}.
         * 
         * @param tagSpecifications
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagSpecifications(java.util.Collection<TagSpecification>)
         */
        Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications);

        /**
         * <p>
         * Checks whether you have the required permissions for the action, without actually making the request, and
         * provides an error response. If you have the required permissions, the error response is
         * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * </p>
         * 
         * @param dryRun
         *        Checks whether you have the required permissions for the action, without actually making the request,
         *        and provides an error response. If you have the required permissions, the error response is
         *        <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dryRun(Boolean dryRun);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private String description;

        private String destinationOutpostArn;

        private String destinationRegion;

        private Boolean encrypted;

        private String kmsKeyId;

        private String presignedUrl;

        private String sourceRegion;

        private String sourceSnapshotId;

        private List<TagSpecification> tagSpecifications = DefaultSdkAutoConstructList.getInstance();

        private Boolean dryRun;

        private BuilderImpl() {
        }

        private BuilderImpl(CopySnapshotRequest model) {
            super(model);
            description(model.description);
            destinationOutpostArn(model.destinationOutpostArn);
            destinationRegion(model.destinationRegion);
            encrypted(model.encrypted);
            kmsKeyId(model.kmsKeyId);
            presignedUrl(model.presignedUrl);
            sourceRegion(model.sourceRegion);
            sourceSnapshotId(model.sourceSnapshotId);
            tagSpecifications(model.tagSpecifications);
            dryRun(model.dryRun);
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getDestinationOutpostArn() {
            return destinationOutpostArn;
        }

        public final void setDestinationOutpostArn(String destinationOutpostArn) {
            this.destinationOutpostArn = destinationOutpostArn;
        }

        @Override
        public final Builder destinationOutpostArn(String destinationOutpostArn) {
            this.destinationOutpostArn = destinationOutpostArn;
            return this;
        }

        public final String getDestinationRegion() {
            return destinationRegion;
        }

        public final void setDestinationRegion(String destinationRegion) {
            this.destinationRegion = destinationRegion;
        }

        @Override
        public final Builder destinationRegion(String destinationRegion) {
            this.destinationRegion = destinationRegion;
            return this;
        }

        public final Boolean getEncrypted() {
            return encrypted;
        }

        public final void setEncrypted(Boolean encrypted) {
            this.encrypted = encrypted;
        }

        @Override
        public final Builder encrypted(Boolean encrypted) {
            this.encrypted = encrypted;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final String getPresignedUrl() {
            return presignedUrl;
        }

        public final void setPresignedUrl(String presignedUrl) {
            this.presignedUrl = presignedUrl;
        }

        @Override
        public final Builder presignedUrl(String presignedUrl) {
            this.presignedUrl = presignedUrl;
            return this;
        }

        public final String getSourceRegion() {
            return sourceRegion;
        }

        public final void setSourceRegion(String sourceRegion) {
            this.sourceRegion = sourceRegion;
        }

        @Override
        public final Builder sourceRegion(String sourceRegion) {
            this.sourceRegion = sourceRegion;
            return this;
        }

        public final String getSourceSnapshotId() {
            return sourceSnapshotId;
        }

        public final void setSourceSnapshotId(String sourceSnapshotId) {
            this.sourceSnapshotId = sourceSnapshotId;
        }

        @Override
        public final Builder sourceSnapshotId(String sourceSnapshotId) {
            this.sourceSnapshotId = sourceSnapshotId;
            return this;
        }

        public final List<TagSpecification.Builder> getTagSpecifications() {
            List<TagSpecification.Builder> result = TagSpecificationListCopier.copyToBuilder(this.tagSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTagSpecifications(Collection<TagSpecification.BuilderImpl> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copyFromBuilder(tagSpecifications);
        }

        @Override
        public final Builder tagSpecifications(Collection<TagSpecification> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copy(tagSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(TagSpecification... tagSpecifications) {
            tagSpecifications(Arrays.asList(tagSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications) {
            tagSpecifications(Stream.of(tagSpecifications).map(c -> TagSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getDryRun() {
            return dryRun;
        }

        public final void setDryRun(Boolean dryRun) {
            this.dryRun = dryRun;
        }

        @Override
        public final Builder dryRun(Boolean dryRun) {
            this.dryRun = dryRun;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CopySnapshotRequest build() {
            return new CopySnapshotRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
