/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details on the Elastic IP address transfer. For more information, see <a
 * href="https://docs.aws.amazon.com/vpc/latest/userguide/vpc-eips.html#transfer-EIPs-intro">Transfer Elastic IP
 * addresses</a> in the <i>Amazon VPC User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AddressTransfer implements SdkPojo, Serializable, ToCopyableBuilder<AddressTransfer.Builder, AddressTransfer> {
    private static final SdkField<String> PUBLIC_IP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PublicIp")
            .getter(getter(AddressTransfer::publicIp))
            .setter(setter(Builder::publicIp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublicIp")
                    .unmarshallLocationName("publicIp").build()).build();

    private static final SdkField<String> ALLOCATION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AllocationId")
            .getter(getter(AddressTransfer::allocationId))
            .setter(setter(Builder::allocationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationId")
                    .unmarshallLocationName("allocationId").build()).build();

    private static final SdkField<String> TRANSFER_ACCOUNT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TransferAccountId")
            .getter(getter(AddressTransfer::transferAccountId))
            .setter(setter(Builder::transferAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransferAccountId")
                    .unmarshallLocationName("transferAccountId").build()).build();

    private static final SdkField<Instant> TRANSFER_OFFER_EXPIRATION_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("TransferOfferExpirationTimestamp")
            .getter(getter(AddressTransfer::transferOfferExpirationTimestamp))
            .setter(setter(Builder::transferOfferExpirationTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransferOfferExpirationTimestamp")
                    .unmarshallLocationName("transferOfferExpirationTimestamp").build()).build();

    private static final SdkField<Instant> TRANSFER_OFFER_ACCEPTED_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("TransferOfferAcceptedTimestamp")
            .getter(getter(AddressTransfer::transferOfferAcceptedTimestamp))
            .setter(setter(Builder::transferOfferAcceptedTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransferOfferAcceptedTimestamp")
                    .unmarshallLocationName("transferOfferAcceptedTimestamp").build()).build();

    private static final SdkField<String> ADDRESS_TRANSFER_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AddressTransferStatus")
            .getter(getter(AddressTransfer::addressTransferStatusAsString))
            .setter(setter(Builder::addressTransferStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AddressTransferStatus")
                    .unmarshallLocationName("addressTransferStatus").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PUBLIC_IP_FIELD,
            ALLOCATION_ID_FIELD, TRANSFER_ACCOUNT_ID_FIELD, TRANSFER_OFFER_EXPIRATION_TIMESTAMP_FIELD,
            TRANSFER_OFFER_ACCEPTED_TIMESTAMP_FIELD, ADDRESS_TRANSFER_STATUS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("PublicIp", PUBLIC_IP_FIELD);
                    put("AllocationId", ALLOCATION_ID_FIELD);
                    put("TransferAccountId", TRANSFER_ACCOUNT_ID_FIELD);
                    put("TransferOfferExpirationTimestamp", TRANSFER_OFFER_EXPIRATION_TIMESTAMP_FIELD);
                    put("TransferOfferAcceptedTimestamp", TRANSFER_OFFER_ACCEPTED_TIMESTAMP_FIELD);
                    put("AddressTransferStatus", ADDRESS_TRANSFER_STATUS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String publicIp;

    private final String allocationId;

    private final String transferAccountId;

    private final Instant transferOfferExpirationTimestamp;

    private final Instant transferOfferAcceptedTimestamp;

    private final String addressTransferStatus;

    private AddressTransfer(BuilderImpl builder) {
        this.publicIp = builder.publicIp;
        this.allocationId = builder.allocationId;
        this.transferAccountId = builder.transferAccountId;
        this.transferOfferExpirationTimestamp = builder.transferOfferExpirationTimestamp;
        this.transferOfferAcceptedTimestamp = builder.transferOfferAcceptedTimestamp;
        this.addressTransferStatus = builder.addressTransferStatus;
    }

    /**
     * <p>
     * The Elastic IP address being transferred.
     * </p>
     * 
     * @return The Elastic IP address being transferred.
     */
    public final String publicIp() {
        return publicIp;
    }

    /**
     * <p>
     * The allocation ID of an Elastic IP address.
     * </p>
     * 
     * @return The allocation ID of an Elastic IP address.
     */
    public final String allocationId() {
        return allocationId;
    }

    /**
     * <p>
     * The ID of the account that you want to transfer the Elastic IP address to.
     * </p>
     * 
     * @return The ID of the account that you want to transfer the Elastic IP address to.
     */
    public final String transferAccountId() {
        return transferAccountId;
    }

    /**
     * <p>
     * The timestamp when the Elastic IP address transfer expired. When the source account starts the transfer, the
     * transfer account has seven hours to allocate the Elastic IP address to complete the transfer, or the Elastic IP
     * address will return to its original owner.
     * </p>
     * 
     * @return The timestamp when the Elastic IP address transfer expired. When the source account starts the transfer,
     *         the transfer account has seven hours to allocate the Elastic IP address to complete the transfer, or the
     *         Elastic IP address will return to its original owner.
     */
    public final Instant transferOfferExpirationTimestamp() {
        return transferOfferExpirationTimestamp;
    }

    /**
     * <p>
     * The timestamp when the Elastic IP address transfer was accepted.
     * </p>
     * 
     * @return The timestamp when the Elastic IP address transfer was accepted.
     */
    public final Instant transferOfferAcceptedTimestamp() {
        return transferOfferAcceptedTimestamp;
    }

    /**
     * <p>
     * The Elastic IP address transfer status.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #addressTransferStatus} will return {@link AddressTransferStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #addressTransferStatusAsString}.
     * </p>
     * 
     * @return The Elastic IP address transfer status.
     * @see AddressTransferStatus
     */
    public final AddressTransferStatus addressTransferStatus() {
        return AddressTransferStatus.fromValue(addressTransferStatus);
    }

    /**
     * <p>
     * The Elastic IP address transfer status.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #addressTransferStatus} will return {@link AddressTransferStatus#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #addressTransferStatusAsString}.
     * </p>
     * 
     * @return The Elastic IP address transfer status.
     * @see AddressTransferStatus
     */
    public final String addressTransferStatusAsString() {
        return addressTransferStatus;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(publicIp());
        hashCode = 31 * hashCode + Objects.hashCode(allocationId());
        hashCode = 31 * hashCode + Objects.hashCode(transferAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(transferOfferExpirationTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(transferOfferAcceptedTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(addressTransferStatusAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AddressTransfer)) {
            return false;
        }
        AddressTransfer other = (AddressTransfer) obj;
        return Objects.equals(publicIp(), other.publicIp()) && Objects.equals(allocationId(), other.allocationId())
                && Objects.equals(transferAccountId(), other.transferAccountId())
                && Objects.equals(transferOfferExpirationTimestamp(), other.transferOfferExpirationTimestamp())
                && Objects.equals(transferOfferAcceptedTimestamp(), other.transferOfferAcceptedTimestamp())
                && Objects.equals(addressTransferStatusAsString(), other.addressTransferStatusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AddressTransfer").add("PublicIp", publicIp()).add("AllocationId", allocationId())
                .add("TransferAccountId", transferAccountId())
                .add("TransferOfferExpirationTimestamp", transferOfferExpirationTimestamp())
                .add("TransferOfferAcceptedTimestamp", transferOfferAcceptedTimestamp())
                .add("AddressTransferStatus", addressTransferStatusAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PublicIp":
            return Optional.ofNullable(clazz.cast(publicIp()));
        case "AllocationId":
            return Optional.ofNullable(clazz.cast(allocationId()));
        case "TransferAccountId":
            return Optional.ofNullable(clazz.cast(transferAccountId()));
        case "TransferOfferExpirationTimestamp":
            return Optional.ofNullable(clazz.cast(transferOfferExpirationTimestamp()));
        case "TransferOfferAcceptedTimestamp":
            return Optional.ofNullable(clazz.cast(transferOfferAcceptedTimestamp()));
        case "AddressTransferStatus":
            return Optional.ofNullable(clazz.cast(addressTransferStatusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<AddressTransfer, T> g) {
        return obj -> g.apply((AddressTransfer) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AddressTransfer> {
        /**
         * <p>
         * The Elastic IP address being transferred.
         * </p>
         * 
         * @param publicIp
         *        The Elastic IP address being transferred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicIp(String publicIp);

        /**
         * <p>
         * The allocation ID of an Elastic IP address.
         * </p>
         * 
         * @param allocationId
         *        The allocation ID of an Elastic IP address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationId(String allocationId);

        /**
         * <p>
         * The ID of the account that you want to transfer the Elastic IP address to.
         * </p>
         * 
         * @param transferAccountId
         *        The ID of the account that you want to transfer the Elastic IP address to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transferAccountId(String transferAccountId);

        /**
         * <p>
         * The timestamp when the Elastic IP address transfer expired. When the source account starts the transfer, the
         * transfer account has seven hours to allocate the Elastic IP address to complete the transfer, or the Elastic
         * IP address will return to its original owner.
         * </p>
         * 
         * @param transferOfferExpirationTimestamp
         *        The timestamp when the Elastic IP address transfer expired. When the source account starts the
         *        transfer, the transfer account has seven hours to allocate the Elastic IP address to complete the
         *        transfer, or the Elastic IP address will return to its original owner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transferOfferExpirationTimestamp(Instant transferOfferExpirationTimestamp);

        /**
         * <p>
         * The timestamp when the Elastic IP address transfer was accepted.
         * </p>
         * 
         * @param transferOfferAcceptedTimestamp
         *        The timestamp when the Elastic IP address transfer was accepted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transferOfferAcceptedTimestamp(Instant transferOfferAcceptedTimestamp);

        /**
         * <p>
         * The Elastic IP address transfer status.
         * </p>
         * 
         * @param addressTransferStatus
         *        The Elastic IP address transfer status.
         * @see AddressTransferStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AddressTransferStatus
         */
        Builder addressTransferStatus(String addressTransferStatus);

        /**
         * <p>
         * The Elastic IP address transfer status.
         * </p>
         * 
         * @param addressTransferStatus
         *        The Elastic IP address transfer status.
         * @see AddressTransferStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AddressTransferStatus
         */
        Builder addressTransferStatus(AddressTransferStatus addressTransferStatus);
    }

    static final class BuilderImpl implements Builder {
        private String publicIp;

        private String allocationId;

        private String transferAccountId;

        private Instant transferOfferExpirationTimestamp;

        private Instant transferOfferAcceptedTimestamp;

        private String addressTransferStatus;

        private BuilderImpl() {
        }

        private BuilderImpl(AddressTransfer model) {
            publicIp(model.publicIp);
            allocationId(model.allocationId);
            transferAccountId(model.transferAccountId);
            transferOfferExpirationTimestamp(model.transferOfferExpirationTimestamp);
            transferOfferAcceptedTimestamp(model.transferOfferAcceptedTimestamp);
            addressTransferStatus(model.addressTransferStatus);
        }

        public final String getPublicIp() {
            return publicIp;
        }

        public final void setPublicIp(String publicIp) {
            this.publicIp = publicIp;
        }

        @Override
        public final Builder publicIp(String publicIp) {
            this.publicIp = publicIp;
            return this;
        }

        public final String getAllocationId() {
            return allocationId;
        }

        public final void setAllocationId(String allocationId) {
            this.allocationId = allocationId;
        }

        @Override
        public final Builder allocationId(String allocationId) {
            this.allocationId = allocationId;
            return this;
        }

        public final String getTransferAccountId() {
            return transferAccountId;
        }

        public final void setTransferAccountId(String transferAccountId) {
            this.transferAccountId = transferAccountId;
        }

        @Override
        public final Builder transferAccountId(String transferAccountId) {
            this.transferAccountId = transferAccountId;
            return this;
        }

        public final Instant getTransferOfferExpirationTimestamp() {
            return transferOfferExpirationTimestamp;
        }

        public final void setTransferOfferExpirationTimestamp(Instant transferOfferExpirationTimestamp) {
            this.transferOfferExpirationTimestamp = transferOfferExpirationTimestamp;
        }

        @Override
        public final Builder transferOfferExpirationTimestamp(Instant transferOfferExpirationTimestamp) {
            this.transferOfferExpirationTimestamp = transferOfferExpirationTimestamp;
            return this;
        }

        public final Instant getTransferOfferAcceptedTimestamp() {
            return transferOfferAcceptedTimestamp;
        }

        public final void setTransferOfferAcceptedTimestamp(Instant transferOfferAcceptedTimestamp) {
            this.transferOfferAcceptedTimestamp = transferOfferAcceptedTimestamp;
        }

        @Override
        public final Builder transferOfferAcceptedTimestamp(Instant transferOfferAcceptedTimestamp) {
            this.transferOfferAcceptedTimestamp = transferOfferAcceptedTimestamp;
            return this;
        }

        public final String getAddressTransferStatus() {
            return addressTransferStatus;
        }

        public final void setAddressTransferStatus(String addressTransferStatus) {
            this.addressTransferStatus = addressTransferStatus;
        }

        @Override
        public final Builder addressTransferStatus(String addressTransferStatus) {
            this.addressTransferStatus = addressTransferStatus;
            return this;
        }

        @Override
        public final Builder addressTransferStatus(AddressTransferStatus addressTransferStatus) {
            this.addressTransferStatus(addressTransferStatus == null ? null : addressTransferStatus.toString());
            return this;
        }

        @Override
        public AddressTransfer build() {
            return new AddressTransfer(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
