/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A resource discovery is an IPAM component that enables IPAM to manage and monitor resources that belong to the owning
 * account.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IpamResourceDiscovery implements SdkPojo, Serializable,
        ToCopyableBuilder<IpamResourceDiscovery.Builder, IpamResourceDiscovery> {
    private static final SdkField<String> OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OwnerId")
            .getter(getter(IpamResourceDiscovery::ownerId))
            .setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId")
                    .unmarshallLocationName("ownerId").build()).build();

    private static final SdkField<String> IPAM_RESOURCE_DISCOVERY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamResourceDiscoveryId")
            .getter(getter(IpamResourceDiscovery::ipamResourceDiscoveryId))
            .setter(setter(Builder::ipamResourceDiscoveryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamResourceDiscoveryId")
                    .unmarshallLocationName("ipamResourceDiscoveryId").build()).build();

    private static final SdkField<String> IPAM_RESOURCE_DISCOVERY_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamResourceDiscoveryArn")
            .getter(getter(IpamResourceDiscovery::ipamResourceDiscoveryArn))
            .setter(setter(Builder::ipamResourceDiscoveryArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamResourceDiscoveryArn")
                    .unmarshallLocationName("ipamResourceDiscoveryArn").build()).build();

    private static final SdkField<String> IPAM_RESOURCE_DISCOVERY_REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamResourceDiscoveryRegion")
            .getter(getter(IpamResourceDiscovery::ipamResourceDiscoveryRegion))
            .setter(setter(Builder::ipamResourceDiscoveryRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamResourceDiscoveryRegion")
                    .unmarshallLocationName("ipamResourceDiscoveryRegion").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(IpamResourceDiscovery::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("description").build()).build();

    private static final SdkField<List<IpamOperatingRegion>> OPERATING_REGIONS_FIELD = SdkField
            .<List<IpamOperatingRegion>> builder(MarshallingType.LIST)
            .memberName("OperatingRegions")
            .getter(getter(IpamResourceDiscovery::operatingRegions))
            .setter(setter(Builder::operatingRegions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OperatingRegionSet")
                    .unmarshallLocationName("operatingRegionSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<IpamOperatingRegion> builder(MarshallingType.SDK_POJO)
                                            .constructor(IpamOperatingRegion::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Boolean> IS_DEFAULT_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsDefault")
            .getter(getter(IpamResourceDiscovery::isDefault))
            .setter(setter(Builder::isDefault))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsDefault")
                    .unmarshallLocationName("isDefault").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(IpamResourceDiscovery::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(IpamResourceDiscovery::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<IpamOrganizationalUnitExclusion>> ORGANIZATIONAL_UNIT_EXCLUSIONS_FIELD = SdkField
            .<List<IpamOrganizationalUnitExclusion>> builder(MarshallingType.LIST)
            .memberName("OrganizationalUnitExclusions")
            .getter(getter(IpamResourceDiscovery::organizationalUnitExclusions))
            .setter(setter(Builder::organizationalUnitExclusions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OrganizationalUnitExclusionSet")
                    .unmarshallLocationName("organizationalUnitExclusionSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<IpamOrganizationalUnitExclusion> builder(MarshallingType.SDK_POJO)
                                            .constructor(IpamOrganizationalUnitExclusion::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OWNER_ID_FIELD,
            IPAM_RESOURCE_DISCOVERY_ID_FIELD, IPAM_RESOURCE_DISCOVERY_ARN_FIELD, IPAM_RESOURCE_DISCOVERY_REGION_FIELD,
            DESCRIPTION_FIELD, OPERATING_REGIONS_FIELD, IS_DEFAULT_FIELD, STATE_FIELD, TAGS_FIELD,
            ORGANIZATIONAL_UNIT_EXCLUSIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("OwnerId", OWNER_ID_FIELD);
                    put("IpamResourceDiscoveryId", IPAM_RESOURCE_DISCOVERY_ID_FIELD);
                    put("IpamResourceDiscoveryArn", IPAM_RESOURCE_DISCOVERY_ARN_FIELD);
                    put("IpamResourceDiscoveryRegion", IPAM_RESOURCE_DISCOVERY_REGION_FIELD);
                    put("Description", DESCRIPTION_FIELD);
                    put("OperatingRegionSet", OPERATING_REGIONS_FIELD);
                    put("IsDefault", IS_DEFAULT_FIELD);
                    put("State", STATE_FIELD);
                    put("TagSet", TAGS_FIELD);
                    put("OrganizationalUnitExclusionSet", ORGANIZATIONAL_UNIT_EXCLUSIONS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String ownerId;

    private final String ipamResourceDiscoveryId;

    private final String ipamResourceDiscoveryArn;

    private final String ipamResourceDiscoveryRegion;

    private final String description;

    private final List<IpamOperatingRegion> operatingRegions;

    private final Boolean isDefault;

    private final String state;

    private final List<Tag> tags;

    private final List<IpamOrganizationalUnitExclusion> organizationalUnitExclusions;

    private IpamResourceDiscovery(BuilderImpl builder) {
        this.ownerId = builder.ownerId;
        this.ipamResourceDiscoveryId = builder.ipamResourceDiscoveryId;
        this.ipamResourceDiscoveryArn = builder.ipamResourceDiscoveryArn;
        this.ipamResourceDiscoveryRegion = builder.ipamResourceDiscoveryRegion;
        this.description = builder.description;
        this.operatingRegions = builder.operatingRegions;
        this.isDefault = builder.isDefault;
        this.state = builder.state;
        this.tags = builder.tags;
        this.organizationalUnitExclusions = builder.organizationalUnitExclusions;
    }

    /**
     * <p>
     * The ID of the owner.
     * </p>
     * 
     * @return The ID of the owner.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * The resource discovery ID.
     * </p>
     * 
     * @return The resource discovery ID.
     */
    public final String ipamResourceDiscoveryId() {
        return ipamResourceDiscoveryId;
    }

    /**
     * <p>
     * The resource discovery Amazon Resource Name (ARN).
     * </p>
     * 
     * @return The resource discovery Amazon Resource Name (ARN).
     */
    public final String ipamResourceDiscoveryArn() {
        return ipamResourceDiscoveryArn;
    }

    /**
     * <p>
     * The resource discovery Region.
     * </p>
     * 
     * @return The resource discovery Region.
     */
    public final String ipamResourceDiscoveryRegion() {
        return ipamResourceDiscoveryRegion;
    }

    /**
     * <p>
     * The resource discovery description.
     * </p>
     * 
     * @return The resource discovery description.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the OperatingRegions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasOperatingRegions() {
        return operatingRegions != null && !(operatingRegions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The operating Regions for the resource discovery. Operating Regions are Amazon Web Services Regions where the
     * IPAM is allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the Amazon Web Services
     * Regions you select as operating Regions.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOperatingRegions} method.
     * </p>
     * 
     * @return The operating Regions for the resource discovery. Operating Regions are Amazon Web Services Regions where
     *         the IPAM is allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the Amazon
     *         Web Services Regions you select as operating Regions.
     */
    public final List<IpamOperatingRegion> operatingRegions() {
        return operatingRegions;
    }

    /**
     * <p>
     * Defines if the resource discovery is the default. The default resource discovery is the resource discovery
     * automatically created when you create an IPAM.
     * </p>
     * 
     * @return Defines if the resource discovery is the default. The default resource discovery is the resource
     *         discovery automatically created when you create an IPAM.
     */
    public final Boolean isDefault() {
        return isDefault;
    }

    /**
     * <p>
     * The lifecycle state of the resource discovery.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>create-in-progress</code> - Resource discovery is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>create-complete</code> - Resource discovery creation is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>create-failed</code> - Resource discovery creation has failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>modify-in-progress</code> - Resource discovery is being modified.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>modify-complete</code> - Resource discovery modification is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>modify-failed</code> - Resource discovery modification has failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>delete-in-progress</code> - Resource discovery is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>delete-complete</code> - Resource discovery deletion is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>delete-failed</code> - Resource discovery deletion has failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery has been
     * removed and the resource discovery is being isolated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>isolate-complete</code> - Resource discovery isolation is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>restore-in-progress</code> - Amazon Web Services account that created the resource discovery and was
     * isolated has been restored.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IpamResourceDiscoveryState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stateAsString}.
     * </p>
     * 
     * @return The lifecycle state of the resource discovery.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>create-in-progress</code> - Resource discovery is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>create-complete</code> - Resource discovery creation is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>create-failed</code> - Resource discovery creation has failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>modify-in-progress</code> - Resource discovery is being modified.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>modify-complete</code> - Resource discovery modification is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>modify-failed</code> - Resource discovery modification has failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>delete-in-progress</code> - Resource discovery is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>delete-complete</code> - Resource discovery deletion is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>delete-failed</code> - Resource discovery deletion has failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery has
     *         been removed and the resource discovery is being isolated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>isolate-complete</code> - Resource discovery isolation is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>restore-in-progress</code> - Amazon Web Services account that created the resource discovery and
     *         was isolated has been restored.
     *         </p>
     *         </li>
     * @see IpamResourceDiscoveryState
     */
    public final IpamResourceDiscoveryState state() {
        return IpamResourceDiscoveryState.fromValue(state);
    }

    /**
     * <p>
     * The lifecycle state of the resource discovery.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>create-in-progress</code> - Resource discovery is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>create-complete</code> - Resource discovery creation is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>create-failed</code> - Resource discovery creation has failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>modify-in-progress</code> - Resource discovery is being modified.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>modify-complete</code> - Resource discovery modification is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>modify-failed</code> - Resource discovery modification has failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>delete-in-progress</code> - Resource discovery is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>delete-complete</code> - Resource discovery deletion is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>delete-failed</code> - Resource discovery deletion has failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery has been
     * removed and the resource discovery is being isolated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>isolate-complete</code> - Resource discovery isolation is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>restore-in-progress</code> - Amazon Web Services account that created the resource discovery and was
     * isolated has been restored.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IpamResourceDiscoveryState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stateAsString}.
     * </p>
     * 
     * @return The lifecycle state of the resource discovery.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>create-in-progress</code> - Resource discovery is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>create-complete</code> - Resource discovery creation is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>create-failed</code> - Resource discovery creation has failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>modify-in-progress</code> - Resource discovery is being modified.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>modify-complete</code> - Resource discovery modification is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>modify-failed</code> - Resource discovery modification has failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>delete-in-progress</code> - Resource discovery is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>delete-complete</code> - Resource discovery deletion is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>delete-failed</code> - Resource discovery deletion has failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery has
     *         been removed and the resource discovery is being isolated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>isolate-complete</code> - Resource discovery isolation is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>restore-in-progress</code> - Amazon Web Services account that created the resource discovery and
     *         was isolated has been restored.
     *         </p>
     *         </li>
     * @see IpamResourceDiscoveryState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an optional
     * value. You can use tags to search and filter your resources or track your Amazon Web Services costs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an
     *         optional value. You can use tags to search and filter your resources or track your Amazon Web Services
     *         costs.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * For responses, this returns true if the service returned a value for the OrganizationalUnitExclusions property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasOrganizationalUnitExclusions() {
        return organizationalUnitExclusions != null && !(organizationalUnitExclusions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If your IPAM is integrated with Amazon Web Services Organizations and you add an organizational unit (OU)
     * exclusion, IPAM will not manage the IP addresses in accounts in that OU exclusion.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOrganizationalUnitExclusions} method.
     * </p>
     * 
     * @return If your IPAM is integrated with Amazon Web Services Organizations and you add an organizational unit (OU)
     *         exclusion, IPAM will not manage the IP addresses in accounts in that OU exclusion.
     */
    public final List<IpamOrganizationalUnitExclusion> organizationalUnitExclusions() {
        return organizationalUnitExclusions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamResourceDiscoveryId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamResourceDiscoveryArn());
        hashCode = 31 * hashCode + Objects.hashCode(ipamResourceDiscoveryRegion());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasOperatingRegions() ? operatingRegions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(isDefault());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasOrganizationalUnitExclusions() ? organizationalUnitExclusions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IpamResourceDiscovery)) {
            return false;
        }
        IpamResourceDiscovery other = (IpamResourceDiscovery) obj;
        return Objects.equals(ownerId(), other.ownerId())
                && Objects.equals(ipamResourceDiscoveryId(), other.ipamResourceDiscoveryId())
                && Objects.equals(ipamResourceDiscoveryArn(), other.ipamResourceDiscoveryArn())
                && Objects.equals(ipamResourceDiscoveryRegion(), other.ipamResourceDiscoveryRegion())
                && Objects.equals(description(), other.description()) && hasOperatingRegions() == other.hasOperatingRegions()
                && Objects.equals(operatingRegions(), other.operatingRegions()) && Objects.equals(isDefault(), other.isDefault())
                && Objects.equals(stateAsString(), other.stateAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && hasOrganizationalUnitExclusions() == other.hasOrganizationalUnitExclusions()
                && Objects.equals(organizationalUnitExclusions(), other.organizationalUnitExclusions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IpamResourceDiscovery").add("OwnerId", ownerId())
                .add("IpamResourceDiscoveryId", ipamResourceDiscoveryId())
                .add("IpamResourceDiscoveryArn", ipamResourceDiscoveryArn())
                .add("IpamResourceDiscoveryRegion", ipamResourceDiscoveryRegion()).add("Description", description())
                .add("OperatingRegions", hasOperatingRegions() ? operatingRegions() : null).add("IsDefault", isDefault())
                .add("State", stateAsString()).add("Tags", hasTags() ? tags() : null)
                .add("OrganizationalUnitExclusions", hasOrganizationalUnitExclusions() ? organizationalUnitExclusions() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "IpamResourceDiscoveryId":
            return Optional.ofNullable(clazz.cast(ipamResourceDiscoveryId()));
        case "IpamResourceDiscoveryArn":
            return Optional.ofNullable(clazz.cast(ipamResourceDiscoveryArn()));
        case "IpamResourceDiscoveryRegion":
            return Optional.ofNullable(clazz.cast(ipamResourceDiscoveryRegion()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "OperatingRegions":
            return Optional.ofNullable(clazz.cast(operatingRegions()));
        case "IsDefault":
            return Optional.ofNullable(clazz.cast(isDefault()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "OrganizationalUnitExclusions":
            return Optional.ofNullable(clazz.cast(organizationalUnitExclusions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<IpamResourceDiscovery, T> g) {
        return obj -> g.apply((IpamResourceDiscovery) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IpamResourceDiscovery> {
        /**
         * <p>
         * The ID of the owner.
         * </p>
         * 
         * @param ownerId
         *        The ID of the owner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * The resource discovery ID.
         * </p>
         * 
         * @param ipamResourceDiscoveryId
         *        The resource discovery ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamResourceDiscoveryId(String ipamResourceDiscoveryId);

        /**
         * <p>
         * The resource discovery Amazon Resource Name (ARN).
         * </p>
         * 
         * @param ipamResourceDiscoveryArn
         *        The resource discovery Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamResourceDiscoveryArn(String ipamResourceDiscoveryArn);

        /**
         * <p>
         * The resource discovery Region.
         * </p>
         * 
         * @param ipamResourceDiscoveryRegion
         *        The resource discovery Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamResourceDiscoveryRegion(String ipamResourceDiscoveryRegion);

        /**
         * <p>
         * The resource discovery description.
         * </p>
         * 
         * @param description
         *        The resource discovery description.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The operating Regions for the resource discovery. Operating Regions are Amazon Web Services Regions where the
         * IPAM is allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the Amazon Web
         * Services Regions you select as operating Regions.
         * </p>
         * 
         * @param operatingRegions
         *        The operating Regions for the resource discovery. Operating Regions are Amazon Web Services Regions
         *        where the IPAM is allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in
         *        the Amazon Web Services Regions you select as operating Regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operatingRegions(Collection<IpamOperatingRegion> operatingRegions);

        /**
         * <p>
         * The operating Regions for the resource discovery. Operating Regions are Amazon Web Services Regions where the
         * IPAM is allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the Amazon Web
         * Services Regions you select as operating Regions.
         * </p>
         * 
         * @param operatingRegions
         *        The operating Regions for the resource discovery. Operating Regions are Amazon Web Services Regions
         *        where the IPAM is allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in
         *        the Amazon Web Services Regions you select as operating Regions.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder operatingRegions(IpamOperatingRegion... operatingRegions);

        /**
         * <p>
         * The operating Regions for the resource discovery. Operating Regions are Amazon Web Services Regions where the
         * IPAM is allowed to manage IP address CIDRs. IPAM only discovers and monitors resources in the Amazon Web
         * Services Regions you select as operating Regions.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.IpamOperatingRegion.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.IpamOperatingRegion#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.IpamOperatingRegion.Builder#build()} is called immediately
         * and its result is passed to {@link #operatingRegions(List<IpamOperatingRegion>)}.
         * 
         * @param operatingRegions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.IpamOperatingRegion.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #operatingRegions(java.util.Collection<IpamOperatingRegion>)
         */
        Builder operatingRegions(Consumer<IpamOperatingRegion.Builder>... operatingRegions);

        /**
         * <p>
         * Defines if the resource discovery is the default. The default resource discovery is the resource discovery
         * automatically created when you create an IPAM.
         * </p>
         * 
         * @param isDefault
         *        Defines if the resource discovery is the default. The default resource discovery is the resource
         *        discovery automatically created when you create an IPAM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isDefault(Boolean isDefault);

        /**
         * <p>
         * The lifecycle state of the resource discovery.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>create-in-progress</code> - Resource discovery is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>create-complete</code> - Resource discovery creation is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>create-failed</code> - Resource discovery creation has failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>modify-in-progress</code> - Resource discovery is being modified.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>modify-complete</code> - Resource discovery modification is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>modify-failed</code> - Resource discovery modification has failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>delete-in-progress</code> - Resource discovery is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>delete-complete</code> - Resource discovery deletion is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>delete-failed</code> - Resource discovery deletion has failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery has been
         * removed and the resource discovery is being isolated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>isolate-complete</code> - Resource discovery isolation is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>restore-in-progress</code> - Amazon Web Services account that created the resource discovery and was
         * isolated has been restored.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The lifecycle state of the resource discovery.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>create-in-progress</code> - Resource discovery is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>create-complete</code> - Resource discovery creation is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>create-failed</code> - Resource discovery creation has failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>modify-in-progress</code> - Resource discovery is being modified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>modify-complete</code> - Resource discovery modification is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>modify-failed</code> - Resource discovery modification has failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>delete-in-progress</code> - Resource discovery is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>delete-complete</code> - Resource discovery deletion is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>delete-failed</code> - Resource discovery deletion has failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery has
         *        been removed and the resource discovery is being isolated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>isolate-complete</code> - Resource discovery isolation is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>restore-in-progress</code> - Amazon Web Services account that created the resource discovery and
         *        was isolated has been restored.
         *        </p>
         *        </li>
         * @see IpamResourceDiscoveryState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamResourceDiscoveryState
         */
        Builder state(String state);

        /**
         * <p>
         * The lifecycle state of the resource discovery.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>create-in-progress</code> - Resource discovery is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>create-complete</code> - Resource discovery creation is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>create-failed</code> - Resource discovery creation has failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>modify-in-progress</code> - Resource discovery is being modified.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>modify-complete</code> - Resource discovery modification is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>modify-failed</code> - Resource discovery modification has failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>delete-in-progress</code> - Resource discovery is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>delete-complete</code> - Resource discovery deletion is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>delete-failed</code> - Resource discovery deletion has failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery has been
         * removed and the resource discovery is being isolated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>isolate-complete</code> - Resource discovery isolation is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>restore-in-progress</code> - Amazon Web Services account that created the resource discovery and was
         * isolated has been restored.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The lifecycle state of the resource discovery.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>create-in-progress</code> - Resource discovery is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>create-complete</code> - Resource discovery creation is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>create-failed</code> - Resource discovery creation has failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>modify-in-progress</code> - Resource discovery is being modified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>modify-complete</code> - Resource discovery modification is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>modify-failed</code> - Resource discovery modification has failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>delete-in-progress</code> - Resource discovery is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>delete-complete</code> - Resource discovery deletion is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>delete-failed</code> - Resource discovery deletion has failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery has
         *        been removed and the resource discovery is being isolated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>isolate-complete</code> - Resource discovery isolation is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>restore-in-progress</code> - Amazon Web Services account that created the resource discovery and
         *        was isolated has been restored.
         *        </p>
         *        </li>
         * @see IpamResourceDiscoveryState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamResourceDiscoveryState
         */
        Builder state(IpamResourceDiscoveryState state);

        /**
         * <p>
         * A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an
         * optional value. You can use tags to search and filter your resources or track your Amazon Web Services costs.
         * </p>
         * 
         * @param tags
         *        A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an
         *        optional value. You can use tags to search and filter your resources or track your Amazon Web Services
         *        costs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an
         * optional value. You can use tags to search and filter your resources or track your Amazon Web Services costs.
         * </p>
         * 
         * @param tags
         *        A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an
         *        optional value. You can use tags to search and filter your resources or track your Amazon Web Services
         *        costs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an
         * optional value. You can use tags to search and filter your resources or track your Amazon Web Services costs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * If your IPAM is integrated with Amazon Web Services Organizations and you add an organizational unit (OU)
         * exclusion, IPAM will not manage the IP addresses in accounts in that OU exclusion.
         * </p>
         * 
         * @param organizationalUnitExclusions
         *        If your IPAM is integrated with Amazon Web Services Organizations and you add an organizational unit
         *        (OU) exclusion, IPAM will not manage the IP addresses in accounts in that OU exclusion.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationalUnitExclusions(Collection<IpamOrganizationalUnitExclusion> organizationalUnitExclusions);

        /**
         * <p>
         * If your IPAM is integrated with Amazon Web Services Organizations and you add an organizational unit (OU)
         * exclusion, IPAM will not manage the IP addresses in accounts in that OU exclusion.
         * </p>
         * 
         * @param organizationalUnitExclusions
         *        If your IPAM is integrated with Amazon Web Services Organizations and you add an organizational unit
         *        (OU) exclusion, IPAM will not manage the IP addresses in accounts in that OU exclusion.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder organizationalUnitExclusions(IpamOrganizationalUnitExclusion... organizationalUnitExclusions);

        /**
         * <p>
         * If your IPAM is integrated with Amazon Web Services Organizations and you add an organizational unit (OU)
         * exclusion, IPAM will not manage the IP addresses in accounts in that OU exclusion.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.IpamOrganizationalUnitExclusion.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.IpamOrganizationalUnitExclusion#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.IpamOrganizationalUnitExclusion.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #organizationalUnitExclusions(List<IpamOrganizationalUnitExclusion>)}.
         * 
         * @param organizationalUnitExclusions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.IpamOrganizationalUnitExclusion.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #organizationalUnitExclusions(java.util.Collection<IpamOrganizationalUnitExclusion>)
         */
        Builder organizationalUnitExclusions(Consumer<IpamOrganizationalUnitExclusion.Builder>... organizationalUnitExclusions);
    }

    static final class BuilderImpl implements Builder {
        private String ownerId;

        private String ipamResourceDiscoveryId;

        private String ipamResourceDiscoveryArn;

        private String ipamResourceDiscoveryRegion;

        private String description;

        private List<IpamOperatingRegion> operatingRegions = DefaultSdkAutoConstructList.getInstance();

        private Boolean isDefault;

        private String state;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private List<IpamOrganizationalUnitExclusion> organizationalUnitExclusions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(IpamResourceDiscovery model) {
            ownerId(model.ownerId);
            ipamResourceDiscoveryId(model.ipamResourceDiscoveryId);
            ipamResourceDiscoveryArn(model.ipamResourceDiscoveryArn);
            ipamResourceDiscoveryRegion(model.ipamResourceDiscoveryRegion);
            description(model.description);
            operatingRegions(model.operatingRegions);
            isDefault(model.isDefault);
            state(model.state);
            tags(model.tags);
            organizationalUnitExclusions(model.organizationalUnitExclusions);
        }

        public final String getOwnerId() {
            return ownerId;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final String getIpamResourceDiscoveryId() {
            return ipamResourceDiscoveryId;
        }

        public final void setIpamResourceDiscoveryId(String ipamResourceDiscoveryId) {
            this.ipamResourceDiscoveryId = ipamResourceDiscoveryId;
        }

        @Override
        public final Builder ipamResourceDiscoveryId(String ipamResourceDiscoveryId) {
            this.ipamResourceDiscoveryId = ipamResourceDiscoveryId;
            return this;
        }

        public final String getIpamResourceDiscoveryArn() {
            return ipamResourceDiscoveryArn;
        }

        public final void setIpamResourceDiscoveryArn(String ipamResourceDiscoveryArn) {
            this.ipamResourceDiscoveryArn = ipamResourceDiscoveryArn;
        }

        @Override
        public final Builder ipamResourceDiscoveryArn(String ipamResourceDiscoveryArn) {
            this.ipamResourceDiscoveryArn = ipamResourceDiscoveryArn;
            return this;
        }

        public final String getIpamResourceDiscoveryRegion() {
            return ipamResourceDiscoveryRegion;
        }

        public final void setIpamResourceDiscoveryRegion(String ipamResourceDiscoveryRegion) {
            this.ipamResourceDiscoveryRegion = ipamResourceDiscoveryRegion;
        }

        @Override
        public final Builder ipamResourceDiscoveryRegion(String ipamResourceDiscoveryRegion) {
            this.ipamResourceDiscoveryRegion = ipamResourceDiscoveryRegion;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final List<IpamOperatingRegion.Builder> getOperatingRegions() {
            List<IpamOperatingRegion.Builder> result = IpamOperatingRegionSetCopier.copyToBuilder(this.operatingRegions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setOperatingRegions(Collection<IpamOperatingRegion.BuilderImpl> operatingRegions) {
            this.operatingRegions = IpamOperatingRegionSetCopier.copyFromBuilder(operatingRegions);
        }

        @Override
        public final Builder operatingRegions(Collection<IpamOperatingRegion> operatingRegions) {
            this.operatingRegions = IpamOperatingRegionSetCopier.copy(operatingRegions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder operatingRegions(IpamOperatingRegion... operatingRegions) {
            operatingRegions(Arrays.asList(operatingRegions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder operatingRegions(Consumer<IpamOperatingRegion.Builder>... operatingRegions) {
            operatingRegions(Stream.of(operatingRegions).map(c -> IpamOperatingRegion.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getIsDefault() {
            return isDefault;
        }

        public final void setIsDefault(Boolean isDefault) {
            this.isDefault = isDefault;
        }

        @Override
        public final Builder isDefault(Boolean isDefault) {
            this.isDefault = isDefault;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(IpamResourceDiscoveryState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<IpamOrganizationalUnitExclusion.Builder> getOrganizationalUnitExclusions() {
            List<IpamOrganizationalUnitExclusion.Builder> result = IpamOrganizationalUnitExclusionSetCopier
                    .copyToBuilder(this.organizationalUnitExclusions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setOrganizationalUnitExclusions(
                Collection<IpamOrganizationalUnitExclusion.BuilderImpl> organizationalUnitExclusions) {
            this.organizationalUnitExclusions = IpamOrganizationalUnitExclusionSetCopier
                    .copyFromBuilder(organizationalUnitExclusions);
        }

        @Override
        public final Builder organizationalUnitExclusions(Collection<IpamOrganizationalUnitExclusion> organizationalUnitExclusions) {
            this.organizationalUnitExclusions = IpamOrganizationalUnitExclusionSetCopier.copy(organizationalUnitExclusions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder organizationalUnitExclusions(IpamOrganizationalUnitExclusion... organizationalUnitExclusions) {
            organizationalUnitExclusions(Arrays.asList(organizationalUnitExclusions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder organizationalUnitExclusions(
                Consumer<IpamOrganizationalUnitExclusion.Builder>... organizationalUnitExclusions) {
            organizationalUnitExclusions(Stream.of(organizationalUnitExclusions)
                    .map(c -> IpamOrganizationalUnitExclusion.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public IpamResourceDiscovery build() {
            return new IpamResourceDiscovery(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
