/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The list of criteria that are evaluated to determine whch AMIs are discoverable and usable in the account in the
 * specified Amazon Web Services Region. Currently, the only criteria that can be specified are AMI providers.
 * </p>
 * <p>
 * Up to 10 <code>imageCriteria</code> objects can be specified, and up to a total of 200 values for all
 * <code>imageProviders</code>. For more information, see <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-allowed-amis.html#allowed-amis-json-configuration">JSON
 * configuration for the Allowed AMIs criteria</a> in the <i>Amazon EC2 User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImageCriterion implements SdkPojo, Serializable, ToCopyableBuilder<ImageCriterion.Builder, ImageCriterion> {
    private static final SdkField<List<String>> IMAGE_PROVIDERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ImageProviders")
            .getter(getter(ImageCriterion::imageProviders))
            .setter(setter(Builder::imageProviders))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageProviderSet")
                    .unmarshallLocationName("imageProviderSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_PROVIDERS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<String> imageProviders;

    private ImageCriterion(BuilderImpl builder) {
        this.imageProviders = builder.imageProviders;
    }

    /**
     * For responses, this returns true if the service returned a value for the ImageProviders property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasImageProviders() {
        return imageProviders != null && !(imageProviders instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of AMI providers whose AMIs are discoverable and useable in the account. Up to a total of 200 values can
     * be specified.
     * </p>
     * <p>
     * Possible values:
     * </p>
     * <p>
     * <code>amazon</code>: Allow AMIs created by Amazon Web Services.
     * </p>
     * <p>
     * <code>aws-marketplace</code>: Allow AMIs created by verified providers in the Amazon Web Services Marketplace.
     * </p>
     * <p>
     * <code>aws-backup-vault</code>: Allow AMIs created by Amazon Web Services Backup.
     * </p>
     * <p>
     * 12-digit account ID: Allow AMIs created by this account. One or more account IDs can be specified.
     * </p>
     * <p>
     * <code>none</code>: Allow AMIs created by your own account only.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasImageProviders} method.
     * </p>
     * 
     * @return A list of AMI providers whose AMIs are discoverable and useable in the account. Up to a total of 200
     *         values can be specified.</p>
     *         <p>
     *         Possible values:
     *         </p>
     *         <p>
     *         <code>amazon</code>: Allow AMIs created by Amazon Web Services.
     *         </p>
     *         <p>
     *         <code>aws-marketplace</code>: Allow AMIs created by verified providers in the Amazon Web Services
     *         Marketplace.
     *         </p>
     *         <p>
     *         <code>aws-backup-vault</code>: Allow AMIs created by Amazon Web Services Backup.
     *         </p>
     *         <p>
     *         12-digit account ID: Allow AMIs created by this account. One or more account IDs can be specified.
     *         </p>
     *         <p>
     *         <code>none</code>: Allow AMIs created by your own account only.
     */
    public final List<String> imageProviders() {
        return imageProviders;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasImageProviders() ? imageProviders() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImageCriterion)) {
            return false;
        }
        ImageCriterion other = (ImageCriterion) obj;
        return hasImageProviders() == other.hasImageProviders() && Objects.equals(imageProviders(), other.imageProviders());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ImageCriterion").add("ImageProviders", hasImageProviders() ? imageProviders() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ImageProviders":
            return Optional.ofNullable(clazz.cast(imageProviders()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ImageProviderSet", IMAGE_PROVIDERS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ImageCriterion, T> g) {
        return obj -> g.apply((ImageCriterion) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ImageCriterion> {
        /**
         * <p>
         * A list of AMI providers whose AMIs are discoverable and useable in the account. Up to a total of 200 values
         * can be specified.
         * </p>
         * <p>
         * Possible values:
         * </p>
         * <p>
         * <code>amazon</code>: Allow AMIs created by Amazon Web Services.
         * </p>
         * <p>
         * <code>aws-marketplace</code>: Allow AMIs created by verified providers in the Amazon Web Services
         * Marketplace.
         * </p>
         * <p>
         * <code>aws-backup-vault</code>: Allow AMIs created by Amazon Web Services Backup.
         * </p>
         * <p>
         * 12-digit account ID: Allow AMIs created by this account. One or more account IDs can be specified.
         * </p>
         * <p>
         * <code>none</code>: Allow AMIs created by your own account only.
         * </p>
         * 
         * @param imageProviders
         *        A list of AMI providers whose AMIs are discoverable and useable in the account. Up to a total of 200
         *        values can be specified.</p>
         *        <p>
         *        Possible values:
         *        </p>
         *        <p>
         *        <code>amazon</code>: Allow AMIs created by Amazon Web Services.
         *        </p>
         *        <p>
         *        <code>aws-marketplace</code>: Allow AMIs created by verified providers in the Amazon Web Services
         *        Marketplace.
         *        </p>
         *        <p>
         *        <code>aws-backup-vault</code>: Allow AMIs created by Amazon Web Services Backup.
         *        </p>
         *        <p>
         *        12-digit account ID: Allow AMIs created by this account. One or more account IDs can be specified.
         *        </p>
         *        <p>
         *        <code>none</code>: Allow AMIs created by your own account only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageProviders(Collection<String> imageProviders);

        /**
         * <p>
         * A list of AMI providers whose AMIs are discoverable and useable in the account. Up to a total of 200 values
         * can be specified.
         * </p>
         * <p>
         * Possible values:
         * </p>
         * <p>
         * <code>amazon</code>: Allow AMIs created by Amazon Web Services.
         * </p>
         * <p>
         * <code>aws-marketplace</code>: Allow AMIs created by verified providers in the Amazon Web Services
         * Marketplace.
         * </p>
         * <p>
         * <code>aws-backup-vault</code>: Allow AMIs created by Amazon Web Services Backup.
         * </p>
         * <p>
         * 12-digit account ID: Allow AMIs created by this account. One or more account IDs can be specified.
         * </p>
         * <p>
         * <code>none</code>: Allow AMIs created by your own account only.
         * </p>
         * 
         * @param imageProviders
         *        A list of AMI providers whose AMIs are discoverable and useable in the account. Up to a total of 200
         *        values can be specified.</p>
         *        <p>
         *        Possible values:
         *        </p>
         *        <p>
         *        <code>amazon</code>: Allow AMIs created by Amazon Web Services.
         *        </p>
         *        <p>
         *        <code>aws-marketplace</code>: Allow AMIs created by verified providers in the Amazon Web Services
         *        Marketplace.
         *        </p>
         *        <p>
         *        <code>aws-backup-vault</code>: Allow AMIs created by Amazon Web Services Backup.
         *        </p>
         *        <p>
         *        12-digit account ID: Allow AMIs created by this account. One or more account IDs can be specified.
         *        </p>
         *        <p>
         *        <code>none</code>: Allow AMIs created by your own account only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageProviders(String... imageProviders);
    }

    static final class BuilderImpl implements Builder {
        private List<String> imageProviders = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ImageCriterion model) {
            imageProviders(model.imageProviders);
        }

        public final Collection<String> getImageProviders() {
            if (imageProviders instanceof SdkAutoConstructList) {
                return null;
            }
            return imageProviders;
        }

        public final void setImageProviders(Collection<String> imageProviders) {
            this.imageProviders = ImageProviderListCopier.copy(imageProviders);
        }

        @Override
        public final Builder imageProviders(Collection<String> imageProviders) {
            this.imageProviders = ImageProviderListCopier.copy(imageProviders);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder imageProviders(String... imageProviders) {
            imageProviders(Arrays.asList(imageProviders));
            return this;
        }

        @Override
        public ImageCriterion build() {
            return new ImageCriterion(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
