/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a Client VPN endpoint.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ClientVpnEndpoint implements SdkPojo, Serializable,
        ToCopyableBuilder<ClientVpnEndpoint.Builder, ClientVpnEndpoint> {
    private static final SdkField<String> CLIENT_VPN_ENDPOINT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientVpnEndpointId")
            .getter(getter(ClientVpnEndpoint::clientVpnEndpointId))
            .setter(setter(Builder::clientVpnEndpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientVpnEndpointId")
                    .unmarshallLocationName("clientVpnEndpointId").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(ClientVpnEndpoint::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("description").build()).build();

    private static final SdkField<ClientVpnEndpointStatus> STATUS_FIELD = SdkField
            .<ClientVpnEndpointStatus> builder(MarshallingType.SDK_POJO)
            .memberName("Status")
            .getter(getter(ClientVpnEndpoint::status))
            .setter(setter(Builder::status))
            .constructor(ClientVpnEndpointStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status")
                    .unmarshallLocationName("status").build()).build();

    private static final SdkField<String> CREATION_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CreationTime")
            .getter(getter(ClientVpnEndpoint::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime")
                    .unmarshallLocationName("creationTime").build()).build();

    private static final SdkField<String> DELETION_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DeletionTime")
            .getter(getter(ClientVpnEndpoint::deletionTime))
            .setter(setter(Builder::deletionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeletionTime")
                    .unmarshallLocationName("deletionTime").build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DnsName")
            .getter(getter(ClientVpnEndpoint::dnsName))
            .setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsName")
                    .unmarshallLocationName("dnsName").build()).build();

    private static final SdkField<String> CLIENT_CIDR_BLOCK_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientCidrBlock")
            .getter(getter(ClientVpnEndpoint::clientCidrBlock))
            .setter(setter(Builder::clientCidrBlock))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientCidrBlock")
                    .unmarshallLocationName("clientCidrBlock").build()).build();

    private static final SdkField<List<String>> DNS_SERVERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DnsServers")
            .getter(getter(ClientVpnEndpoint::dnsServers))
            .setter(setter(Builder::dnsServers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsServer")
                    .unmarshallLocationName("dnsServer").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Boolean> SPLIT_TUNNEL_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SplitTunnel")
            .getter(getter(ClientVpnEndpoint::splitTunnel))
            .setter(setter(Builder::splitTunnel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SplitTunnel")
                    .unmarshallLocationName("splitTunnel").build()).build();

    private static final SdkField<String> VPN_PROTOCOL_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpnProtocol")
            .getter(getter(ClientVpnEndpoint::vpnProtocolAsString))
            .setter(setter(Builder::vpnProtocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpnProtocol")
                    .unmarshallLocationName("vpnProtocol").build()).build();

    private static final SdkField<String> TRANSPORT_PROTOCOL_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TransportProtocol")
            .getter(getter(ClientVpnEndpoint::transportProtocolAsString))
            .setter(setter(Builder::transportProtocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransportProtocol")
                    .unmarshallLocationName("transportProtocol").build()).build();

    private static final SdkField<Integer> VPN_PORT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("VpnPort")
            .getter(getter(ClientVpnEndpoint::vpnPort))
            .setter(setter(Builder::vpnPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpnPort")
                    .unmarshallLocationName("vpnPort").build()).build();

    private static final SdkField<List<AssociatedTargetNetwork>> ASSOCIATED_TARGET_NETWORKS_FIELD = SdkField
            .<List<AssociatedTargetNetwork>> builder(MarshallingType.LIST)
            .memberName("AssociatedTargetNetworks")
            .getter(getter(ClientVpnEndpoint::associatedTargetNetworks))
            .setter(setter(Builder::associatedTargetNetworks))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociatedTargetNetwork")
                    .unmarshallLocationName("associatedTargetNetwork").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<AssociatedTargetNetwork> builder(MarshallingType.SDK_POJO)
                                            .constructor(AssociatedTargetNetwork::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> SERVER_CERTIFICATE_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ServerCertificateArn")
            .getter(getter(ClientVpnEndpoint::serverCertificateArn))
            .setter(setter(Builder::serverCertificateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerCertificateArn")
                    .unmarshallLocationName("serverCertificateArn").build()).build();

    private static final SdkField<List<ClientVpnAuthentication>> AUTHENTICATION_OPTIONS_FIELD = SdkField
            .<List<ClientVpnAuthentication>> builder(MarshallingType.LIST)
            .memberName("AuthenticationOptions")
            .getter(getter(ClientVpnEndpoint::authenticationOptions))
            .setter(setter(Builder::authenticationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AuthenticationOptions")
                    .unmarshallLocationName("authenticationOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<ClientVpnAuthentication> builder(MarshallingType.SDK_POJO)
                                            .constructor(ClientVpnAuthentication::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<ConnectionLogResponseOptions> CONNECTION_LOG_OPTIONS_FIELD = SdkField
            .<ConnectionLogResponseOptions> builder(MarshallingType.SDK_POJO)
            .memberName("ConnectionLogOptions")
            .getter(getter(ClientVpnEndpoint::connectionLogOptions))
            .setter(setter(Builder::connectionLogOptions))
            .constructor(ConnectionLogResponseOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConnectionLogOptions")
                    .unmarshallLocationName("connectionLogOptions").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(ClientVpnEndpoint::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroupIds")
            .getter(getter(ClientVpnEndpoint::securityGroupIds))
            .setter(setter(Builder::securityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupIdSet")
                    .unmarshallLocationName("securityGroupIdSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcId")
            .getter(getter(ClientVpnEndpoint::vpcId))
            .setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId")
                    .unmarshallLocationName("vpcId").build()).build();

    private static final SdkField<String> SELF_SERVICE_PORTAL_URL_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SelfServicePortalUrl")
            .getter(getter(ClientVpnEndpoint::selfServicePortalUrl))
            .setter(setter(Builder::selfServicePortalUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SelfServicePortalUrl")
                    .unmarshallLocationName("selfServicePortalUrl").build()).build();

    private static final SdkField<ClientConnectResponseOptions> CLIENT_CONNECT_OPTIONS_FIELD = SdkField
            .<ClientConnectResponseOptions> builder(MarshallingType.SDK_POJO)
            .memberName("ClientConnectOptions")
            .getter(getter(ClientVpnEndpoint::clientConnectOptions))
            .setter(setter(Builder::clientConnectOptions))
            .constructor(ClientConnectResponseOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientConnectOptions")
                    .unmarshallLocationName("clientConnectOptions").build()).build();

    private static final SdkField<Integer> SESSION_TIMEOUT_HOURS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("SessionTimeoutHours")
            .getter(getter(ClientVpnEndpoint::sessionTimeoutHours))
            .setter(setter(Builder::sessionTimeoutHours))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SessionTimeoutHours")
                    .unmarshallLocationName("sessionTimeoutHours").build()).build();

    private static final SdkField<ClientLoginBannerResponseOptions> CLIENT_LOGIN_BANNER_OPTIONS_FIELD = SdkField
            .<ClientLoginBannerResponseOptions> builder(MarshallingType.SDK_POJO)
            .memberName("ClientLoginBannerOptions")
            .getter(getter(ClientVpnEndpoint::clientLoginBannerOptions))
            .setter(setter(Builder::clientLoginBannerOptions))
            .constructor(ClientLoginBannerResponseOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientLoginBannerOptions")
                    .unmarshallLocationName("clientLoginBannerOptions").build()).build();

    private static final SdkField<Boolean> DISCONNECT_ON_SESSION_TIMEOUT_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DisconnectOnSessionTimeout")
            .getter(getter(ClientVpnEndpoint::disconnectOnSessionTimeout))
            .setter(setter(Builder::disconnectOnSessionTimeout))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisconnectOnSessionTimeout")
                    .unmarshallLocationName("disconnectOnSessionTimeout").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLIENT_VPN_ENDPOINT_ID_FIELD,
            DESCRIPTION_FIELD, STATUS_FIELD, CREATION_TIME_FIELD, DELETION_TIME_FIELD, DNS_NAME_FIELD, CLIENT_CIDR_BLOCK_FIELD,
            DNS_SERVERS_FIELD, SPLIT_TUNNEL_FIELD, VPN_PROTOCOL_FIELD, TRANSPORT_PROTOCOL_FIELD, VPN_PORT_FIELD,
            ASSOCIATED_TARGET_NETWORKS_FIELD, SERVER_CERTIFICATE_ARN_FIELD, AUTHENTICATION_OPTIONS_FIELD,
            CONNECTION_LOG_OPTIONS_FIELD, TAGS_FIELD, SECURITY_GROUP_IDS_FIELD, VPC_ID_FIELD, SELF_SERVICE_PORTAL_URL_FIELD,
            CLIENT_CONNECT_OPTIONS_FIELD, SESSION_TIMEOUT_HOURS_FIELD, CLIENT_LOGIN_BANNER_OPTIONS_FIELD,
            DISCONNECT_ON_SESSION_TIMEOUT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String clientVpnEndpointId;

    private final String description;

    private final ClientVpnEndpointStatus status;

    private final String creationTime;

    private final String deletionTime;

    private final String dnsName;

    private final String clientCidrBlock;

    private final List<String> dnsServers;

    private final Boolean splitTunnel;

    private final String vpnProtocol;

    private final String transportProtocol;

    private final Integer vpnPort;

    private final List<AssociatedTargetNetwork> associatedTargetNetworks;

    private final String serverCertificateArn;

    private final List<ClientVpnAuthentication> authenticationOptions;

    private final ConnectionLogResponseOptions connectionLogOptions;

    private final List<Tag> tags;

    private final List<String> securityGroupIds;

    private final String vpcId;

    private final String selfServicePortalUrl;

    private final ClientConnectResponseOptions clientConnectOptions;

    private final Integer sessionTimeoutHours;

    private final ClientLoginBannerResponseOptions clientLoginBannerOptions;

    private final Boolean disconnectOnSessionTimeout;

    private ClientVpnEndpoint(BuilderImpl builder) {
        this.clientVpnEndpointId = builder.clientVpnEndpointId;
        this.description = builder.description;
        this.status = builder.status;
        this.creationTime = builder.creationTime;
        this.deletionTime = builder.deletionTime;
        this.dnsName = builder.dnsName;
        this.clientCidrBlock = builder.clientCidrBlock;
        this.dnsServers = builder.dnsServers;
        this.splitTunnel = builder.splitTunnel;
        this.vpnProtocol = builder.vpnProtocol;
        this.transportProtocol = builder.transportProtocol;
        this.vpnPort = builder.vpnPort;
        this.associatedTargetNetworks = builder.associatedTargetNetworks;
        this.serverCertificateArn = builder.serverCertificateArn;
        this.authenticationOptions = builder.authenticationOptions;
        this.connectionLogOptions = builder.connectionLogOptions;
        this.tags = builder.tags;
        this.securityGroupIds = builder.securityGroupIds;
        this.vpcId = builder.vpcId;
        this.selfServicePortalUrl = builder.selfServicePortalUrl;
        this.clientConnectOptions = builder.clientConnectOptions;
        this.sessionTimeoutHours = builder.sessionTimeoutHours;
        this.clientLoginBannerOptions = builder.clientLoginBannerOptions;
        this.disconnectOnSessionTimeout = builder.disconnectOnSessionTimeout;
    }

    /**
     * <p>
     * The ID of the Client VPN endpoint.
     * </p>
     * 
     * @return The ID of the Client VPN endpoint.
     */
    public final String clientVpnEndpointId() {
        return clientVpnEndpointId;
    }

    /**
     * <p>
     * A brief description of the endpoint.
     * </p>
     * 
     * @return A brief description of the endpoint.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The current state of the Client VPN endpoint.
     * </p>
     * 
     * @return The current state of the Client VPN endpoint.
     */
    public final ClientVpnEndpointStatus status() {
        return status;
    }

    /**
     * <p>
     * The date and time the Client VPN endpoint was created.
     * </p>
     * 
     * @return The date and time the Client VPN endpoint was created.
     */
    public final String creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The date and time the Client VPN endpoint was deleted, if applicable.
     * </p>
     * 
     * @return The date and time the Client VPN endpoint was deleted, if applicable.
     */
    public final String deletionTime() {
        return deletionTime;
    }

    /**
     * <p>
     * The DNS name to be used by clients when connecting to the Client VPN endpoint.
     * </p>
     * 
     * @return The DNS name to be used by clients when connecting to the Client VPN endpoint.
     */
    public final String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * The IPv4 address range, in CIDR notation, from which client IP addresses are assigned.
     * </p>
     * 
     * @return The IPv4 address range, in CIDR notation, from which client IP addresses are assigned.
     */
    public final String clientCidrBlock() {
        return clientCidrBlock;
    }

    /**
     * For responses, this returns true if the service returned a value for the DnsServers property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDnsServers() {
        return dnsServers != null && !(dnsServers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the DNS servers to be used for DNS resolution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDnsServers} method.
     * </p>
     * 
     * @return Information about the DNS servers to be used for DNS resolution.
     */
    public final List<String> dnsServers() {
        return dnsServers;
    }

    /**
     * <p>
     * Indicates whether split-tunnel is enabled in the Client VPN endpoint.
     * </p>
     * <p>
     * For information about split-tunnel VPN endpoints, see <a
     * href="https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/split-tunnel-vpn.html">Split-Tunnel Client VPN
     * endpoint</a> in the <i>Client VPN Administrator Guide</i>.
     * </p>
     * 
     * @return Indicates whether split-tunnel is enabled in the Client VPN endpoint.</p>
     *         <p>
     *         For information about split-tunnel VPN endpoints, see <a
     *         href="https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/split-tunnel-vpn.html">Split-Tunnel Client
     *         VPN endpoint</a> in the <i>Client VPN Administrator Guide</i>.
     */
    public final Boolean splitTunnel() {
        return splitTunnel;
    }

    /**
     * <p>
     * The protocol used by the VPN session.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #vpnProtocol} will
     * return {@link VpnProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #vpnProtocolAsString}.
     * </p>
     * 
     * @return The protocol used by the VPN session.
     * @see VpnProtocol
     */
    public final VpnProtocol vpnProtocol() {
        return VpnProtocol.fromValue(vpnProtocol);
    }

    /**
     * <p>
     * The protocol used by the VPN session.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #vpnProtocol} will
     * return {@link VpnProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #vpnProtocolAsString}.
     * </p>
     * 
     * @return The protocol used by the VPN session.
     * @see VpnProtocol
     */
    public final String vpnProtocolAsString() {
        return vpnProtocol;
    }

    /**
     * <p>
     * The transport protocol used by the Client VPN endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #transportProtocol}
     * will return {@link TransportProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #transportProtocolAsString}.
     * </p>
     * 
     * @return The transport protocol used by the Client VPN endpoint.
     * @see TransportProtocol
     */
    public final TransportProtocol transportProtocol() {
        return TransportProtocol.fromValue(transportProtocol);
    }

    /**
     * <p>
     * The transport protocol used by the Client VPN endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #transportProtocol}
     * will return {@link TransportProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #transportProtocolAsString}.
     * </p>
     * 
     * @return The transport protocol used by the Client VPN endpoint.
     * @see TransportProtocol
     */
    public final String transportProtocolAsString() {
        return transportProtocol;
    }

    /**
     * <p>
     * The port number for the Client VPN endpoint.
     * </p>
     * 
     * @return The port number for the Client VPN endpoint.
     */
    public final Integer vpnPort() {
        return vpnPort;
    }

    /**
     * For responses, this returns true if the service returned a value for the AssociatedTargetNetworks property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     *
     * @deprecated This property is deprecated. To view the target networks associated with a Client VPN endpoint, call
     *             DescribeClientVpnTargetNetworks and inspect the clientVpnTargetNetworks response element.
     */
    @Deprecated
    public final boolean hasAssociatedTargetNetworks() {
        return associatedTargetNetworks != null && !(associatedTargetNetworks instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the associated target networks. A target network is a subnet in a VPC.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAssociatedTargetNetworks} method.
     * </p>
     * 
     * @return Information about the associated target networks. A target network is a subnet in a VPC.
     * @deprecated This property is deprecated. To view the target networks associated with a Client VPN endpoint, call
     *             DescribeClientVpnTargetNetworks and inspect the clientVpnTargetNetworks response element.
     */
    @Deprecated
    public final List<AssociatedTargetNetwork> associatedTargetNetworks() {
        return associatedTargetNetworks;
    }

    /**
     * <p>
     * The ARN of the server certificate.
     * </p>
     * 
     * @return The ARN of the server certificate.
     */
    public final String serverCertificateArn() {
        return serverCertificateArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the AuthenticationOptions property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAuthenticationOptions() {
        return authenticationOptions != null && !(authenticationOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the authentication method used by the Client VPN endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAuthenticationOptions} method.
     * </p>
     * 
     * @return Information about the authentication method used by the Client VPN endpoint.
     */
    public final List<ClientVpnAuthentication> authenticationOptions() {
        return authenticationOptions;
    }

    /**
     * <p>
     * Information about the client connection logging options for the Client VPN endpoint.
     * </p>
     * 
     * @return Information about the client connection logging options for the Client VPN endpoint.
     */
    public final ConnectionLogResponseOptions connectionLogOptions() {
        return connectionLogOptions;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Any tags assigned to the Client VPN endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Any tags assigned to the Client VPN endpoint.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroupIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroupIds() {
        return securityGroupIds != null && !(securityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the security groups for the target network.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroupIds} method.
     * </p>
     * 
     * @return The IDs of the security groups for the target network.
     */
    public final List<String> securityGroupIds() {
        return securityGroupIds;
    }

    /**
     * <p>
     * The ID of the VPC.
     * </p>
     * 
     * @return The ID of the VPC.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * The URL of the self-service portal.
     * </p>
     * 
     * @return The URL of the self-service portal.
     */
    public final String selfServicePortalUrl() {
        return selfServicePortalUrl;
    }

    /**
     * <p>
     * The options for managing connection authorization for new client connections.
     * </p>
     * 
     * @return The options for managing connection authorization for new client connections.
     */
    public final ClientConnectResponseOptions clientConnectOptions() {
        return clientConnectOptions;
    }

    /**
     * <p>
     * The maximum VPN session duration time in hours.
     * </p>
     * <p>
     * Valid values: <code>8 | 10 | 12 | 24</code>
     * </p>
     * <p>
     * Default value: <code>24</code>
     * </p>
     * 
     * @return The maximum VPN session duration time in hours.</p>
     *         <p>
     *         Valid values: <code>8 | 10 | 12 | 24</code>
     *         </p>
     *         <p>
     *         Default value: <code>24</code>
     */
    public final Integer sessionTimeoutHours() {
        return sessionTimeoutHours;
    }

    /**
     * <p>
     * Options for enabling a customizable text banner that will be displayed on Amazon Web Services provided clients
     * when a VPN session is established.
     * </p>
     * 
     * @return Options for enabling a customizable text banner that will be displayed on Amazon Web Services provided
     *         clients when a VPN session is established.
     */
    public final ClientLoginBannerResponseOptions clientLoginBannerOptions() {
        return clientLoginBannerOptions;
    }

    /**
     * <p>
     * Indicates whether the client VPN session is disconnected after the maximum <code>sessionTimeoutHours</code> is
     * reached. If <code>true</code>, users are prompted to reconnect client VPN. If <code>false</code>, client VPN
     * attempts to reconnect automatically. The default value is <code>false</code>.
     * </p>
     * 
     * @return Indicates whether the client VPN session is disconnected after the maximum
     *         <code>sessionTimeoutHours</code> is reached. If <code>true</code>, users are prompted to reconnect client
     *         VPN. If <code>false</code>, client VPN attempts to reconnect automatically. The default value is
     *         <code>false</code>.
     */
    public final Boolean disconnectOnSessionTimeout() {
        return disconnectOnSessionTimeout;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(clientVpnEndpointId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(deletionTime());
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(clientCidrBlock());
        hashCode = 31 * hashCode + Objects.hashCode(hasDnsServers() ? dnsServers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(splitTunnel());
        hashCode = 31 * hashCode + Objects.hashCode(vpnProtocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(transportProtocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpnPort());
        hashCode = 31 * hashCode + Objects.hashCode(hasAssociatedTargetNetworks() ? associatedTargetNetworks() : null);
        hashCode = 31 * hashCode + Objects.hashCode(serverCertificateArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasAuthenticationOptions() ? authenticationOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(connectionLogOptions());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupIds() ? securityGroupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(selfServicePortalUrl());
        hashCode = 31 * hashCode + Objects.hashCode(clientConnectOptions());
        hashCode = 31 * hashCode + Objects.hashCode(sessionTimeoutHours());
        hashCode = 31 * hashCode + Objects.hashCode(clientLoginBannerOptions());
        hashCode = 31 * hashCode + Objects.hashCode(disconnectOnSessionTimeout());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ClientVpnEndpoint)) {
            return false;
        }
        ClientVpnEndpoint other = (ClientVpnEndpoint) obj;
        return Objects.equals(clientVpnEndpointId(), other.clientVpnEndpointId())
                && Objects.equals(description(), other.description()) && Objects.equals(status(), other.status())
                && Objects.equals(creationTime(), other.creationTime()) && Objects.equals(deletionTime(), other.deletionTime())
                && Objects.equals(dnsName(), other.dnsName()) && Objects.equals(clientCidrBlock(), other.clientCidrBlock())
                && hasDnsServers() == other.hasDnsServers() && Objects.equals(dnsServers(), other.dnsServers())
                && Objects.equals(splitTunnel(), other.splitTunnel())
                && Objects.equals(vpnProtocolAsString(), other.vpnProtocolAsString())
                && Objects.equals(transportProtocolAsString(), other.transportProtocolAsString())
                && Objects.equals(vpnPort(), other.vpnPort())
                && hasAssociatedTargetNetworks() == other.hasAssociatedTargetNetworks()
                && Objects.equals(associatedTargetNetworks(), other.associatedTargetNetworks())
                && Objects.equals(serverCertificateArn(), other.serverCertificateArn())
                && hasAuthenticationOptions() == other.hasAuthenticationOptions()
                && Objects.equals(authenticationOptions(), other.authenticationOptions())
                && Objects.equals(connectionLogOptions(), other.connectionLogOptions()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && hasSecurityGroupIds() == other.hasSecurityGroupIds()
                && Objects.equals(securityGroupIds(), other.securityGroupIds()) && Objects.equals(vpcId(), other.vpcId())
                && Objects.equals(selfServicePortalUrl(), other.selfServicePortalUrl())
                && Objects.equals(clientConnectOptions(), other.clientConnectOptions())
                && Objects.equals(sessionTimeoutHours(), other.sessionTimeoutHours())
                && Objects.equals(clientLoginBannerOptions(), other.clientLoginBannerOptions())
                && Objects.equals(disconnectOnSessionTimeout(), other.disconnectOnSessionTimeout());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ClientVpnEndpoint").add("ClientVpnEndpointId", clientVpnEndpointId())
                .add("Description", description()).add("Status", status()).add("CreationTime", creationTime())
                .add("DeletionTime", deletionTime()).add("DnsName", dnsName()).add("ClientCidrBlock", clientCidrBlock())
                .add("DnsServers", hasDnsServers() ? dnsServers() : null).add("SplitTunnel", splitTunnel())
                .add("VpnProtocol", vpnProtocolAsString()).add("TransportProtocol", transportProtocolAsString())
                .add("VpnPort", vpnPort())
                .add("AssociatedTargetNetworks", hasAssociatedTargetNetworks() ? associatedTargetNetworks() : null)
                .add("ServerCertificateArn", serverCertificateArn())
                .add("AuthenticationOptions", hasAuthenticationOptions() ? authenticationOptions() : null)
                .add("ConnectionLogOptions", connectionLogOptions()).add("Tags", hasTags() ? tags() : null)
                .add("SecurityGroupIds", hasSecurityGroupIds() ? securityGroupIds() : null).add("VpcId", vpcId())
                .add("SelfServicePortalUrl", selfServicePortalUrl()).add("ClientConnectOptions", clientConnectOptions())
                .add("SessionTimeoutHours", sessionTimeoutHours()).add("ClientLoginBannerOptions", clientLoginBannerOptions())
                .add("DisconnectOnSessionTimeout", disconnectOnSessionTimeout()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClientVpnEndpointId":
            return Optional.ofNullable(clazz.cast(clientVpnEndpointId()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "DeletionTime":
            return Optional.ofNullable(clazz.cast(deletionTime()));
        case "DnsName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "ClientCidrBlock":
            return Optional.ofNullable(clazz.cast(clientCidrBlock()));
        case "DnsServers":
            return Optional.ofNullable(clazz.cast(dnsServers()));
        case "SplitTunnel":
            return Optional.ofNullable(clazz.cast(splitTunnel()));
        case "VpnProtocol":
            return Optional.ofNullable(clazz.cast(vpnProtocolAsString()));
        case "TransportProtocol":
            return Optional.ofNullable(clazz.cast(transportProtocolAsString()));
        case "VpnPort":
            return Optional.ofNullable(clazz.cast(vpnPort()));
        case "AssociatedTargetNetworks":
            return Optional.ofNullable(clazz.cast(associatedTargetNetworks()));
        case "ServerCertificateArn":
            return Optional.ofNullable(clazz.cast(serverCertificateArn()));
        case "AuthenticationOptions":
            return Optional.ofNullable(clazz.cast(authenticationOptions()));
        case "ConnectionLogOptions":
            return Optional.ofNullable(clazz.cast(connectionLogOptions()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "SecurityGroupIds":
            return Optional.ofNullable(clazz.cast(securityGroupIds()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "SelfServicePortalUrl":
            return Optional.ofNullable(clazz.cast(selfServicePortalUrl()));
        case "ClientConnectOptions":
            return Optional.ofNullable(clazz.cast(clientConnectOptions()));
        case "SessionTimeoutHours":
            return Optional.ofNullable(clazz.cast(sessionTimeoutHours()));
        case "ClientLoginBannerOptions":
            return Optional.ofNullable(clazz.cast(clientLoginBannerOptions()));
        case "DisconnectOnSessionTimeout":
            return Optional.ofNullable(clazz.cast(disconnectOnSessionTimeout()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ClientVpnEndpointId", CLIENT_VPN_ENDPOINT_ID_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("CreationTime", CREATION_TIME_FIELD);
        map.put("DeletionTime", DELETION_TIME_FIELD);
        map.put("DnsName", DNS_NAME_FIELD);
        map.put("ClientCidrBlock", CLIENT_CIDR_BLOCK_FIELD);
        map.put("DnsServer", DNS_SERVERS_FIELD);
        map.put("SplitTunnel", SPLIT_TUNNEL_FIELD);
        map.put("VpnProtocol", VPN_PROTOCOL_FIELD);
        map.put("TransportProtocol", TRANSPORT_PROTOCOL_FIELD);
        map.put("VpnPort", VPN_PORT_FIELD);
        map.put("AssociatedTargetNetwork", ASSOCIATED_TARGET_NETWORKS_FIELD);
        map.put("ServerCertificateArn", SERVER_CERTIFICATE_ARN_FIELD);
        map.put("AuthenticationOptions", AUTHENTICATION_OPTIONS_FIELD);
        map.put("ConnectionLogOptions", CONNECTION_LOG_OPTIONS_FIELD);
        map.put("TagSet", TAGS_FIELD);
        map.put("SecurityGroupIdSet", SECURITY_GROUP_IDS_FIELD);
        map.put("VpcId", VPC_ID_FIELD);
        map.put("SelfServicePortalUrl", SELF_SERVICE_PORTAL_URL_FIELD);
        map.put("ClientConnectOptions", CLIENT_CONNECT_OPTIONS_FIELD);
        map.put("SessionTimeoutHours", SESSION_TIMEOUT_HOURS_FIELD);
        map.put("ClientLoginBannerOptions", CLIENT_LOGIN_BANNER_OPTIONS_FIELD);
        map.put("DisconnectOnSessionTimeout", DISCONNECT_ON_SESSION_TIMEOUT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ClientVpnEndpoint, T> g) {
        return obj -> g.apply((ClientVpnEndpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ClientVpnEndpoint> {
        /**
         * <p>
         * The ID of the Client VPN endpoint.
         * </p>
         * 
         * @param clientVpnEndpointId
         *        The ID of the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientVpnEndpointId(String clientVpnEndpointId);

        /**
         * <p>
         * A brief description of the endpoint.
         * </p>
         * 
         * @param description
         *        A brief description of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The current state of the Client VPN endpoint.
         * </p>
         * 
         * @param status
         *        The current state of the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(ClientVpnEndpointStatus status);

        /**
         * <p>
         * The current state of the Client VPN endpoint.
         * </p>
         * This is a convenience method that creates an instance of the {@link ClientVpnEndpointStatus.Builder} avoiding
         * the need to create one manually via {@link ClientVpnEndpointStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ClientVpnEndpointStatus.Builder#build()} is called immediately
         * and its result is passed to {@link #status(ClientVpnEndpointStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link ClientVpnEndpointStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(ClientVpnEndpointStatus)
         */
        default Builder status(Consumer<ClientVpnEndpointStatus.Builder> status) {
            return status(ClientVpnEndpointStatus.builder().applyMutation(status).build());
        }

        /**
         * <p>
         * The date and time the Client VPN endpoint was created.
         * </p>
         * 
         * @param creationTime
         *        The date and time the Client VPN endpoint was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(String creationTime);

        /**
         * <p>
         * The date and time the Client VPN endpoint was deleted, if applicable.
         * </p>
         * 
         * @param deletionTime
         *        The date and time the Client VPN endpoint was deleted, if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletionTime(String deletionTime);

        /**
         * <p>
         * The DNS name to be used by clients when connecting to the Client VPN endpoint.
         * </p>
         * 
         * @param dnsName
         *        The DNS name to be used by clients when connecting to the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * The IPv4 address range, in CIDR notation, from which client IP addresses are assigned.
         * </p>
         * 
         * @param clientCidrBlock
         *        The IPv4 address range, in CIDR notation, from which client IP addresses are assigned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientCidrBlock(String clientCidrBlock);

        /**
         * <p>
         * Information about the DNS servers to be used for DNS resolution.
         * </p>
         * 
         * @param dnsServers
         *        Information about the DNS servers to be used for DNS resolution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsServers(Collection<String> dnsServers);

        /**
         * <p>
         * Information about the DNS servers to be used for DNS resolution.
         * </p>
         * 
         * @param dnsServers
         *        Information about the DNS servers to be used for DNS resolution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsServers(String... dnsServers);

        /**
         * <p>
         * Indicates whether split-tunnel is enabled in the Client VPN endpoint.
         * </p>
         * <p>
         * For information about split-tunnel VPN endpoints, see <a
         * href="https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/split-tunnel-vpn.html">Split-Tunnel Client VPN
         * endpoint</a> in the <i>Client VPN Administrator Guide</i>.
         * </p>
         * 
         * @param splitTunnel
         *        Indicates whether split-tunnel is enabled in the Client VPN endpoint.</p>
         *        <p>
         *        For information about split-tunnel VPN endpoints, see <a
         *        href="https://docs.aws.amazon.com/vpn/latest/clientvpn-admin/split-tunnel-vpn.html">Split-Tunnel
         *        Client VPN endpoint</a> in the <i>Client VPN Administrator Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder splitTunnel(Boolean splitTunnel);

        /**
         * <p>
         * The protocol used by the VPN session.
         * </p>
         * 
         * @param vpnProtocol
         *        The protocol used by the VPN session.
         * @see VpnProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpnProtocol
         */
        Builder vpnProtocol(String vpnProtocol);

        /**
         * <p>
         * The protocol used by the VPN session.
         * </p>
         * 
         * @param vpnProtocol
         *        The protocol used by the VPN session.
         * @see VpnProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpnProtocol
         */
        Builder vpnProtocol(VpnProtocol vpnProtocol);

        /**
         * <p>
         * The transport protocol used by the Client VPN endpoint.
         * </p>
         * 
         * @param transportProtocol
         *        The transport protocol used by the Client VPN endpoint.
         * @see TransportProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransportProtocol
         */
        Builder transportProtocol(String transportProtocol);

        /**
         * <p>
         * The transport protocol used by the Client VPN endpoint.
         * </p>
         * 
         * @param transportProtocol
         *        The transport protocol used by the Client VPN endpoint.
         * @see TransportProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransportProtocol
         */
        Builder transportProtocol(TransportProtocol transportProtocol);

        /**
         * <p>
         * The port number for the Client VPN endpoint.
         * </p>
         * 
         * @param vpnPort
         *        The port number for the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpnPort(Integer vpnPort);

        /**
         * <p>
         * Information about the associated target networks. A target network is a subnet in a VPC.
         * </p>
         * 
         * @param associatedTargetNetworks
         *        Information about the associated target networks. A target network is a subnet in a VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This property is deprecated. To view the target networks associated with a Client VPN endpoint,
         *             call DescribeClientVpnTargetNetworks and inspect the clientVpnTargetNetworks response element.
         */
        @Deprecated
        Builder associatedTargetNetworks(Collection<AssociatedTargetNetwork> associatedTargetNetworks);

        /**
         * <p>
         * Information about the associated target networks. A target network is a subnet in a VPC.
         * </p>
         * 
         * @param associatedTargetNetworks
         *        Information about the associated target networks. A target network is a subnet in a VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         * @deprecated This property is deprecated. To view the target networks associated with a Client VPN endpoint,
         *             call DescribeClientVpnTargetNetworks and inspect the clientVpnTargetNetworks response element.
         */
        @Deprecated
        Builder associatedTargetNetworks(AssociatedTargetNetwork... associatedTargetNetworks);

        /**
         * <p>
         * Information about the associated target networks. A target network is a subnet in a VPC.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.AssociatedTargetNetwork.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.ec2.model.AssociatedTargetNetwork#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.AssociatedTargetNetwork.Builder#build()} is called
         * immediately and its result is passed to {@link #associatedTargetNetworks(List<AssociatedTargetNetwork>)}.
         * 
         * @param associatedTargetNetworks
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.AssociatedTargetNetwork.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #associatedTargetNetworks(java.util.Collection<AssociatedTargetNetwork>)
         * @deprecated This property is deprecated. To view the target networks associated with a Client VPN endpoint,
         *             call DescribeClientVpnTargetNetworks and inspect the clientVpnTargetNetworks response element.
         */
        @Deprecated
        Builder associatedTargetNetworks(Consumer<AssociatedTargetNetwork.Builder>... associatedTargetNetworks);

        /**
         * <p>
         * The ARN of the server certificate.
         * </p>
         * 
         * @param serverCertificateArn
         *        The ARN of the server certificate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverCertificateArn(String serverCertificateArn);

        /**
         * <p>
         * Information about the authentication method used by the Client VPN endpoint.
         * </p>
         * 
         * @param authenticationOptions
         *        Information about the authentication method used by the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authenticationOptions(Collection<ClientVpnAuthentication> authenticationOptions);

        /**
         * <p>
         * Information about the authentication method used by the Client VPN endpoint.
         * </p>
         * 
         * @param authenticationOptions
         *        Information about the authentication method used by the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authenticationOptions(ClientVpnAuthentication... authenticationOptions);

        /**
         * <p>
         * Information about the authentication method used by the Client VPN endpoint.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.ClientVpnAuthentication.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.ec2.model.ClientVpnAuthentication#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.ClientVpnAuthentication.Builder#build()} is called
         * immediately and its result is passed to {@link #authenticationOptions(List<ClientVpnAuthentication>)}.
         * 
         * @param authenticationOptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.ClientVpnAuthentication.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #authenticationOptions(java.util.Collection<ClientVpnAuthentication>)
         */
        Builder authenticationOptions(Consumer<ClientVpnAuthentication.Builder>... authenticationOptions);

        /**
         * <p>
         * Information about the client connection logging options for the Client VPN endpoint.
         * </p>
         * 
         * @param connectionLogOptions
         *        Information about the client connection logging options for the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder connectionLogOptions(ConnectionLogResponseOptions connectionLogOptions);

        /**
         * <p>
         * Information about the client connection logging options for the Client VPN endpoint.
         * </p>
         * This is a convenience method that creates an instance of the {@link ConnectionLogResponseOptions.Builder}
         * avoiding the need to create one manually via {@link ConnectionLogResponseOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ConnectionLogResponseOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #connectionLogOptions(ConnectionLogResponseOptions)}.
         * 
         * @param connectionLogOptions
         *        a consumer that will call methods on {@link ConnectionLogResponseOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #connectionLogOptions(ConnectionLogResponseOptions)
         */
        default Builder connectionLogOptions(Consumer<ConnectionLogResponseOptions.Builder> connectionLogOptions) {
            return connectionLogOptions(ConnectionLogResponseOptions.builder().applyMutation(connectionLogOptions).build());
        }

        /**
         * <p>
         * Any tags assigned to the Client VPN endpoint.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Any tags assigned to the Client VPN endpoint.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the Client VPN endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Any tags assigned to the Client VPN endpoint.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The IDs of the security groups for the target network.
         * </p>
         * 
         * @param securityGroupIds
         *        The IDs of the security groups for the target network.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * The IDs of the security groups for the target network.
         * </p>
         * 
         * @param securityGroupIds
         *        The IDs of the security groups for the target network.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);

        /**
         * <p>
         * The ID of the VPC.
         * </p>
         * 
         * @param vpcId
         *        The ID of the VPC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The URL of the self-service portal.
         * </p>
         * 
         * @param selfServicePortalUrl
         *        The URL of the self-service portal.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder selfServicePortalUrl(String selfServicePortalUrl);

        /**
         * <p>
         * The options for managing connection authorization for new client connections.
         * </p>
         * 
         * @param clientConnectOptions
         *        The options for managing connection authorization for new client connections.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientConnectOptions(ClientConnectResponseOptions clientConnectOptions);

        /**
         * <p>
         * The options for managing connection authorization for new client connections.
         * </p>
         * This is a convenience method that creates an instance of the {@link ClientConnectResponseOptions.Builder}
         * avoiding the need to create one manually via {@link ClientConnectResponseOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ClientConnectResponseOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #clientConnectOptions(ClientConnectResponseOptions)}.
         * 
         * @param clientConnectOptions
         *        a consumer that will call methods on {@link ClientConnectResponseOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #clientConnectOptions(ClientConnectResponseOptions)
         */
        default Builder clientConnectOptions(Consumer<ClientConnectResponseOptions.Builder> clientConnectOptions) {
            return clientConnectOptions(ClientConnectResponseOptions.builder().applyMutation(clientConnectOptions).build());
        }

        /**
         * <p>
         * The maximum VPN session duration time in hours.
         * </p>
         * <p>
         * Valid values: <code>8 | 10 | 12 | 24</code>
         * </p>
         * <p>
         * Default value: <code>24</code>
         * </p>
         * 
         * @param sessionTimeoutHours
         *        The maximum VPN session duration time in hours.</p>
         *        <p>
         *        Valid values: <code>8 | 10 | 12 | 24</code>
         *        </p>
         *        <p>
         *        Default value: <code>24</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sessionTimeoutHours(Integer sessionTimeoutHours);

        /**
         * <p>
         * Options for enabling a customizable text banner that will be displayed on Amazon Web Services provided
         * clients when a VPN session is established.
         * </p>
         * 
         * @param clientLoginBannerOptions
         *        Options for enabling a customizable text banner that will be displayed on Amazon Web Services provided
         *        clients when a VPN session is established.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientLoginBannerOptions(ClientLoginBannerResponseOptions clientLoginBannerOptions);

        /**
         * <p>
         * Options for enabling a customizable text banner that will be displayed on Amazon Web Services provided
         * clients when a VPN session is established.
         * </p>
         * This is a convenience method that creates an instance of the {@link ClientLoginBannerResponseOptions.Builder}
         * avoiding the need to create one manually via {@link ClientLoginBannerResponseOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ClientLoginBannerResponseOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #clientLoginBannerOptions(ClientLoginBannerResponseOptions)}.
         * 
         * @param clientLoginBannerOptions
         *        a consumer that will call methods on {@link ClientLoginBannerResponseOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #clientLoginBannerOptions(ClientLoginBannerResponseOptions)
         */
        default Builder clientLoginBannerOptions(Consumer<ClientLoginBannerResponseOptions.Builder> clientLoginBannerOptions) {
            return clientLoginBannerOptions(ClientLoginBannerResponseOptions.builder().applyMutation(clientLoginBannerOptions)
                    .build());
        }

        /**
         * <p>
         * Indicates whether the client VPN session is disconnected after the maximum <code>sessionTimeoutHours</code>
         * is reached. If <code>true</code>, users are prompted to reconnect client VPN. If <code>false</code>, client
         * VPN attempts to reconnect automatically. The default value is <code>false</code>.
         * </p>
         * 
         * @param disconnectOnSessionTimeout
         *        Indicates whether the client VPN session is disconnected after the maximum
         *        <code>sessionTimeoutHours</code> is reached. If <code>true</code>, users are prompted to reconnect
         *        client VPN. If <code>false</code>, client VPN attempts to reconnect automatically. The default value
         *        is <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder disconnectOnSessionTimeout(Boolean disconnectOnSessionTimeout);
    }

    static final class BuilderImpl implements Builder {
        private String clientVpnEndpointId;

        private String description;

        private ClientVpnEndpointStatus status;

        private String creationTime;

        private String deletionTime;

        private String dnsName;

        private String clientCidrBlock;

        private List<String> dnsServers = DefaultSdkAutoConstructList.getInstance();

        private Boolean splitTunnel;

        private String vpnProtocol;

        private String transportProtocol;

        private Integer vpnPort;

        private List<AssociatedTargetNetwork> associatedTargetNetworks = DefaultSdkAutoConstructList.getInstance();

        private String serverCertificateArn;

        private List<ClientVpnAuthentication> authenticationOptions = DefaultSdkAutoConstructList.getInstance();

        private ConnectionLogResponseOptions connectionLogOptions;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private List<String> securityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private String vpcId;

        private String selfServicePortalUrl;

        private ClientConnectResponseOptions clientConnectOptions;

        private Integer sessionTimeoutHours;

        private ClientLoginBannerResponseOptions clientLoginBannerOptions;

        private Boolean disconnectOnSessionTimeout;

        private BuilderImpl() {
        }

        private BuilderImpl(ClientVpnEndpoint model) {
            clientVpnEndpointId(model.clientVpnEndpointId);
            description(model.description);
            status(model.status);
            creationTime(model.creationTime);
            deletionTime(model.deletionTime);
            dnsName(model.dnsName);
            clientCidrBlock(model.clientCidrBlock);
            dnsServers(model.dnsServers);
            splitTunnel(model.splitTunnel);
            vpnProtocol(model.vpnProtocol);
            transportProtocol(model.transportProtocol);
            vpnPort(model.vpnPort);
            associatedTargetNetworks(model.associatedTargetNetworks);
            serverCertificateArn(model.serverCertificateArn);
            authenticationOptions(model.authenticationOptions);
            connectionLogOptions(model.connectionLogOptions);
            tags(model.tags);
            securityGroupIds(model.securityGroupIds);
            vpcId(model.vpcId);
            selfServicePortalUrl(model.selfServicePortalUrl);
            clientConnectOptions(model.clientConnectOptions);
            sessionTimeoutHours(model.sessionTimeoutHours);
            clientLoginBannerOptions(model.clientLoginBannerOptions);
            disconnectOnSessionTimeout(model.disconnectOnSessionTimeout);
        }

        public final String getClientVpnEndpointId() {
            return clientVpnEndpointId;
        }

        public final void setClientVpnEndpointId(String clientVpnEndpointId) {
            this.clientVpnEndpointId = clientVpnEndpointId;
        }

        @Override
        public final Builder clientVpnEndpointId(String clientVpnEndpointId) {
            this.clientVpnEndpointId = clientVpnEndpointId;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final ClientVpnEndpointStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        public final void setStatus(ClientVpnEndpointStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        @Override
        public final Builder status(ClientVpnEndpointStatus status) {
            this.status = status;
            return this;
        }

        public final String getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(String creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(String creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getDeletionTime() {
            return deletionTime;
        }

        public final void setDeletionTime(String deletionTime) {
            this.deletionTime = deletionTime;
        }

        @Override
        public final Builder deletionTime(String deletionTime) {
            this.deletionTime = deletionTime;
            return this;
        }

        public final String getDnsName() {
            return dnsName;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final String getClientCidrBlock() {
            return clientCidrBlock;
        }

        public final void setClientCidrBlock(String clientCidrBlock) {
            this.clientCidrBlock = clientCidrBlock;
        }

        @Override
        public final Builder clientCidrBlock(String clientCidrBlock) {
            this.clientCidrBlock = clientCidrBlock;
            return this;
        }

        public final Collection<String> getDnsServers() {
            if (dnsServers instanceof SdkAutoConstructList) {
                return null;
            }
            return dnsServers;
        }

        public final void setDnsServers(Collection<String> dnsServers) {
            this.dnsServers = ValueStringListCopier.copy(dnsServers);
        }

        @Override
        public final Builder dnsServers(Collection<String> dnsServers) {
            this.dnsServers = ValueStringListCopier.copy(dnsServers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dnsServers(String... dnsServers) {
            dnsServers(Arrays.asList(dnsServers));
            return this;
        }

        public final Boolean getSplitTunnel() {
            return splitTunnel;
        }

        public final void setSplitTunnel(Boolean splitTunnel) {
            this.splitTunnel = splitTunnel;
        }

        @Override
        public final Builder splitTunnel(Boolean splitTunnel) {
            this.splitTunnel = splitTunnel;
            return this;
        }

        public final String getVpnProtocol() {
            return vpnProtocol;
        }

        public final void setVpnProtocol(String vpnProtocol) {
            this.vpnProtocol = vpnProtocol;
        }

        @Override
        public final Builder vpnProtocol(String vpnProtocol) {
            this.vpnProtocol = vpnProtocol;
            return this;
        }

        @Override
        public final Builder vpnProtocol(VpnProtocol vpnProtocol) {
            this.vpnProtocol(vpnProtocol == null ? null : vpnProtocol.toString());
            return this;
        }

        public final String getTransportProtocol() {
            return transportProtocol;
        }

        public final void setTransportProtocol(String transportProtocol) {
            this.transportProtocol = transportProtocol;
        }

        @Override
        public final Builder transportProtocol(String transportProtocol) {
            this.transportProtocol = transportProtocol;
            return this;
        }

        @Override
        public final Builder transportProtocol(TransportProtocol transportProtocol) {
            this.transportProtocol(transportProtocol == null ? null : transportProtocol.toString());
            return this;
        }

        public final Integer getVpnPort() {
            return vpnPort;
        }

        public final void setVpnPort(Integer vpnPort) {
            this.vpnPort = vpnPort;
        }

        @Override
        public final Builder vpnPort(Integer vpnPort) {
            this.vpnPort = vpnPort;
            return this;
        }

        @Deprecated
        public final List<AssociatedTargetNetwork.Builder> getAssociatedTargetNetworks() {
            List<AssociatedTargetNetwork.Builder> result = AssociatedTargetNetworkSetCopier
                    .copyToBuilder(this.associatedTargetNetworks);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Deprecated
        public final void setAssociatedTargetNetworks(Collection<AssociatedTargetNetwork.BuilderImpl> associatedTargetNetworks) {
            this.associatedTargetNetworks = AssociatedTargetNetworkSetCopier.copyFromBuilder(associatedTargetNetworks);
        }

        @Override
        @Deprecated
        public final Builder associatedTargetNetworks(Collection<AssociatedTargetNetwork> associatedTargetNetworks) {
            this.associatedTargetNetworks = AssociatedTargetNetworkSetCopier.copy(associatedTargetNetworks);
            return this;
        }

        @Override
        @SafeVarargs
        @Deprecated
        public final Builder associatedTargetNetworks(AssociatedTargetNetwork... associatedTargetNetworks) {
            associatedTargetNetworks(Arrays.asList(associatedTargetNetworks));
            return this;
        }

        @Override
        @SafeVarargs
        @Deprecated
        public final Builder associatedTargetNetworks(Consumer<AssociatedTargetNetwork.Builder>... associatedTargetNetworks) {
            associatedTargetNetworks(Stream.of(associatedTargetNetworks)
                    .map(c -> AssociatedTargetNetwork.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getServerCertificateArn() {
            return serverCertificateArn;
        }

        public final void setServerCertificateArn(String serverCertificateArn) {
            this.serverCertificateArn = serverCertificateArn;
        }

        @Override
        public final Builder serverCertificateArn(String serverCertificateArn) {
            this.serverCertificateArn = serverCertificateArn;
            return this;
        }

        public final List<ClientVpnAuthentication.Builder> getAuthenticationOptions() {
            List<ClientVpnAuthentication.Builder> result = ClientVpnAuthenticationListCopier
                    .copyToBuilder(this.authenticationOptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAuthenticationOptions(Collection<ClientVpnAuthentication.BuilderImpl> authenticationOptions) {
            this.authenticationOptions = ClientVpnAuthenticationListCopier.copyFromBuilder(authenticationOptions);
        }

        @Override
        public final Builder authenticationOptions(Collection<ClientVpnAuthentication> authenticationOptions) {
            this.authenticationOptions = ClientVpnAuthenticationListCopier.copy(authenticationOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder authenticationOptions(ClientVpnAuthentication... authenticationOptions) {
            authenticationOptions(Arrays.asList(authenticationOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder authenticationOptions(Consumer<ClientVpnAuthentication.Builder>... authenticationOptions) {
            authenticationOptions(Stream.of(authenticationOptions)
                    .map(c -> ClientVpnAuthentication.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final ConnectionLogResponseOptions.Builder getConnectionLogOptions() {
            return connectionLogOptions != null ? connectionLogOptions.toBuilder() : null;
        }

        public final void setConnectionLogOptions(ConnectionLogResponseOptions.BuilderImpl connectionLogOptions) {
            this.connectionLogOptions = connectionLogOptions != null ? connectionLogOptions.build() : null;
        }

        @Override
        public final Builder connectionLogOptions(ConnectionLogResponseOptions connectionLogOptions) {
            this.connectionLogOptions = connectionLogOptions;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Collection<String> getSecurityGroupIds() {
            if (securityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupIds;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = ClientVpnSecurityGroupIdSetCopier.copy(securityGroupIds);
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = ClientVpnSecurityGroupIdSetCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final String getSelfServicePortalUrl() {
            return selfServicePortalUrl;
        }

        public final void setSelfServicePortalUrl(String selfServicePortalUrl) {
            this.selfServicePortalUrl = selfServicePortalUrl;
        }

        @Override
        public final Builder selfServicePortalUrl(String selfServicePortalUrl) {
            this.selfServicePortalUrl = selfServicePortalUrl;
            return this;
        }

        public final ClientConnectResponseOptions.Builder getClientConnectOptions() {
            return clientConnectOptions != null ? clientConnectOptions.toBuilder() : null;
        }

        public final void setClientConnectOptions(ClientConnectResponseOptions.BuilderImpl clientConnectOptions) {
            this.clientConnectOptions = clientConnectOptions != null ? clientConnectOptions.build() : null;
        }

        @Override
        public final Builder clientConnectOptions(ClientConnectResponseOptions clientConnectOptions) {
            this.clientConnectOptions = clientConnectOptions;
            return this;
        }

        public final Integer getSessionTimeoutHours() {
            return sessionTimeoutHours;
        }

        public final void setSessionTimeoutHours(Integer sessionTimeoutHours) {
            this.sessionTimeoutHours = sessionTimeoutHours;
        }

        @Override
        public final Builder sessionTimeoutHours(Integer sessionTimeoutHours) {
            this.sessionTimeoutHours = sessionTimeoutHours;
            return this;
        }

        public final ClientLoginBannerResponseOptions.Builder getClientLoginBannerOptions() {
            return clientLoginBannerOptions != null ? clientLoginBannerOptions.toBuilder() : null;
        }

        public final void setClientLoginBannerOptions(ClientLoginBannerResponseOptions.BuilderImpl clientLoginBannerOptions) {
            this.clientLoginBannerOptions = clientLoginBannerOptions != null ? clientLoginBannerOptions.build() : null;
        }

        @Override
        public final Builder clientLoginBannerOptions(ClientLoginBannerResponseOptions clientLoginBannerOptions) {
            this.clientLoginBannerOptions = clientLoginBannerOptions;
            return this;
        }

        public final Boolean getDisconnectOnSessionTimeout() {
            return disconnectOnSessionTimeout;
        }

        public final void setDisconnectOnSessionTimeout(Boolean disconnectOnSessionTimeout) {
            this.disconnectOnSessionTimeout = disconnectOnSessionTimeout;
        }

        @Override
        public final Builder disconnectOnSessionTimeout(Boolean disconnectOnSessionTimeout) {
            this.disconnectOnSessionTimeout = disconnectOnSessionTimeout;
            return this;
        }

        @Override
        public ClientVpnEndpoint build() {
            return new ClientVpnEndpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
