/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a route in the route server's routing database.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RouteServerRoute implements SdkPojo, Serializable,
        ToCopyableBuilder<RouteServerRoute.Builder, RouteServerRoute> {
    private static final SdkField<String> ROUTE_SERVER_ENDPOINT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RouteServerEndpointId")
            .getter(getter(RouteServerRoute::routeServerEndpointId))
            .setter(setter(Builder::routeServerEndpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RouteServerEndpointId")
                    .unmarshallLocationName("routeServerEndpointId").build()).build();

    private static final SdkField<String> ROUTE_SERVER_PEER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RouteServerPeerId")
            .getter(getter(RouteServerRoute::routeServerPeerId))
            .setter(setter(Builder::routeServerPeerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RouteServerPeerId")
                    .unmarshallLocationName("routeServerPeerId").build()).build();

    private static final SdkField<List<RouteServerRouteInstallationDetail>> ROUTE_INSTALLATION_DETAILS_FIELD = SdkField
            .<List<RouteServerRouteInstallationDetail>> builder(MarshallingType.LIST)
            .memberName("RouteInstallationDetails")
            .getter(getter(RouteServerRoute::routeInstallationDetails))
            .setter(setter(Builder::routeInstallationDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RouteInstallationDetailSet")
                    .unmarshallLocationName("routeInstallationDetailSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<RouteServerRouteInstallationDetail> builder(MarshallingType.SDK_POJO)
                                            .constructor(RouteServerRouteInstallationDetail::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> ROUTE_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RouteStatus")
            .getter(getter(RouteServerRoute::routeStatusAsString))
            .setter(setter(Builder::routeStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RouteStatus")
                    .unmarshallLocationName("routeStatus").build()).build();

    private static final SdkField<String> PREFIX_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Prefix")
            .getter(getter(RouteServerRoute::prefix))
            .setter(setter(Builder::prefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Prefix")
                    .unmarshallLocationName("prefix").build()).build();

    private static final SdkField<List<String>> AS_PATHS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AsPaths")
            .getter(getter(RouteServerRoute::asPaths))
            .setter(setter(Builder::asPaths))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AsPathSet")
                    .unmarshallLocationName("asPathSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Integer> MED_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("Med")
            .getter(getter(RouteServerRoute::med))
            .setter(setter(Builder::med))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Med").unmarshallLocationName("med")
                    .build()).build();

    private static final SdkField<String> NEXT_HOP_IP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NextHopIp")
            .getter(getter(RouteServerRoute::nextHopIp))
            .setter(setter(Builder::nextHopIp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextHopIp")
                    .unmarshallLocationName("nextHopIp").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ROUTE_SERVER_ENDPOINT_ID_FIELD, ROUTE_SERVER_PEER_ID_FIELD, ROUTE_INSTALLATION_DETAILS_FIELD, ROUTE_STATUS_FIELD,
            PREFIX_FIELD, AS_PATHS_FIELD, MED_FIELD, NEXT_HOP_IP_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String routeServerEndpointId;

    private final String routeServerPeerId;

    private final List<RouteServerRouteInstallationDetail> routeInstallationDetails;

    private final String routeStatus;

    private final String prefix;

    private final List<String> asPaths;

    private final Integer med;

    private final String nextHopIp;

    private RouteServerRoute(BuilderImpl builder) {
        this.routeServerEndpointId = builder.routeServerEndpointId;
        this.routeServerPeerId = builder.routeServerPeerId;
        this.routeInstallationDetails = builder.routeInstallationDetails;
        this.routeStatus = builder.routeStatus;
        this.prefix = builder.prefix;
        this.asPaths = builder.asPaths;
        this.med = builder.med;
        this.nextHopIp = builder.nextHopIp;
    }

    /**
     * <p>
     * The ID of the route server endpoint that received this route.
     * </p>
     * 
     * @return The ID of the route server endpoint that received this route.
     */
    public final String routeServerEndpointId() {
        return routeServerEndpointId;
    }

    /**
     * <p>
     * The ID of the route server peer that advertised this route.
     * </p>
     * 
     * @return The ID of the route server peer that advertised this route.
     */
    public final String routeServerPeerId() {
        return routeServerPeerId;
    }

    /**
     * For responses, this returns true if the service returned a value for the RouteInstallationDetails property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRouteInstallationDetails() {
        return routeInstallationDetails != null && !(routeInstallationDetails instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Details about the installation status of this route in route tables.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRouteInstallationDetails} method.
     * </p>
     * 
     * @return Details about the installation status of this route in route tables.
     */
    public final List<RouteServerRouteInstallationDetail> routeInstallationDetails() {
        return routeInstallationDetails;
    }

    /**
     * <p>
     * The current status of the route in the routing database. Values are <code>in-rib</code> or <code>in-fib</code>
     * depending on if the routes are in the RIB or the FIB database.
     * </p>
     * <p>
     * The <a href="https://en.wikipedia.org/wiki/Routing_table">Routing Information Base (RIB)</a> serves as a database
     * that stores all the routing information and network topology data collected by a router or routing system, such
     * as routes learned from BGP peers. The RIB is constantly updated as new routing information is received or
     * existing routes change. This ensures that the route server always has the most current view of the network
     * topology and can make optimal routing decisions.
     * </p>
     * <p>
     * The <a href="https://en.wikipedia.org/wiki/Forwarding_information_base">Forwarding Information Base (FIB)</a>
     * serves as a forwarding table for what route server has determined are the best-path routes in the RIB after
     * evaluating all available routing information and policies. The FIB routes are installed on the route tables. The
     * FIB is recomputed whenever there are changes to the RIB.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #routeStatus} will
     * return {@link RouteServerRouteStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #routeStatusAsString}.
     * </p>
     * 
     * @return The current status of the route in the routing database. Values are <code>in-rib</code> or
     *         <code>in-fib</code> depending on if the routes are in the RIB or the FIB database.</p>
     *         <p>
     *         The <a href="https://en.wikipedia.org/wiki/Routing_table">Routing Information Base (RIB)</a> serves as a
     *         database that stores all the routing information and network topology data collected by a router or
     *         routing system, such as routes learned from BGP peers. The RIB is constantly updated as new routing
     *         information is received or existing routes change. This ensures that the route server always has the most
     *         current view of the network topology and can make optimal routing decisions.
     *         </p>
     *         <p>
     *         The <a href="https://en.wikipedia.org/wiki/Forwarding_information_base">Forwarding Information Base
     *         (FIB)</a> serves as a forwarding table for what route server has determined are the best-path routes in
     *         the RIB after evaluating all available routing information and policies. The FIB routes are installed on
     *         the route tables. The FIB is recomputed whenever there are changes to the RIB.
     * @see RouteServerRouteStatus
     */
    public final RouteServerRouteStatus routeStatus() {
        return RouteServerRouteStatus.fromValue(routeStatus);
    }

    /**
     * <p>
     * The current status of the route in the routing database. Values are <code>in-rib</code> or <code>in-fib</code>
     * depending on if the routes are in the RIB or the FIB database.
     * </p>
     * <p>
     * The <a href="https://en.wikipedia.org/wiki/Routing_table">Routing Information Base (RIB)</a> serves as a database
     * that stores all the routing information and network topology data collected by a router or routing system, such
     * as routes learned from BGP peers. The RIB is constantly updated as new routing information is received or
     * existing routes change. This ensures that the route server always has the most current view of the network
     * topology and can make optimal routing decisions.
     * </p>
     * <p>
     * The <a href="https://en.wikipedia.org/wiki/Forwarding_information_base">Forwarding Information Base (FIB)</a>
     * serves as a forwarding table for what route server has determined are the best-path routes in the RIB after
     * evaluating all available routing information and policies. The FIB routes are installed on the route tables. The
     * FIB is recomputed whenever there are changes to the RIB.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #routeStatus} will
     * return {@link RouteServerRouteStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #routeStatusAsString}.
     * </p>
     * 
     * @return The current status of the route in the routing database. Values are <code>in-rib</code> or
     *         <code>in-fib</code> depending on if the routes are in the RIB or the FIB database.</p>
     *         <p>
     *         The <a href="https://en.wikipedia.org/wiki/Routing_table">Routing Information Base (RIB)</a> serves as a
     *         database that stores all the routing information and network topology data collected by a router or
     *         routing system, such as routes learned from BGP peers. The RIB is constantly updated as new routing
     *         information is received or existing routes change. This ensures that the route server always has the most
     *         current view of the network topology and can make optimal routing decisions.
     *         </p>
     *         <p>
     *         The <a href="https://en.wikipedia.org/wiki/Forwarding_information_base">Forwarding Information Base
     *         (FIB)</a> serves as a forwarding table for what route server has determined are the best-path routes in
     *         the RIB after evaluating all available routing information and policies. The FIB routes are installed on
     *         the route tables. The FIB is recomputed whenever there are changes to the RIB.
     * @see RouteServerRouteStatus
     */
    public final String routeStatusAsString() {
        return routeStatus;
    }

    /**
     * <p>
     * The destination CIDR block of the route.
     * </p>
     * 
     * @return The destination CIDR block of the route.
     */
    public final String prefix() {
        return prefix;
    }

    /**
     * For responses, this returns true if the service returned a value for the AsPaths property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAsPaths() {
        return asPaths != null && !(asPaths instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The AS path attributes of the BGP route.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAsPaths} method.
     * </p>
     * 
     * @return The AS path attributes of the BGP route.
     */
    public final List<String> asPaths() {
        return asPaths;
    }

    /**
     * <p>
     * The Multi-Exit Discriminator (MED) value of the BGP route.
     * </p>
     * 
     * @return The Multi-Exit Discriminator (MED) value of the BGP route.
     */
    public final Integer med() {
        return med;
    }

    /**
     * <p>
     * The IP address for the next hop.
     * </p>
     * 
     * @return The IP address for the next hop.
     */
    public final String nextHopIp() {
        return nextHopIp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(routeServerEndpointId());
        hashCode = 31 * hashCode + Objects.hashCode(routeServerPeerId());
        hashCode = 31 * hashCode + Objects.hashCode(hasRouteInstallationDetails() ? routeInstallationDetails() : null);
        hashCode = 31 * hashCode + Objects.hashCode(routeStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(prefix());
        hashCode = 31 * hashCode + Objects.hashCode(hasAsPaths() ? asPaths() : null);
        hashCode = 31 * hashCode + Objects.hashCode(med());
        hashCode = 31 * hashCode + Objects.hashCode(nextHopIp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RouteServerRoute)) {
            return false;
        }
        RouteServerRoute other = (RouteServerRoute) obj;
        return Objects.equals(routeServerEndpointId(), other.routeServerEndpointId())
                && Objects.equals(routeServerPeerId(), other.routeServerPeerId())
                && hasRouteInstallationDetails() == other.hasRouteInstallationDetails()
                && Objects.equals(routeInstallationDetails(), other.routeInstallationDetails())
                && Objects.equals(routeStatusAsString(), other.routeStatusAsString()) && Objects.equals(prefix(), other.prefix())
                && hasAsPaths() == other.hasAsPaths() && Objects.equals(asPaths(), other.asPaths())
                && Objects.equals(med(), other.med()) && Objects.equals(nextHopIp(), other.nextHopIp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RouteServerRoute").add("RouteServerEndpointId", routeServerEndpointId())
                .add("RouteServerPeerId", routeServerPeerId())
                .add("RouteInstallationDetails", hasRouteInstallationDetails() ? routeInstallationDetails() : null)
                .add("RouteStatus", routeStatusAsString()).add("Prefix", prefix())
                .add("AsPaths", hasAsPaths() ? asPaths() : null).add("Med", med()).add("NextHopIp", nextHopIp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RouteServerEndpointId":
            return Optional.ofNullable(clazz.cast(routeServerEndpointId()));
        case "RouteServerPeerId":
            return Optional.ofNullable(clazz.cast(routeServerPeerId()));
        case "RouteInstallationDetails":
            return Optional.ofNullable(clazz.cast(routeInstallationDetails()));
        case "RouteStatus":
            return Optional.ofNullable(clazz.cast(routeStatusAsString()));
        case "Prefix":
            return Optional.ofNullable(clazz.cast(prefix()));
        case "AsPaths":
            return Optional.ofNullable(clazz.cast(asPaths()));
        case "Med":
            return Optional.ofNullable(clazz.cast(med()));
        case "NextHopIp":
            return Optional.ofNullable(clazz.cast(nextHopIp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RouteServerEndpointId", ROUTE_SERVER_ENDPOINT_ID_FIELD);
        map.put("RouteServerPeerId", ROUTE_SERVER_PEER_ID_FIELD);
        map.put("RouteInstallationDetailSet", ROUTE_INSTALLATION_DETAILS_FIELD);
        map.put("RouteStatus", ROUTE_STATUS_FIELD);
        map.put("Prefix", PREFIX_FIELD);
        map.put("AsPathSet", AS_PATHS_FIELD);
        map.put("Med", MED_FIELD);
        map.put("NextHopIp", NEXT_HOP_IP_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RouteServerRoute, T> g) {
        return obj -> g.apply((RouteServerRoute) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RouteServerRoute> {
        /**
         * <p>
         * The ID of the route server endpoint that received this route.
         * </p>
         * 
         * @param routeServerEndpointId
         *        The ID of the route server endpoint that received this route.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeServerEndpointId(String routeServerEndpointId);

        /**
         * <p>
         * The ID of the route server peer that advertised this route.
         * </p>
         * 
         * @param routeServerPeerId
         *        The ID of the route server peer that advertised this route.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeServerPeerId(String routeServerPeerId);

        /**
         * <p>
         * Details about the installation status of this route in route tables.
         * </p>
         * 
         * @param routeInstallationDetails
         *        Details about the installation status of this route in route tables.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeInstallationDetails(Collection<RouteServerRouteInstallationDetail> routeInstallationDetails);

        /**
         * <p>
         * Details about the installation status of this route in route tables.
         * </p>
         * 
         * @param routeInstallationDetails
         *        Details about the installation status of this route in route tables.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeInstallationDetails(RouteServerRouteInstallationDetail... routeInstallationDetails);

        /**
         * <p>
         * Details about the installation status of this route in route tables.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.RouteServerRouteInstallationDetail.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.RouteServerRouteInstallationDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.RouteServerRouteInstallationDetail.Builder#build()} is
         * called immediately and its result is passed to {@link
         * #routeInstallationDetails(List<RouteServerRouteInstallationDetail>)}.
         * 
         * @param routeInstallationDetails
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.RouteServerRouteInstallationDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #routeInstallationDetails(java.util.Collection<RouteServerRouteInstallationDetail>)
         */
        Builder routeInstallationDetails(Consumer<RouteServerRouteInstallationDetail.Builder>... routeInstallationDetails);

        /**
         * <p>
         * The current status of the route in the routing database. Values are <code>in-rib</code> or
         * <code>in-fib</code> depending on if the routes are in the RIB or the FIB database.
         * </p>
         * <p>
         * The <a href="https://en.wikipedia.org/wiki/Routing_table">Routing Information Base (RIB)</a> serves as a
         * database that stores all the routing information and network topology data collected by a router or routing
         * system, such as routes learned from BGP peers. The RIB is constantly updated as new routing information is
         * received or existing routes change. This ensures that the route server always has the most current view of
         * the network topology and can make optimal routing decisions.
         * </p>
         * <p>
         * The <a href="https://en.wikipedia.org/wiki/Forwarding_information_base">Forwarding Information Base (FIB)</a>
         * serves as a forwarding table for what route server has determined are the best-path routes in the RIB after
         * evaluating all available routing information and policies. The FIB routes are installed on the route tables.
         * The FIB is recomputed whenever there are changes to the RIB.
         * </p>
         * 
         * @param routeStatus
         *        The current status of the route in the routing database. Values are <code>in-rib</code> or
         *        <code>in-fib</code> depending on if the routes are in the RIB or the FIB database.</p>
         *        <p>
         *        The <a href="https://en.wikipedia.org/wiki/Routing_table">Routing Information Base (RIB)</a> serves as
         *        a database that stores all the routing information and network topology data collected by a router or
         *        routing system, such as routes learned from BGP peers. The RIB is constantly updated as new routing
         *        information is received or existing routes change. This ensures that the route server always has the
         *        most current view of the network topology and can make optimal routing decisions.
         *        </p>
         *        <p>
         *        The <a href="https://en.wikipedia.org/wiki/Forwarding_information_base">Forwarding Information Base
         *        (FIB)</a> serves as a forwarding table for what route server has determined are the best-path routes
         *        in the RIB after evaluating all available routing information and policies. The FIB routes are
         *        installed on the route tables. The FIB is recomputed whenever there are changes to the RIB.
         * @see RouteServerRouteStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteServerRouteStatus
         */
        Builder routeStatus(String routeStatus);

        /**
         * <p>
         * The current status of the route in the routing database. Values are <code>in-rib</code> or
         * <code>in-fib</code> depending on if the routes are in the RIB or the FIB database.
         * </p>
         * <p>
         * The <a href="https://en.wikipedia.org/wiki/Routing_table">Routing Information Base (RIB)</a> serves as a
         * database that stores all the routing information and network topology data collected by a router or routing
         * system, such as routes learned from BGP peers. The RIB is constantly updated as new routing information is
         * received or existing routes change. This ensures that the route server always has the most current view of
         * the network topology and can make optimal routing decisions.
         * </p>
         * <p>
         * The <a href="https://en.wikipedia.org/wiki/Forwarding_information_base">Forwarding Information Base (FIB)</a>
         * serves as a forwarding table for what route server has determined are the best-path routes in the RIB after
         * evaluating all available routing information and policies. The FIB routes are installed on the route tables.
         * The FIB is recomputed whenever there are changes to the RIB.
         * </p>
         * 
         * @param routeStatus
         *        The current status of the route in the routing database. Values are <code>in-rib</code> or
         *        <code>in-fib</code> depending on if the routes are in the RIB or the FIB database.</p>
         *        <p>
         *        The <a href="https://en.wikipedia.org/wiki/Routing_table">Routing Information Base (RIB)</a> serves as
         *        a database that stores all the routing information and network topology data collected by a router or
         *        routing system, such as routes learned from BGP peers. The RIB is constantly updated as new routing
         *        information is received or existing routes change. This ensures that the route server always has the
         *        most current view of the network topology and can make optimal routing decisions.
         *        </p>
         *        <p>
         *        The <a href="https://en.wikipedia.org/wiki/Forwarding_information_base">Forwarding Information Base
         *        (FIB)</a> serves as a forwarding table for what route server has determined are the best-path routes
         *        in the RIB after evaluating all available routing information and policies. The FIB routes are
         *        installed on the route tables. The FIB is recomputed whenever there are changes to the RIB.
         * @see RouteServerRouteStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteServerRouteStatus
         */
        Builder routeStatus(RouteServerRouteStatus routeStatus);

        /**
         * <p>
         * The destination CIDR block of the route.
         * </p>
         * 
         * @param prefix
         *        The destination CIDR block of the route.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder prefix(String prefix);

        /**
         * <p>
         * The AS path attributes of the BGP route.
         * </p>
         * 
         * @param asPaths
         *        The AS path attributes of the BGP route.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder asPaths(Collection<String> asPaths);

        /**
         * <p>
         * The AS path attributes of the BGP route.
         * </p>
         * 
         * @param asPaths
         *        The AS path attributes of the BGP route.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder asPaths(String... asPaths);

        /**
         * <p>
         * The Multi-Exit Discriminator (MED) value of the BGP route.
         * </p>
         * 
         * @param med
         *        The Multi-Exit Discriminator (MED) value of the BGP route.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder med(Integer med);

        /**
         * <p>
         * The IP address for the next hop.
         * </p>
         * 
         * @param nextHopIp
         *        The IP address for the next hop.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextHopIp(String nextHopIp);
    }

    static final class BuilderImpl implements Builder {
        private String routeServerEndpointId;

        private String routeServerPeerId;

        private List<RouteServerRouteInstallationDetail> routeInstallationDetails = DefaultSdkAutoConstructList.getInstance();

        private String routeStatus;

        private String prefix;

        private List<String> asPaths = DefaultSdkAutoConstructList.getInstance();

        private Integer med;

        private String nextHopIp;

        private BuilderImpl() {
        }

        private BuilderImpl(RouteServerRoute model) {
            routeServerEndpointId(model.routeServerEndpointId);
            routeServerPeerId(model.routeServerPeerId);
            routeInstallationDetails(model.routeInstallationDetails);
            routeStatus(model.routeStatus);
            prefix(model.prefix);
            asPaths(model.asPaths);
            med(model.med);
            nextHopIp(model.nextHopIp);
        }

        public final String getRouteServerEndpointId() {
            return routeServerEndpointId;
        }

        public final void setRouteServerEndpointId(String routeServerEndpointId) {
            this.routeServerEndpointId = routeServerEndpointId;
        }

        @Override
        public final Builder routeServerEndpointId(String routeServerEndpointId) {
            this.routeServerEndpointId = routeServerEndpointId;
            return this;
        }

        public final String getRouteServerPeerId() {
            return routeServerPeerId;
        }

        public final void setRouteServerPeerId(String routeServerPeerId) {
            this.routeServerPeerId = routeServerPeerId;
        }

        @Override
        public final Builder routeServerPeerId(String routeServerPeerId) {
            this.routeServerPeerId = routeServerPeerId;
            return this;
        }

        public final List<RouteServerRouteInstallationDetail.Builder> getRouteInstallationDetails() {
            List<RouteServerRouteInstallationDetail.Builder> result = RouteServerRouteInstallationDetailsCopier
                    .copyToBuilder(this.routeInstallationDetails);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRouteInstallationDetails(
                Collection<RouteServerRouteInstallationDetail.BuilderImpl> routeInstallationDetails) {
            this.routeInstallationDetails = RouteServerRouteInstallationDetailsCopier.copyFromBuilder(routeInstallationDetails);
        }

        @Override
        public final Builder routeInstallationDetails(Collection<RouteServerRouteInstallationDetail> routeInstallationDetails) {
            this.routeInstallationDetails = RouteServerRouteInstallationDetailsCopier.copy(routeInstallationDetails);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder routeInstallationDetails(RouteServerRouteInstallationDetail... routeInstallationDetails) {
            routeInstallationDetails(Arrays.asList(routeInstallationDetails));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder routeInstallationDetails(
                Consumer<RouteServerRouteInstallationDetail.Builder>... routeInstallationDetails) {
            routeInstallationDetails(Stream.of(routeInstallationDetails)
                    .map(c -> RouteServerRouteInstallationDetail.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getRouteStatus() {
            return routeStatus;
        }

        public final void setRouteStatus(String routeStatus) {
            this.routeStatus = routeStatus;
        }

        @Override
        public final Builder routeStatus(String routeStatus) {
            this.routeStatus = routeStatus;
            return this;
        }

        @Override
        public final Builder routeStatus(RouteServerRouteStatus routeStatus) {
            this.routeStatus(routeStatus == null ? null : routeStatus.toString());
            return this;
        }

        public final String getPrefix() {
            return prefix;
        }

        public final void setPrefix(String prefix) {
            this.prefix = prefix;
        }

        @Override
        public final Builder prefix(String prefix) {
            this.prefix = prefix;
            return this;
        }

        public final Collection<String> getAsPaths() {
            if (asPaths instanceof SdkAutoConstructList) {
                return null;
            }
            return asPaths;
        }

        public final void setAsPaths(Collection<String> asPaths) {
            this.asPaths = AsPathCopier.copy(asPaths);
        }

        @Override
        public final Builder asPaths(Collection<String> asPaths) {
            this.asPaths = AsPathCopier.copy(asPaths);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder asPaths(String... asPaths) {
            asPaths(Arrays.asList(asPaths));
            return this;
        }

        public final Integer getMed() {
            return med;
        }

        public final void setMed(Integer med) {
            this.med = med;
        }

        @Override
        public final Builder med(Integer med) {
            this.med = med;
            return this;
        }

        public final String getNextHopIp() {
            return nextHopIp;
        }

        public final void setNextHopIp(String nextHopIp) {
            this.nextHopIp = nextHopIp;
        }

        @Override
        public final Builder nextHopIp(String nextHopIp) {
            this.nextHopIp = nextHopIp;
            return this;
        }

        @Override
        public RouteServerRoute build() {
            return new RouteServerRoute(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
