/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateIpamPoolRequest extends Ec2Request implements
        ToCopyableBuilder<CreateIpamPoolRequest.Builder, CreateIpamPoolRequest> {
    private static final SdkField<Boolean> DRY_RUN_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DryRun")
            .getter(getter(CreateIpamPoolRequest::dryRun))
            .setter(setter(Builder::dryRun))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DryRun")
                    .unmarshallLocationName("DryRun").build()).build();

    private static final SdkField<String> IPAM_SCOPE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamScopeId")
            .getter(getter(CreateIpamPoolRequest::ipamScopeId))
            .setter(setter(Builder::ipamScopeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamScopeId")
                    .unmarshallLocationName("IpamScopeId").build()).build();

    private static final SdkField<String> LOCALE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Locale")
            .getter(getter(CreateIpamPoolRequest::locale))
            .setter(setter(Builder::locale))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Locale")
                    .unmarshallLocationName("Locale").build()).build();

    private static final SdkField<String> SOURCE_IPAM_POOL_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SourceIpamPoolId")
            .getter(getter(CreateIpamPoolRequest::sourceIpamPoolId))
            .setter(setter(Builder::sourceIpamPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceIpamPoolId")
                    .unmarshallLocationName("SourceIpamPoolId").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(CreateIpamPoolRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("Description").build()).build();

    private static final SdkField<String> ADDRESS_FAMILY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AddressFamily")
            .getter(getter(CreateIpamPoolRequest::addressFamilyAsString))
            .setter(setter(Builder::addressFamily))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AddressFamily")
                    .unmarshallLocationName("AddressFamily").build()).build();

    private static final SdkField<Boolean> AUTO_IMPORT_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AutoImport")
            .getter(getter(CreateIpamPoolRequest::autoImport))
            .setter(setter(Builder::autoImport))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoImport")
                    .unmarshallLocationName("AutoImport").build()).build();

    private static final SdkField<Boolean> PUBLICLY_ADVERTISABLE_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("PubliclyAdvertisable")
            .getter(getter(CreateIpamPoolRequest::publiclyAdvertisable))
            .setter(setter(Builder::publiclyAdvertisable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PubliclyAdvertisable")
                    .unmarshallLocationName("PubliclyAdvertisable").build()).build();

    private static final SdkField<Integer> ALLOCATION_MIN_NETMASK_LENGTH_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("AllocationMinNetmaskLength")
            .getter(getter(CreateIpamPoolRequest::allocationMinNetmaskLength))
            .setter(setter(Builder::allocationMinNetmaskLength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationMinNetmaskLength")
                    .unmarshallLocationName("AllocationMinNetmaskLength").build()).build();

    private static final SdkField<Integer> ALLOCATION_MAX_NETMASK_LENGTH_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("AllocationMaxNetmaskLength")
            .getter(getter(CreateIpamPoolRequest::allocationMaxNetmaskLength))
            .setter(setter(Builder::allocationMaxNetmaskLength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationMaxNetmaskLength")
                    .unmarshallLocationName("AllocationMaxNetmaskLength").build()).build();

    private static final SdkField<Integer> ALLOCATION_DEFAULT_NETMASK_LENGTH_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("AllocationDefaultNetmaskLength")
            .getter(getter(CreateIpamPoolRequest::allocationDefaultNetmaskLength))
            .setter(setter(Builder::allocationDefaultNetmaskLength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationDefaultNetmaskLength")
                    .unmarshallLocationName("AllocationDefaultNetmaskLength").build()).build();

    private static final SdkField<List<RequestIpamResourceTag>> ALLOCATION_RESOURCE_TAGS_FIELD = SdkField
            .<List<RequestIpamResourceTag>> builder(MarshallingType.LIST)
            .memberName("AllocationResourceTags")
            .getter(getter(CreateIpamPoolRequest::allocationResourceTags))
            .setter(setter(Builder::allocationResourceTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationResourceTag")
                    .unmarshallLocationName("AllocationResourceTag").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<RequestIpamResourceTag> builder(MarshallingType.SDK_POJO)
                                            .constructor(RequestIpamResourceTag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<TagSpecification>> TAG_SPECIFICATIONS_FIELD = SdkField
            .<List<TagSpecification>> builder(MarshallingType.LIST)
            .memberName("TagSpecifications")
            .getter(getter(CreateIpamPoolRequest::tagSpecifications))
            .setter(setter(Builder::tagSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSpecification")
                    .unmarshallLocationName("TagSpecification").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<TagSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientToken")
            .getter(getter(CreateIpamPoolRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientToken")
                    .unmarshallLocationName("ClientToken").build(), DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<String> AWS_SERVICE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AwsService")
            .getter(getter(CreateIpamPoolRequest::awsServiceAsString))
            .setter(setter(Builder::awsService))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AwsService")
                    .unmarshallLocationName("AwsService").build()).build();

    private static final SdkField<String> PUBLIC_IP_SOURCE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PublicIpSource")
            .getter(getter(CreateIpamPoolRequest::publicIpSourceAsString))
            .setter(setter(Builder::publicIpSource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublicIpSource")
                    .unmarshallLocationName("PublicIpSource").build()).build();

    private static final SdkField<IpamPoolSourceResourceRequest> SOURCE_RESOURCE_FIELD = SdkField
            .<IpamPoolSourceResourceRequest> builder(MarshallingType.SDK_POJO)
            .memberName("SourceResource")
            .getter(getter(CreateIpamPoolRequest::sourceResource))
            .setter(setter(Builder::sourceResource))
            .constructor(IpamPoolSourceResourceRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceResource")
                    .unmarshallLocationName("SourceResource").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DRY_RUN_FIELD,
            IPAM_SCOPE_ID_FIELD, LOCALE_FIELD, SOURCE_IPAM_POOL_ID_FIELD, DESCRIPTION_FIELD, ADDRESS_FAMILY_FIELD,
            AUTO_IMPORT_FIELD, PUBLICLY_ADVERTISABLE_FIELD, ALLOCATION_MIN_NETMASK_LENGTH_FIELD,
            ALLOCATION_MAX_NETMASK_LENGTH_FIELD, ALLOCATION_DEFAULT_NETMASK_LENGTH_FIELD, ALLOCATION_RESOURCE_TAGS_FIELD,
            TAG_SPECIFICATIONS_FIELD, CLIENT_TOKEN_FIELD, AWS_SERVICE_FIELD, PUBLIC_IP_SOURCE_FIELD, SOURCE_RESOURCE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final Boolean dryRun;

    private final String ipamScopeId;

    private final String locale;

    private final String sourceIpamPoolId;

    private final String description;

    private final String addressFamily;

    private final Boolean autoImport;

    private final Boolean publiclyAdvertisable;

    private final Integer allocationMinNetmaskLength;

    private final Integer allocationMaxNetmaskLength;

    private final Integer allocationDefaultNetmaskLength;

    private final List<RequestIpamResourceTag> allocationResourceTags;

    private final List<TagSpecification> tagSpecifications;

    private final String clientToken;

    private final String awsService;

    private final String publicIpSource;

    private final IpamPoolSourceResourceRequest sourceResource;

    private CreateIpamPoolRequest(BuilderImpl builder) {
        super(builder);
        this.dryRun = builder.dryRun;
        this.ipamScopeId = builder.ipamScopeId;
        this.locale = builder.locale;
        this.sourceIpamPoolId = builder.sourceIpamPoolId;
        this.description = builder.description;
        this.addressFamily = builder.addressFamily;
        this.autoImport = builder.autoImport;
        this.publiclyAdvertisable = builder.publiclyAdvertisable;
        this.allocationMinNetmaskLength = builder.allocationMinNetmaskLength;
        this.allocationMaxNetmaskLength = builder.allocationMaxNetmaskLength;
        this.allocationDefaultNetmaskLength = builder.allocationDefaultNetmaskLength;
        this.allocationResourceTags = builder.allocationResourceTags;
        this.tagSpecifications = builder.tagSpecifications;
        this.clientToken = builder.clientToken;
        this.awsService = builder.awsService;
        this.publicIpSource = builder.publicIpSource;
        this.sourceResource = builder.sourceResource;
    }

    /**
     * <p>
     * A check for whether you have the required permissions for the action without actually making the request and
     * provides an error response. If you have the required permissions, the error response is
     * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     * </p>
     * 
     * @return A check for whether you have the required permissions for the action without actually making the request
     *         and provides an error response. If you have the required permissions, the error response is
     *         <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     */
    public final Boolean dryRun() {
        return dryRun;
    }

    /**
     * <p>
     * The ID of the scope in which you would like to create the IPAM pool.
     * </p>
     * 
     * @return The ID of the scope in which you would like to create the IPAM pool.
     */
    public final String ipamScopeId() {
        return ipamScopeId;
    }

    /**
     * <p>
     * The locale for the pool should be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An Amazon Web Services Region where you want this IPAM pool to be available for allocations.
     * </p>
     * </li>
     * <li>
     * <p>
     * The network border group for an Amazon Web Services Local Zone where you want this IPAM pool to be available for
     * allocations (<a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html#byoip-zone-avail">supported Local
     * Zones</a>). This option is only available for IPAM IPv4 pools in the public scope.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Possible values: Any Amazon Web Services Region or supported Amazon Web Services Local Zone. Default is
     * <code>none</code> and means any locale.
     * </p>
     * 
     * @return The locale for the pool should be one of the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         An Amazon Web Services Region where you want this IPAM pool to be available for allocations.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The network border group for an Amazon Web Services Local Zone where you want this IPAM pool to be
     *         available for allocations (<a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html#byoip-zone-avail">supported
     *         Local Zones</a>). This option is only available for IPAM IPv4 pools in the public scope.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Possible values: Any Amazon Web Services Region or supported Amazon Web Services Local Zone. Default is
     *         <code>none</code> and means any locale.
     */
    public final String locale() {
        return locale;
    }

    /**
     * <p>
     * The ID of the source IPAM pool. Use this option to create a pool within an existing pool. Note that the CIDR you
     * provision for the pool within the source pool must be available in the source pool's CIDR range.
     * </p>
     * 
     * @return The ID of the source IPAM pool. Use this option to create a pool within an existing pool. Note that the
     *         CIDR you provision for the pool within the source pool must be available in the source pool's CIDR range.
     */
    public final String sourceIpamPoolId() {
        return sourceIpamPoolId;
    }

    /**
     * <p>
     * A description for the IPAM pool.
     * </p>
     * 
     * @return A description for the IPAM pool.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The IP protocol assigned to this IPAM pool. You must choose either IPv4 or IPv6 protocol for a pool.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #addressFamily}
     * will return {@link AddressFamily#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #addressFamilyAsString}.
     * </p>
     * 
     * @return The IP protocol assigned to this IPAM pool. You must choose either IPv4 or IPv6 protocol for a pool.
     * @see AddressFamily
     */
    public final AddressFamily addressFamily() {
        return AddressFamily.fromValue(addressFamily);
    }

    /**
     * <p>
     * The IP protocol assigned to this IPAM pool. You must choose either IPv4 or IPv6 protocol for a pool.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #addressFamily}
     * will return {@link AddressFamily#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #addressFamilyAsString}.
     * </p>
     * 
     * @return The IP protocol assigned to this IPAM pool. You must choose either IPv4 or IPv6 protocol for a pool.
     * @see AddressFamily
     */
    public final String addressFamilyAsString() {
        return addressFamily;
    }

    /**
     * <p>
     * If selected, IPAM will continuously look for resources within the CIDR range of this pool and automatically
     * import them as allocations into your IPAM. The CIDRs that will be allocated for these resources must not already
     * be allocated to other resources in order for the import to succeed. IPAM will import a CIDR regardless of its
     * compliance with the pool's allocation rules, so a resource might be imported and subsequently marked as
     * noncompliant. If IPAM discovers multiple CIDRs that overlap, IPAM will import the largest CIDR only. If IPAM
     * discovers multiple CIDRs with matching CIDRs, IPAM will randomly import one of them only.
     * </p>
     * <p>
     * A locale must be set on the pool for this feature to work.
     * </p>
     * 
     * @return If selected, IPAM will continuously look for resources within the CIDR range of this pool and
     *         automatically import them as allocations into your IPAM. The CIDRs that will be allocated for these
     *         resources must not already be allocated to other resources in order for the import to succeed. IPAM will
     *         import a CIDR regardless of its compliance with the pool's allocation rules, so a resource might be
     *         imported and subsequently marked as noncompliant. If IPAM discovers multiple CIDRs that overlap, IPAM
     *         will import the largest CIDR only. If IPAM discovers multiple CIDRs with matching CIDRs, IPAM will
     *         randomly import one of them only. </p>
     *         <p>
     *         A locale must be set on the pool for this feature to work.
     */
    public final Boolean autoImport() {
        return autoImport;
    }

    /**
     * <p>
     * Determines if the pool is publicly advertisable. The request can only contain <code>PubliclyAdvertisable</code>
     * if <code>AddressFamily</code> is <code>ipv6</code> and <code>PublicIpSource</code> is <code>byoip</code>.
     * </p>
     * 
     * @return Determines if the pool is publicly advertisable. The request can only contain
     *         <code>PubliclyAdvertisable</code> if <code>AddressFamily</code> is <code>ipv6</code> and
     *         <code>PublicIpSource</code> is <code>byoip</code>.
     */
    public final Boolean publiclyAdvertisable() {
        return publiclyAdvertisable;
    }

    /**
     * <p>
     * The minimum netmask length required for CIDR allocations in this IPAM pool to be compliant. The minimum netmask
     * length must be less than the maximum netmask length. Possible netmask lengths for IPv4 addresses are 0 - 32.
     * Possible netmask lengths for IPv6 addresses are 0 - 128.
     * </p>
     * 
     * @return The minimum netmask length required for CIDR allocations in this IPAM pool to be compliant. The minimum
     *         netmask length must be less than the maximum netmask length. Possible netmask lengths for IPv4 addresses
     *         are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
     */
    public final Integer allocationMinNetmaskLength() {
        return allocationMinNetmaskLength;
    }

    /**
     * <p>
     * The maximum netmask length possible for CIDR allocations in this IPAM pool to be compliant. The maximum netmask
     * length must be greater than the minimum netmask length. Possible netmask lengths for IPv4 addresses are 0 - 32.
     * Possible netmask lengths for IPv6 addresses are 0 - 128.
     * </p>
     * 
     * @return The maximum netmask length possible for CIDR allocations in this IPAM pool to be compliant. The maximum
     *         netmask length must be greater than the minimum netmask length. Possible netmask lengths for IPv4
     *         addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
     */
    public final Integer allocationMaxNetmaskLength() {
        return allocationMaxNetmaskLength;
    }

    /**
     * <p>
     * The default netmask length for allocations added to this pool. If, for example, the CIDR assigned to this pool is
     * 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16.
     * </p>
     * 
     * @return The default netmask length for allocations added to this pool. If, for example, the CIDR assigned to this
     *         pool is 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16.
     */
    public final Integer allocationDefaultNetmaskLength() {
        return allocationDefaultNetmaskLength;
    }

    /**
     * For responses, this returns true if the service returned a value for the AllocationResourceTags property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAllocationResourceTags() {
        return allocationResourceTags != null && !(allocationResourceTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these tags
     * will not be allowed to allocate space from the pool. If the resources have their tags changed after they have
     * allocated space or if the allocation tagging requirements are changed on the pool, the resource may be marked as
     * noncompliant.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAllocationResourceTags} method.
     * </p>
     * 
     * @return Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these
     *         tags will not be allowed to allocate space from the pool. If the resources have their tags changed after
     *         they have allocated space or if the allocation tagging requirements are changed on the pool, the resource
     *         may be marked as noncompliant.
     */
    public final List<RequestIpamResourceTag> allocationResourceTags() {
        return allocationResourceTags;
    }

    /**
     * For responses, this returns true if the service returned a value for the TagSpecifications property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTagSpecifications() {
        return tagSpecifications != null && !(tagSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag value
     * as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code> and the
     * value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code> for the
     * filter value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTagSpecifications} method.
     * </p>
     * 
     * @return The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the
     *         tag value as the filter value. For example, to find all resources that have a tag with the key
     *         <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name
     *         and <code>TeamA</code> for the filter value.
     */
    public final List<TagSpecification> tagSpecifications() {
        return tagSpecifications;
    }

    /**
     * <p>
     * A unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
     * information, see <a href="https://docs.aws.amazon.com/ec2/latest/devguide/ec2-api-idempotency.html">Ensuring
     * idempotency</a>.
     * </p>
     * 
     * @return A unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/ec2/latest/devguide/ec2-api-idempotency.html">Ensuring idempotency</a>.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * <p>
     * Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows users to use
     * space for Elastic IP addresses and VPCs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #awsService} will
     * return {@link IpamPoolAwsService#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #awsServiceAsString}.
     * </p>
     * 
     * @return Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows
     *         users to use space for Elastic IP addresses and VPCs.
     * @see IpamPoolAwsService
     */
    public final IpamPoolAwsService awsService() {
        return IpamPoolAwsService.fromValue(awsService);
    }

    /**
     * <p>
     * Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows users to use
     * space for Elastic IP addresses and VPCs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #awsService} will
     * return {@link IpamPoolAwsService#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #awsServiceAsString}.
     * </p>
     * 
     * @return Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows
     *         users to use space for Elastic IP addresses and VPCs.
     * @see IpamPoolAwsService
     */
    public final String awsServiceAsString() {
        return awsService;
    }

    /**
     * <p>
     * The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools in the
     * public scope. Default is <code>byoip</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a> in the
     * <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR block to a
     * top-level IPv6 pool if PublicIpSource is <code>amazon</code>. For information on increasing the default limit,
     * see <a href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html"> Quotas for your IPAM</a> in the
     * <i>Amazon VPC IPAM User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #publicIpSource}
     * will return {@link IpamPoolPublicIpSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #publicIpSourceAsString}.
     * </p>
     * 
     * @return The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools
     *         in the public scope. Default is <code>byoip</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a> in
     *         the <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR block
     *         to a top-level IPv6 pool if PublicIpSource is <code>amazon</code>. For information on increasing the
     *         default limit, see <a href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html"> Quotas for
     *         your IPAM</a> in the <i>Amazon VPC IPAM User Guide</i>.
     * @see IpamPoolPublicIpSource
     */
    public final IpamPoolPublicIpSource publicIpSource() {
        return IpamPoolPublicIpSource.fromValue(publicIpSource);
    }

    /**
     * <p>
     * The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools in the
     * public scope. Default is <code>byoip</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a> in the
     * <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR block to a
     * top-level IPv6 pool if PublicIpSource is <code>amazon</code>. For information on increasing the default limit,
     * see <a href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html"> Quotas for your IPAM</a> in the
     * <i>Amazon VPC IPAM User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #publicIpSource}
     * will return {@link IpamPoolPublicIpSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #publicIpSourceAsString}.
     * </p>
     * 
     * @return The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools
     *         in the public scope. Default is <code>byoip</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a> in
     *         the <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR block
     *         to a top-level IPv6 pool if PublicIpSource is <code>amazon</code>. For information on increasing the
     *         default limit, see <a href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html"> Quotas for
     *         your IPAM</a> in the <i>Amazon VPC IPAM User Guide</i>.
     * @see IpamPoolPublicIpSource
     */
    public final String publicIpSourceAsString() {
        return publicIpSource;
    }

    /**
     * <p>
     * The resource used to provision CIDRs to a resource planning pool.
     * </p>
     * 
     * @return The resource used to provision CIDRs to a resource planning pool.
     */
    public final IpamPoolSourceResourceRequest sourceResource() {
        return sourceResource;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(dryRun());
        hashCode = 31 * hashCode + Objects.hashCode(ipamScopeId());
        hashCode = 31 * hashCode + Objects.hashCode(locale());
        hashCode = 31 * hashCode + Objects.hashCode(sourceIpamPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(addressFamilyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(autoImport());
        hashCode = 31 * hashCode + Objects.hashCode(publiclyAdvertisable());
        hashCode = 31 * hashCode + Objects.hashCode(allocationMinNetmaskLength());
        hashCode = 31 * hashCode + Objects.hashCode(allocationMaxNetmaskLength());
        hashCode = 31 * hashCode + Objects.hashCode(allocationDefaultNetmaskLength());
        hashCode = 31 * hashCode + Objects.hashCode(hasAllocationResourceTags() ? allocationResourceTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTagSpecifications() ? tagSpecifications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(awsServiceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(publicIpSourceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(sourceResource());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateIpamPoolRequest)) {
            return false;
        }
        CreateIpamPoolRequest other = (CreateIpamPoolRequest) obj;
        return Objects.equals(dryRun(), other.dryRun()) && Objects.equals(ipamScopeId(), other.ipamScopeId())
                && Objects.equals(locale(), other.locale()) && Objects.equals(sourceIpamPoolId(), other.sourceIpamPoolId())
                && Objects.equals(description(), other.description())
                && Objects.equals(addressFamilyAsString(), other.addressFamilyAsString())
                && Objects.equals(autoImport(), other.autoImport())
                && Objects.equals(publiclyAdvertisable(), other.publiclyAdvertisable())
                && Objects.equals(allocationMinNetmaskLength(), other.allocationMinNetmaskLength())
                && Objects.equals(allocationMaxNetmaskLength(), other.allocationMaxNetmaskLength())
                && Objects.equals(allocationDefaultNetmaskLength(), other.allocationDefaultNetmaskLength())
                && hasAllocationResourceTags() == other.hasAllocationResourceTags()
                && Objects.equals(allocationResourceTags(), other.allocationResourceTags())
                && hasTagSpecifications() == other.hasTagSpecifications()
                && Objects.equals(tagSpecifications(), other.tagSpecifications())
                && Objects.equals(clientToken(), other.clientToken())
                && Objects.equals(awsServiceAsString(), other.awsServiceAsString())
                && Objects.equals(publicIpSourceAsString(), other.publicIpSourceAsString())
                && Objects.equals(sourceResource(), other.sourceResource());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateIpamPoolRequest").add("DryRun", dryRun()).add("IpamScopeId", ipamScopeId())
                .add("Locale", locale()).add("SourceIpamPoolId", sourceIpamPoolId()).add("Description", description())
                .add("AddressFamily", addressFamilyAsString()).add("AutoImport", autoImport())
                .add("PubliclyAdvertisable", publiclyAdvertisable())
                .add("AllocationMinNetmaskLength", allocationMinNetmaskLength())
                .add("AllocationMaxNetmaskLength", allocationMaxNetmaskLength())
                .add("AllocationDefaultNetmaskLength", allocationDefaultNetmaskLength())
                .add("AllocationResourceTags", hasAllocationResourceTags() ? allocationResourceTags() : null)
                .add("TagSpecifications", hasTagSpecifications() ? tagSpecifications() : null).add("ClientToken", clientToken())
                .add("AwsService", awsServiceAsString()).add("PublicIpSource", publicIpSourceAsString())
                .add("SourceResource", sourceResource()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DryRun":
            return Optional.ofNullable(clazz.cast(dryRun()));
        case "IpamScopeId":
            return Optional.ofNullable(clazz.cast(ipamScopeId()));
        case "Locale":
            return Optional.ofNullable(clazz.cast(locale()));
        case "SourceIpamPoolId":
            return Optional.ofNullable(clazz.cast(sourceIpamPoolId()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "AddressFamily":
            return Optional.ofNullable(clazz.cast(addressFamilyAsString()));
        case "AutoImport":
            return Optional.ofNullable(clazz.cast(autoImport()));
        case "PubliclyAdvertisable":
            return Optional.ofNullable(clazz.cast(publiclyAdvertisable()));
        case "AllocationMinNetmaskLength":
            return Optional.ofNullable(clazz.cast(allocationMinNetmaskLength()));
        case "AllocationMaxNetmaskLength":
            return Optional.ofNullable(clazz.cast(allocationMaxNetmaskLength()));
        case "AllocationDefaultNetmaskLength":
            return Optional.ofNullable(clazz.cast(allocationDefaultNetmaskLength()));
        case "AllocationResourceTags":
            return Optional.ofNullable(clazz.cast(allocationResourceTags()));
        case "TagSpecifications":
            return Optional.ofNullable(clazz.cast(tagSpecifications()));
        case "ClientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "AwsService":
            return Optional.ofNullable(clazz.cast(awsServiceAsString()));
        case "PublicIpSource":
            return Optional.ofNullable(clazz.cast(publicIpSourceAsString()));
        case "SourceResource":
            return Optional.ofNullable(clazz.cast(sourceResource()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("DryRun", DRY_RUN_FIELD);
        map.put("IpamScopeId", IPAM_SCOPE_ID_FIELD);
        map.put("Locale", LOCALE_FIELD);
        map.put("SourceIpamPoolId", SOURCE_IPAM_POOL_ID_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("AddressFamily", ADDRESS_FAMILY_FIELD);
        map.put("AutoImport", AUTO_IMPORT_FIELD);
        map.put("PubliclyAdvertisable", PUBLICLY_ADVERTISABLE_FIELD);
        map.put("AllocationMinNetmaskLength", ALLOCATION_MIN_NETMASK_LENGTH_FIELD);
        map.put("AllocationMaxNetmaskLength", ALLOCATION_MAX_NETMASK_LENGTH_FIELD);
        map.put("AllocationDefaultNetmaskLength", ALLOCATION_DEFAULT_NETMASK_LENGTH_FIELD);
        map.put("AllocationResourceTag", ALLOCATION_RESOURCE_TAGS_FIELD);
        map.put("TagSpecification", TAG_SPECIFICATIONS_FIELD);
        map.put("ClientToken", CLIENT_TOKEN_FIELD);
        map.put("AwsService", AWS_SERVICE_FIELD);
        map.put("PublicIpSource", PUBLIC_IP_SOURCE_FIELD);
        map.put("SourceResource", SOURCE_RESOURCE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateIpamPoolRequest, T> g) {
        return obj -> g.apply((CreateIpamPoolRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateIpamPoolRequest> {
        /**
         * <p>
         * A check for whether you have the required permissions for the action without actually making the request and
         * provides an error response. If you have the required permissions, the error response is
         * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * </p>
         * 
         * @param dryRun
         *        A check for whether you have the required permissions for the action without actually making the
         *        request and provides an error response. If you have the required permissions, the error response is
         *        <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dryRun(Boolean dryRun);

        /**
         * <p>
         * The ID of the scope in which you would like to create the IPAM pool.
         * </p>
         * 
         * @param ipamScopeId
         *        The ID of the scope in which you would like to create the IPAM pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamScopeId(String ipamScopeId);

        /**
         * <p>
         * The locale for the pool should be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * An Amazon Web Services Region where you want this IPAM pool to be available for allocations.
         * </p>
         * </li>
         * <li>
         * <p>
         * The network border group for an Amazon Web Services Local Zone where you want this IPAM pool to be available
         * for allocations (<a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html#byoip-zone-avail">supported Local
         * Zones</a>). This option is only available for IPAM IPv4 pools in the public scope.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Possible values: Any Amazon Web Services Region or supported Amazon Web Services Local Zone. Default is
         * <code>none</code> and means any locale.
         * </p>
         * 
         * @param locale
         *        The locale for the pool should be one of the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        An Amazon Web Services Region where you want this IPAM pool to be available for allocations.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The network border group for an Amazon Web Services Local Zone where you want this IPAM pool to be
         *        available for allocations (<a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html#byoip-zone-avail">supported
         *        Local Zones</a>). This option is only available for IPAM IPv4 pools in the public scope.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Possible values: Any Amazon Web Services Region or supported Amazon Web Services Local Zone. Default
         *        is <code>none</code> and means any locale.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locale(String locale);

        /**
         * <p>
         * The ID of the source IPAM pool. Use this option to create a pool within an existing pool. Note that the CIDR
         * you provision for the pool within the source pool must be available in the source pool's CIDR range.
         * </p>
         * 
         * @param sourceIpamPoolId
         *        The ID of the source IPAM pool. Use this option to create a pool within an existing pool. Note that
         *        the CIDR you provision for the pool within the source pool must be available in the source pool's CIDR
         *        range.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceIpamPoolId(String sourceIpamPoolId);

        /**
         * <p>
         * A description for the IPAM pool.
         * </p>
         * 
         * @param description
         *        A description for the IPAM pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The IP protocol assigned to this IPAM pool. You must choose either IPv4 or IPv6 protocol for a pool.
         * </p>
         * 
         * @param addressFamily
         *        The IP protocol assigned to this IPAM pool. You must choose either IPv4 or IPv6 protocol for a pool.
         * @see AddressFamily
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AddressFamily
         */
        Builder addressFamily(String addressFamily);

        /**
         * <p>
         * The IP protocol assigned to this IPAM pool. You must choose either IPv4 or IPv6 protocol for a pool.
         * </p>
         * 
         * @param addressFamily
         *        The IP protocol assigned to this IPAM pool. You must choose either IPv4 or IPv6 protocol for a pool.
         * @see AddressFamily
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AddressFamily
         */
        Builder addressFamily(AddressFamily addressFamily);

        /**
         * <p>
         * If selected, IPAM will continuously look for resources within the CIDR range of this pool and automatically
         * import them as allocations into your IPAM. The CIDRs that will be allocated for these resources must not
         * already be allocated to other resources in order for the import to succeed. IPAM will import a CIDR
         * regardless of its compliance with the pool's allocation rules, so a resource might be imported and
         * subsequently marked as noncompliant. If IPAM discovers multiple CIDRs that overlap, IPAM will import the
         * largest CIDR only. If IPAM discovers multiple CIDRs with matching CIDRs, IPAM will randomly import one of
         * them only.
         * </p>
         * <p>
         * A locale must be set on the pool for this feature to work.
         * </p>
         * 
         * @param autoImport
         *        If selected, IPAM will continuously look for resources within the CIDR range of this pool and
         *        automatically import them as allocations into your IPAM. The CIDRs that will be allocated for these
         *        resources must not already be allocated to other resources in order for the import to succeed. IPAM
         *        will import a CIDR regardless of its compliance with the pool's allocation rules, so a resource might
         *        be imported and subsequently marked as noncompliant. If IPAM discovers multiple CIDRs that overlap,
         *        IPAM will import the largest CIDR only. If IPAM discovers multiple CIDRs with matching CIDRs, IPAM
         *        will randomly import one of them only. </p>
         *        <p>
         *        A locale must be set on the pool for this feature to work.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoImport(Boolean autoImport);

        /**
         * <p>
         * Determines if the pool is publicly advertisable. The request can only contain
         * <code>PubliclyAdvertisable</code> if <code>AddressFamily</code> is <code>ipv6</code> and
         * <code>PublicIpSource</code> is <code>byoip</code>.
         * </p>
         * 
         * @param publiclyAdvertisable
         *        Determines if the pool is publicly advertisable. The request can only contain
         *        <code>PubliclyAdvertisable</code> if <code>AddressFamily</code> is <code>ipv6</code> and
         *        <code>PublicIpSource</code> is <code>byoip</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publiclyAdvertisable(Boolean publiclyAdvertisable);

        /**
         * <p>
         * The minimum netmask length required for CIDR allocations in this IPAM pool to be compliant. The minimum
         * netmask length must be less than the maximum netmask length. Possible netmask lengths for IPv4 addresses are
         * 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
         * </p>
         * 
         * @param allocationMinNetmaskLength
         *        The minimum netmask length required for CIDR allocations in this IPAM pool to be compliant. The
         *        minimum netmask length must be less than the maximum netmask length. Possible netmask lengths for IPv4
         *        addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationMinNetmaskLength(Integer allocationMinNetmaskLength);

        /**
         * <p>
         * The maximum netmask length possible for CIDR allocations in this IPAM pool to be compliant. The maximum
         * netmask length must be greater than the minimum netmask length. Possible netmask lengths for IPv4 addresses
         * are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
         * </p>
         * 
         * @param allocationMaxNetmaskLength
         *        The maximum netmask length possible for CIDR allocations in this IPAM pool to be compliant. The
         *        maximum netmask length must be greater than the minimum netmask length. Possible netmask lengths for
         *        IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationMaxNetmaskLength(Integer allocationMaxNetmaskLength);

        /**
         * <p>
         * The default netmask length for allocations added to this pool. If, for example, the CIDR assigned to this
         * pool is 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16.
         * </p>
         * 
         * @param allocationDefaultNetmaskLength
         *        The default netmask length for allocations added to this pool. If, for example, the CIDR assigned to
         *        this pool is 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationDefaultNetmaskLength(Integer allocationDefaultNetmaskLength);

        /**
         * <p>
         * Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these
         * tags will not be allowed to allocate space from the pool. If the resources have their tags changed after they
         * have allocated space or if the allocation tagging requirements are changed on the pool, the resource may be
         * marked as noncompliant.
         * </p>
         * 
         * @param allocationResourceTags
         *        Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have
         *        these tags will not be allowed to allocate space from the pool. If the resources have their tags
         *        changed after they have allocated space or if the allocation tagging requirements are changed on the
         *        pool, the resource may be marked as noncompliant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationResourceTags(Collection<RequestIpamResourceTag> allocationResourceTags);

        /**
         * <p>
         * Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these
         * tags will not be allowed to allocate space from the pool. If the resources have their tags changed after they
         * have allocated space or if the allocation tagging requirements are changed on the pool, the resource may be
         * marked as noncompliant.
         * </p>
         * 
         * @param allocationResourceTags
         *        Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have
         *        these tags will not be allowed to allocate space from the pool. If the resources have their tags
         *        changed after they have allocated space or if the allocation tagging requirements are changed on the
         *        pool, the resource may be marked as noncompliant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationResourceTags(RequestIpamResourceTag... allocationResourceTags);

        /**
         * <p>
         * Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these
         * tags will not be allowed to allocate space from the pool. If the resources have their tags changed after they
         * have allocated space or if the allocation tagging requirements are changed on the pool, the resource may be
         * marked as noncompliant.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.RequestIpamResourceTag.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.ec2.model.RequestIpamResourceTag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.RequestIpamResourceTag.Builder#build()} is called
         * immediately and its result is passed to {@link #allocationResourceTags(List<RequestIpamResourceTag>)}.
         * 
         * @param allocationResourceTags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.RequestIpamResourceTag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #allocationResourceTags(java.util.Collection<RequestIpamResourceTag>)
         */
        Builder allocationResourceTags(Consumer<RequestIpamResourceTag.Builder>... allocationResourceTags);

        /**
         * <p>
         * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag
         * value as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code>
         * and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code>
         * for the filter value.
         * </p>
         * 
         * @param tagSpecifications
         *        The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and
         *        the tag value as the filter value. For example, to find all resources that have a tag with the key
         *        <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter
         *        name and <code>TeamA</code> for the filter value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(Collection<TagSpecification> tagSpecifications);

        /**
         * <p>
         * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag
         * value as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code>
         * and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code>
         * for the filter value.
         * </p>
         * 
         * @param tagSpecifications
         *        The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and
         *        the tag value as the filter value. For example, to find all resources that have a tag with the key
         *        <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter
         *        name and <code>TeamA</code> for the filter value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(TagSpecification... tagSpecifications);

        /**
         * <p>
         * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag
         * value as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code>
         * and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code>
         * for the filter value.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.TagSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder#build()} is called immediately and
         * its result is passed to {@link #tagSpecifications(List<TagSpecification>)}.
         * 
         * @param tagSpecifications
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagSpecifications(java.util.Collection<TagSpecification>)
         */
        Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications);

        /**
         * <p>
         * A unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
         * information, see <a href="https://docs.aws.amazon.com/ec2/latest/devguide/ec2-api-idempotency.html">Ensuring
         * idempotency</a>.
         * </p>
         * 
         * @param clientToken
         *        A unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/ec2/latest/devguide/ec2-api-idempotency.html">Ensuring
         *        idempotency</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows users to
         * use space for Elastic IP addresses and VPCs.
         * </p>
         * 
         * @param awsService
         *        Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows
         *        users to use space for Elastic IP addresses and VPCs.
         * @see IpamPoolAwsService
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPoolAwsService
         */
        Builder awsService(String awsService);

        /**
         * <p>
         * Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows users to
         * use space for Elastic IP addresses and VPCs.
         * </p>
         * 
         * @param awsService
         *        Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows
         *        users to use space for Elastic IP addresses and VPCs.
         * @see IpamPoolAwsService
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPoolAwsService
         */
        Builder awsService(IpamPoolAwsService awsService);

        /**
         * <p>
         * The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools in
         * the public scope. Default is <code>byoip</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a> in the
         * <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR block to a
         * top-level IPv6 pool if PublicIpSource is <code>amazon</code>. For information on increasing the default
         * limit, see <a href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html"> Quotas for your IPAM</a>
         * in the <i>Amazon VPC IPAM User Guide</i>.
         * </p>
         * 
         * @param publicIpSource
         *        The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to
         *        pools in the public scope. Default is <code>byoip</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a>
         *        in the <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR
         *        block to a top-level IPv6 pool if PublicIpSource is <code>amazon</code>. For information on increasing
         *        the default limit, see <a href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html"> Quotas
         *        for your IPAM</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * @see IpamPoolPublicIpSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPoolPublicIpSource
         */
        Builder publicIpSource(String publicIpSource);

        /**
         * <p>
         * The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools in
         * the public scope. Default is <code>byoip</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a> in the
         * <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR block to a
         * top-level IPv6 pool if PublicIpSource is <code>amazon</code>. For information on increasing the default
         * limit, see <a href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html"> Quotas for your IPAM</a>
         * in the <i>Amazon VPC IPAM User Guide</i>.
         * </p>
         * 
         * @param publicIpSource
         *        The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to
         *        pools in the public scope. Default is <code>byoip</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a>
         *        in the <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR
         *        block to a top-level IPv6 pool if PublicIpSource is <code>amazon</code>. For information on increasing
         *        the default limit, see <a href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html"> Quotas
         *        for your IPAM</a> in the <i>Amazon VPC IPAM User Guide</i>.
         * @see IpamPoolPublicIpSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPoolPublicIpSource
         */
        Builder publicIpSource(IpamPoolPublicIpSource publicIpSource);

        /**
         * <p>
         * The resource used to provision CIDRs to a resource planning pool.
         * </p>
         * 
         * @param sourceResource
         *        The resource used to provision CIDRs to a resource planning pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceResource(IpamPoolSourceResourceRequest sourceResource);

        /**
         * <p>
         * The resource used to provision CIDRs to a resource planning pool.
         * </p>
         * This is a convenience method that creates an instance of the {@link IpamPoolSourceResourceRequest.Builder}
         * avoiding the need to create one manually via {@link IpamPoolSourceResourceRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IpamPoolSourceResourceRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #sourceResource(IpamPoolSourceResourceRequest)}.
         * 
         * @param sourceResource
         *        a consumer that will call methods on {@link IpamPoolSourceResourceRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceResource(IpamPoolSourceResourceRequest)
         */
        default Builder sourceResource(Consumer<IpamPoolSourceResourceRequest.Builder> sourceResource) {
            return sourceResource(IpamPoolSourceResourceRequest.builder().applyMutation(sourceResource).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private Boolean dryRun;

        private String ipamScopeId;

        private String locale;

        private String sourceIpamPoolId;

        private String description;

        private String addressFamily;

        private Boolean autoImport;

        private Boolean publiclyAdvertisable;

        private Integer allocationMinNetmaskLength;

        private Integer allocationMaxNetmaskLength;

        private Integer allocationDefaultNetmaskLength;

        private List<RequestIpamResourceTag> allocationResourceTags = DefaultSdkAutoConstructList.getInstance();

        private List<TagSpecification> tagSpecifications = DefaultSdkAutoConstructList.getInstance();

        private String clientToken;

        private String awsService;

        private String publicIpSource;

        private IpamPoolSourceResourceRequest sourceResource;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateIpamPoolRequest model) {
            super(model);
            dryRun(model.dryRun);
            ipamScopeId(model.ipamScopeId);
            locale(model.locale);
            sourceIpamPoolId(model.sourceIpamPoolId);
            description(model.description);
            addressFamily(model.addressFamily);
            autoImport(model.autoImport);
            publiclyAdvertisable(model.publiclyAdvertisable);
            allocationMinNetmaskLength(model.allocationMinNetmaskLength);
            allocationMaxNetmaskLength(model.allocationMaxNetmaskLength);
            allocationDefaultNetmaskLength(model.allocationDefaultNetmaskLength);
            allocationResourceTags(model.allocationResourceTags);
            tagSpecifications(model.tagSpecifications);
            clientToken(model.clientToken);
            awsService(model.awsService);
            publicIpSource(model.publicIpSource);
            sourceResource(model.sourceResource);
        }

        public final Boolean getDryRun() {
            return dryRun;
        }

        public final void setDryRun(Boolean dryRun) {
            this.dryRun = dryRun;
        }

        @Override
        public final Builder dryRun(Boolean dryRun) {
            this.dryRun = dryRun;
            return this;
        }

        public final String getIpamScopeId() {
            return ipamScopeId;
        }

        public final void setIpamScopeId(String ipamScopeId) {
            this.ipamScopeId = ipamScopeId;
        }

        @Override
        public final Builder ipamScopeId(String ipamScopeId) {
            this.ipamScopeId = ipamScopeId;
            return this;
        }

        public final String getLocale() {
            return locale;
        }

        public final void setLocale(String locale) {
            this.locale = locale;
        }

        @Override
        public final Builder locale(String locale) {
            this.locale = locale;
            return this;
        }

        public final String getSourceIpamPoolId() {
            return sourceIpamPoolId;
        }

        public final void setSourceIpamPoolId(String sourceIpamPoolId) {
            this.sourceIpamPoolId = sourceIpamPoolId;
        }

        @Override
        public final Builder sourceIpamPoolId(String sourceIpamPoolId) {
            this.sourceIpamPoolId = sourceIpamPoolId;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getAddressFamily() {
            return addressFamily;
        }

        public final void setAddressFamily(String addressFamily) {
            this.addressFamily = addressFamily;
        }

        @Override
        public final Builder addressFamily(String addressFamily) {
            this.addressFamily = addressFamily;
            return this;
        }

        @Override
        public final Builder addressFamily(AddressFamily addressFamily) {
            this.addressFamily(addressFamily == null ? null : addressFamily.toString());
            return this;
        }

        public final Boolean getAutoImport() {
            return autoImport;
        }

        public final void setAutoImport(Boolean autoImport) {
            this.autoImport = autoImport;
        }

        @Override
        public final Builder autoImport(Boolean autoImport) {
            this.autoImport = autoImport;
            return this;
        }

        public final Boolean getPubliclyAdvertisable() {
            return publiclyAdvertisable;
        }

        public final void setPubliclyAdvertisable(Boolean publiclyAdvertisable) {
            this.publiclyAdvertisable = publiclyAdvertisable;
        }

        @Override
        public final Builder publiclyAdvertisable(Boolean publiclyAdvertisable) {
            this.publiclyAdvertisable = publiclyAdvertisable;
            return this;
        }

        public final Integer getAllocationMinNetmaskLength() {
            return allocationMinNetmaskLength;
        }

        public final void setAllocationMinNetmaskLength(Integer allocationMinNetmaskLength) {
            this.allocationMinNetmaskLength = allocationMinNetmaskLength;
        }

        @Override
        public final Builder allocationMinNetmaskLength(Integer allocationMinNetmaskLength) {
            this.allocationMinNetmaskLength = allocationMinNetmaskLength;
            return this;
        }

        public final Integer getAllocationMaxNetmaskLength() {
            return allocationMaxNetmaskLength;
        }

        public final void setAllocationMaxNetmaskLength(Integer allocationMaxNetmaskLength) {
            this.allocationMaxNetmaskLength = allocationMaxNetmaskLength;
        }

        @Override
        public final Builder allocationMaxNetmaskLength(Integer allocationMaxNetmaskLength) {
            this.allocationMaxNetmaskLength = allocationMaxNetmaskLength;
            return this;
        }

        public final Integer getAllocationDefaultNetmaskLength() {
            return allocationDefaultNetmaskLength;
        }

        public final void setAllocationDefaultNetmaskLength(Integer allocationDefaultNetmaskLength) {
            this.allocationDefaultNetmaskLength = allocationDefaultNetmaskLength;
        }

        @Override
        public final Builder allocationDefaultNetmaskLength(Integer allocationDefaultNetmaskLength) {
            this.allocationDefaultNetmaskLength = allocationDefaultNetmaskLength;
            return this;
        }

        public final List<RequestIpamResourceTag.Builder> getAllocationResourceTags() {
            List<RequestIpamResourceTag.Builder> result = RequestIpamResourceTagListCopier
                    .copyToBuilder(this.allocationResourceTags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAllocationResourceTags(Collection<RequestIpamResourceTag.BuilderImpl> allocationResourceTags) {
            this.allocationResourceTags = RequestIpamResourceTagListCopier.copyFromBuilder(allocationResourceTags);
        }

        @Override
        public final Builder allocationResourceTags(Collection<RequestIpamResourceTag> allocationResourceTags) {
            this.allocationResourceTags = RequestIpamResourceTagListCopier.copy(allocationResourceTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allocationResourceTags(RequestIpamResourceTag... allocationResourceTags) {
            allocationResourceTags(Arrays.asList(allocationResourceTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allocationResourceTags(Consumer<RequestIpamResourceTag.Builder>... allocationResourceTags) {
            allocationResourceTags(Stream.of(allocationResourceTags)
                    .map(c -> RequestIpamResourceTag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<TagSpecification.Builder> getTagSpecifications() {
            List<TagSpecification.Builder> result = TagSpecificationListCopier.copyToBuilder(this.tagSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTagSpecifications(Collection<TagSpecification.BuilderImpl> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copyFromBuilder(tagSpecifications);
        }

        @Override
        public final Builder tagSpecifications(Collection<TagSpecification> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copy(tagSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(TagSpecification... tagSpecifications) {
            tagSpecifications(Arrays.asList(tagSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications) {
            tagSpecifications(Stream.of(tagSpecifications).map(c -> TagSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final String getAwsService() {
            return awsService;
        }

        public final void setAwsService(String awsService) {
            this.awsService = awsService;
        }

        @Override
        public final Builder awsService(String awsService) {
            this.awsService = awsService;
            return this;
        }

        @Override
        public final Builder awsService(IpamPoolAwsService awsService) {
            this.awsService(awsService == null ? null : awsService.toString());
            return this;
        }

        public final String getPublicIpSource() {
            return publicIpSource;
        }

        public final void setPublicIpSource(String publicIpSource) {
            this.publicIpSource = publicIpSource;
        }

        @Override
        public final Builder publicIpSource(String publicIpSource) {
            this.publicIpSource = publicIpSource;
            return this;
        }

        @Override
        public final Builder publicIpSource(IpamPoolPublicIpSource publicIpSource) {
            this.publicIpSource(publicIpSource == null ? null : publicIpSource.toString());
            return this;
        }

        public final IpamPoolSourceResourceRequest.Builder getSourceResource() {
            return sourceResource != null ? sourceResource.toBuilder() : null;
        }

        public final void setSourceResource(IpamPoolSourceResourceRequest.BuilderImpl sourceResource) {
            this.sourceResource = sourceResource != null ? sourceResource.build() : null;
        }

        @Override
        public final Builder sourceResource(IpamPoolSourceResourceRequest sourceResource) {
            this.sourceResource = sourceResource;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateIpamPoolRequest build() {
            return new CreateIpamPoolRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
