/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a System Integrity Protection (SIP) modification task or volume ownership delegation task for an
 * Amazon EC2 Mac instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MacModificationTask implements SdkPojo, Serializable,
        ToCopyableBuilder<MacModificationTask.Builder, MacModificationTask> {
    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceId")
            .getter(getter(MacModificationTask::instanceId))
            .setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId")
                    .unmarshallLocationName("instanceId").build()).build();

    private static final SdkField<String> MAC_MODIFICATION_TASK_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("MacModificationTaskId")
            .getter(getter(MacModificationTask::macModificationTaskId))
            .setter(setter(Builder::macModificationTaskId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MacModificationTaskId")
                    .unmarshallLocationName("macModificationTaskId").build()).build();

    private static final SdkField<MacSystemIntegrityProtectionConfiguration> MAC_SYSTEM_INTEGRITY_PROTECTION_CONFIG_FIELD = SdkField
            .<MacSystemIntegrityProtectionConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("MacSystemIntegrityProtectionConfig")
            .getter(getter(MacModificationTask::macSystemIntegrityProtectionConfig))
            .setter(setter(Builder::macSystemIntegrityProtectionConfig))
            .constructor(MacSystemIntegrityProtectionConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MacSystemIntegrityProtectionConfig")
                    .unmarshallLocationName("macSystemIntegrityProtectionConfig").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime")
            .getter(getter(MacModificationTask::startTime))
            .setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime")
                    .unmarshallLocationName("startTime").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(MacModificationTask::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> TASK_STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TaskState")
            .getter(getter(MacModificationTask::taskStateAsString))
            .setter(setter(Builder::taskState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskState")
                    .unmarshallLocationName("taskState").build()).build();

    private static final SdkField<String> TASK_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TaskType")
            .getter(getter(MacModificationTask::taskTypeAsString))
            .setter(setter(Builder::taskType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskType")
                    .unmarshallLocationName("taskType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_ID_FIELD,
            MAC_MODIFICATION_TASK_ID_FIELD, MAC_SYSTEM_INTEGRITY_PROTECTION_CONFIG_FIELD, START_TIME_FIELD, TAGS_FIELD,
            TASK_STATE_FIELD, TASK_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String instanceId;

    private final String macModificationTaskId;

    private final MacSystemIntegrityProtectionConfiguration macSystemIntegrityProtectionConfig;

    private final Instant startTime;

    private final List<Tag> tags;

    private final String taskState;

    private final String taskType;

    private MacModificationTask(BuilderImpl builder) {
        this.instanceId = builder.instanceId;
        this.macModificationTaskId = builder.macModificationTaskId;
        this.macSystemIntegrityProtectionConfig = builder.macSystemIntegrityProtectionConfig;
        this.startTime = builder.startTime;
        this.tags = builder.tags;
        this.taskState = builder.taskState;
        this.taskType = builder.taskType;
    }

    /**
     * <p>
     * The ID of the Amazon EC2 Mac instance.
     * </p>
     * 
     * @return The ID of the Amazon EC2 Mac instance.
     */
    public final String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The ID of task.
     * </p>
     * 
     * @return The ID of task.
     */
    public final String macModificationTaskId() {
        return macModificationTaskId;
    }

    /**
     * <p>
     * [SIP modification tasks only] Information about the SIP configuration.
     * </p>
     * 
     * @return [SIP modification tasks only] Information about the SIP configuration.
     */
    public final MacSystemIntegrityProtectionConfiguration macSystemIntegrityProtectionConfig() {
        return macSystemIntegrityProtectionConfig;
    }

    /**
     * <p>
     * The date and time the task was created, in the UTC timezone (<code>YYYY-MM-DDThh:mm:ss.sssZ</code>).
     * </p>
     * 
     * @return The date and time the task was created, in the UTC timezone (<code>YYYY-MM-DDThh:mm:ss.sssZ</code>).
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags assigned to the task.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags assigned to the task.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The state of the task.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskState} will
     * return {@link MacModificationTaskState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #taskStateAsString}.
     * </p>
     * 
     * @return The state of the task.
     * @see MacModificationTaskState
     */
    public final MacModificationTaskState taskState() {
        return MacModificationTaskState.fromValue(taskState);
    }

    /**
     * <p>
     * The state of the task.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskState} will
     * return {@link MacModificationTaskState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #taskStateAsString}.
     * </p>
     * 
     * @return The state of the task.
     * @see MacModificationTaskState
     */
    public final String taskStateAsString() {
        return taskState;
    }

    /**
     * <p>
     * The type of task.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskType} will
     * return {@link MacModificationTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #taskTypeAsString}.
     * </p>
     * 
     * @return The type of task.
     * @see MacModificationTaskType
     */
    public final MacModificationTaskType taskType() {
        return MacModificationTaskType.fromValue(taskType);
    }

    /**
     * <p>
     * The type of task.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #taskType} will
     * return {@link MacModificationTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #taskTypeAsString}.
     * </p>
     * 
     * @return The type of task.
     * @see MacModificationTaskType
     */
    public final String taskTypeAsString() {
        return taskType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(macModificationTaskId());
        hashCode = 31 * hashCode + Objects.hashCode(macSystemIntegrityProtectionConfig());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(taskStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(taskTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MacModificationTask)) {
            return false;
        }
        MacModificationTask other = (MacModificationTask) obj;
        return Objects.equals(instanceId(), other.instanceId())
                && Objects.equals(macModificationTaskId(), other.macModificationTaskId())
                && Objects.equals(macSystemIntegrityProtectionConfig(), other.macSystemIntegrityProtectionConfig())
                && Objects.equals(startTime(), other.startTime()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(taskStateAsString(), other.taskStateAsString())
                && Objects.equals(taskTypeAsString(), other.taskTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MacModificationTask").add("InstanceId", instanceId())
                .add("MacModificationTaskId", macModificationTaskId())
                .add("MacSystemIntegrityProtectionConfig", macSystemIntegrityProtectionConfig()).add("StartTime", startTime())
                .add("Tags", hasTags() ? tags() : null).add("TaskState", taskStateAsString()).add("TaskType", taskTypeAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "MacModificationTaskId":
            return Optional.ofNullable(clazz.cast(macModificationTaskId()));
        case "MacSystemIntegrityProtectionConfig":
            return Optional.ofNullable(clazz.cast(macSystemIntegrityProtectionConfig()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "TaskState":
            return Optional.ofNullable(clazz.cast(taskStateAsString()));
        case "TaskType":
            return Optional.ofNullable(clazz.cast(taskTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("InstanceId", INSTANCE_ID_FIELD);
        map.put("MacModificationTaskId", MAC_MODIFICATION_TASK_ID_FIELD);
        map.put("MacSystemIntegrityProtectionConfig", MAC_SYSTEM_INTEGRITY_PROTECTION_CONFIG_FIELD);
        map.put("StartTime", START_TIME_FIELD);
        map.put("TagSet", TAGS_FIELD);
        map.put("TaskState", TASK_STATE_FIELD);
        map.put("TaskType", TASK_TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<MacModificationTask, T> g) {
        return obj -> g.apply((MacModificationTask) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MacModificationTask> {
        /**
         * <p>
         * The ID of the Amazon EC2 Mac instance.
         * </p>
         * 
         * @param instanceId
         *        The ID of the Amazon EC2 Mac instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The ID of task.
         * </p>
         * 
         * @param macModificationTaskId
         *        The ID of task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder macModificationTaskId(String macModificationTaskId);

        /**
         * <p>
         * [SIP modification tasks only] Information about the SIP configuration.
         * </p>
         * 
         * @param macSystemIntegrityProtectionConfig
         *        [SIP modification tasks only] Information about the SIP configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder macSystemIntegrityProtectionConfig(MacSystemIntegrityProtectionConfiguration macSystemIntegrityProtectionConfig);

        /**
         * <p>
         * [SIP modification tasks only] Information about the SIP configuration.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link MacSystemIntegrityProtectionConfiguration.Builder} avoiding the need to create one manually via
         * {@link MacSystemIntegrityProtectionConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MacSystemIntegrityProtectionConfiguration.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #macSystemIntegrityProtectionConfig(MacSystemIntegrityProtectionConfiguration)}.
         * 
         * @param macSystemIntegrityProtectionConfig
         *        a consumer that will call methods on {@link MacSystemIntegrityProtectionConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #macSystemIntegrityProtectionConfig(MacSystemIntegrityProtectionConfiguration)
         */
        default Builder macSystemIntegrityProtectionConfig(
                Consumer<MacSystemIntegrityProtectionConfiguration.Builder> macSystemIntegrityProtectionConfig) {
            return macSystemIntegrityProtectionConfig(MacSystemIntegrityProtectionConfiguration.builder()
                    .applyMutation(macSystemIntegrityProtectionConfig).build());
        }

        /**
         * <p>
         * The date and time the task was created, in the UTC timezone (<code>YYYY-MM-DDThh:mm:ss.sssZ</code>).
         * </p>
         * 
         * @param startTime
         *        The date and time the task was created, in the UTC timezone (<code>YYYY-MM-DDThh:mm:ss.sssZ</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The tags assigned to the task.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags assigned to the task.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags assigned to the task.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The state of the task.
         * </p>
         * 
         * @param taskState
         *        The state of the task.
         * @see MacModificationTaskState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MacModificationTaskState
         */
        Builder taskState(String taskState);

        /**
         * <p>
         * The state of the task.
         * </p>
         * 
         * @param taskState
         *        The state of the task.
         * @see MacModificationTaskState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MacModificationTaskState
         */
        Builder taskState(MacModificationTaskState taskState);

        /**
         * <p>
         * The type of task.
         * </p>
         * 
         * @param taskType
         *        The type of task.
         * @see MacModificationTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MacModificationTaskType
         */
        Builder taskType(String taskType);

        /**
         * <p>
         * The type of task.
         * </p>
         * 
         * @param taskType
         *        The type of task.
         * @see MacModificationTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MacModificationTaskType
         */
        Builder taskType(MacModificationTaskType taskType);
    }

    static final class BuilderImpl implements Builder {
        private String instanceId;

        private String macModificationTaskId;

        private MacSystemIntegrityProtectionConfiguration macSystemIntegrityProtectionConfig;

        private Instant startTime;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String taskState;

        private String taskType;

        private BuilderImpl() {
        }

        private BuilderImpl(MacModificationTask model) {
            instanceId(model.instanceId);
            macModificationTaskId(model.macModificationTaskId);
            macSystemIntegrityProtectionConfig(model.macSystemIntegrityProtectionConfig);
            startTime(model.startTime);
            tags(model.tags);
            taskState(model.taskState);
            taskType(model.taskType);
        }

        public final String getInstanceId() {
            return instanceId;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final String getMacModificationTaskId() {
            return macModificationTaskId;
        }

        public final void setMacModificationTaskId(String macModificationTaskId) {
            this.macModificationTaskId = macModificationTaskId;
        }

        @Override
        public final Builder macModificationTaskId(String macModificationTaskId) {
            this.macModificationTaskId = macModificationTaskId;
            return this;
        }

        public final MacSystemIntegrityProtectionConfiguration.Builder getMacSystemIntegrityProtectionConfig() {
            return macSystemIntegrityProtectionConfig != null ? macSystemIntegrityProtectionConfig.toBuilder() : null;
        }

        public final void setMacSystemIntegrityProtectionConfig(
                MacSystemIntegrityProtectionConfiguration.BuilderImpl macSystemIntegrityProtectionConfig) {
            this.macSystemIntegrityProtectionConfig = macSystemIntegrityProtectionConfig != null ? macSystemIntegrityProtectionConfig
                    .build() : null;
        }

        @Override
        public final Builder macSystemIntegrityProtectionConfig(
                MacSystemIntegrityProtectionConfiguration macSystemIntegrityProtectionConfig) {
            this.macSystemIntegrityProtectionConfig = macSystemIntegrityProtectionConfig;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getTaskState() {
            return taskState;
        }

        public final void setTaskState(String taskState) {
            this.taskState = taskState;
        }

        @Override
        public final Builder taskState(String taskState) {
            this.taskState = taskState;
            return this;
        }

        @Override
        public final Builder taskState(MacModificationTaskState taskState) {
            this.taskState(taskState == null ? null : taskState.toString());
            return this;
        }

        public final String getTaskType() {
            return taskType;
        }

        public final void setTaskType(String taskType) {
            this.taskType = taskType;
        }

        @Override
        public final Builder taskType(String taskType) {
            this.taskType = taskType;
            return this;
        }

        @Override
        public final Builder taskType(MacModificationTaskType taskType) {
            this.taskType(taskType == null ? null : taskType.toString());
            return this;
        }

        @Override
        public MacModificationTask build() {
            return new MacModificationTask(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
