/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The task placement strategy for a task or service. For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html">Task Placement
 * Strategies</a> in the <i>Amazon Elastic Container Service Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PlacementStrategy implements SdkPojo, Serializable,
        ToCopyableBuilder<PlacementStrategy.Builder, PlacementStrategy> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(PlacementStrategy::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> FIELD_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("field")
            .getter(getter(PlacementStrategy::field)).setter(setter(Builder::field))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("field").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, FIELD_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String field;

    private PlacementStrategy(BuilderImpl builder) {
        this.type = builder.type;
        this.field = builder.field;
    }

    /**
     * <p>
     * The type of placement strategy. The <code>random</code> placement strategy randomly places tasks on available
     * candidates. The <code>spread</code> placement strategy spreads placement across available candidates evenly based
     * on the <code>field</code> parameter. The <code>binpack</code> strategy places tasks on available candidates that
     * have the least available amount of the resource that's specified with the <code>field</code> parameter. For
     * example, if you binpack on memory, a task is placed on the instance with the least amount of remaining memory but
     * still enough to run the task.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link PlacementStrategyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of placement strategy. The <code>random</code> placement strategy randomly places tasks on
     *         available candidates. The <code>spread</code> placement strategy spreads placement across available
     *         candidates evenly based on the <code>field</code> parameter. The <code>binpack</code> strategy places
     *         tasks on available candidates that have the least available amount of the resource that's specified with
     *         the <code>field</code> parameter. For example, if you binpack on memory, a task is placed on the instance
     *         with the least amount of remaining memory but still enough to run the task.
     * @see PlacementStrategyType
     */
    public final PlacementStrategyType type() {
        return PlacementStrategyType.fromValue(type);
    }

    /**
     * <p>
     * The type of placement strategy. The <code>random</code> placement strategy randomly places tasks on available
     * candidates. The <code>spread</code> placement strategy spreads placement across available candidates evenly based
     * on the <code>field</code> parameter. The <code>binpack</code> strategy places tasks on available candidates that
     * have the least available amount of the resource that's specified with the <code>field</code> parameter. For
     * example, if you binpack on memory, a task is placed on the instance with the least amount of remaining memory but
     * still enough to run the task.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link PlacementStrategyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of placement strategy. The <code>random</code> placement strategy randomly places tasks on
     *         available candidates. The <code>spread</code> placement strategy spreads placement across available
     *         candidates evenly based on the <code>field</code> parameter. The <code>binpack</code> strategy places
     *         tasks on available candidates that have the least available amount of the resource that's specified with
     *         the <code>field</code> parameter. For example, if you binpack on memory, a task is placed on the instance
     *         with the least amount of remaining memory but still enough to run the task.
     * @see PlacementStrategyType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The field to apply the placement strategy against. For the <code>spread</code> placement strategy, valid values
     * are <code>instanceId</code> (or <code>host</code>, which has the same effect), or any platform or custom
     * attribute that's applied to a container instance, such as <code>attribute:ecs.availability-zone</code>. For the
     * <code>binpack</code> placement strategy, valid values are <code>cpu</code> and <code>memory</code>. For the
     * <code>random</code> placement strategy, this field is not used.
     * </p>
     * 
     * @return The field to apply the placement strategy against. For the <code>spread</code> placement strategy, valid
     *         values are <code>instanceId</code> (or <code>host</code>, which has the same effect), or any platform or
     *         custom attribute that's applied to a container instance, such as
     *         <code>attribute:ecs.availability-zone</code>. For the <code>binpack</code> placement strategy, valid
     *         values are <code>cpu</code> and <code>memory</code>. For the <code>random</code> placement strategy, this
     *         field is not used.
     */
    public final String field() {
        return field;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(field());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PlacementStrategy)) {
            return false;
        }
        PlacementStrategy other = (PlacementStrategy) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(field(), other.field());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PlacementStrategy").add("Type", typeAsString()).add("Field", field()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "field":
            return Optional.ofNullable(clazz.cast(field()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PlacementStrategy, T> g) {
        return obj -> g.apply((PlacementStrategy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PlacementStrategy> {
        /**
         * <p>
         * The type of placement strategy. The <code>random</code> placement strategy randomly places tasks on available
         * candidates. The <code>spread</code> placement strategy spreads placement across available candidates evenly
         * based on the <code>field</code> parameter. The <code>binpack</code> strategy places tasks on available
         * candidates that have the least available amount of the resource that's specified with the <code>field</code>
         * parameter. For example, if you binpack on memory, a task is placed on the instance with the least amount of
         * remaining memory but still enough to run the task.
         * </p>
         * 
         * @param type
         *        The type of placement strategy. The <code>random</code> placement strategy randomly places tasks on
         *        available candidates. The <code>spread</code> placement strategy spreads placement across available
         *        candidates evenly based on the <code>field</code> parameter. The <code>binpack</code> strategy places
         *        tasks on available candidates that have the least available amount of the resource that's specified
         *        with the <code>field</code> parameter. For example, if you binpack on memory, a task is placed on the
         *        instance with the least amount of remaining memory but still enough to run the task.
         * @see PlacementStrategyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlacementStrategyType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of placement strategy. The <code>random</code> placement strategy randomly places tasks on available
         * candidates. The <code>spread</code> placement strategy spreads placement across available candidates evenly
         * based on the <code>field</code> parameter. The <code>binpack</code> strategy places tasks on available
         * candidates that have the least available amount of the resource that's specified with the <code>field</code>
         * parameter. For example, if you binpack on memory, a task is placed on the instance with the least amount of
         * remaining memory but still enough to run the task.
         * </p>
         * 
         * @param type
         *        The type of placement strategy. The <code>random</code> placement strategy randomly places tasks on
         *        available candidates. The <code>spread</code> placement strategy spreads placement across available
         *        candidates evenly based on the <code>field</code> parameter. The <code>binpack</code> strategy places
         *        tasks on available candidates that have the least available amount of the resource that's specified
         *        with the <code>field</code> parameter. For example, if you binpack on memory, a task is placed on the
         *        instance with the least amount of remaining memory but still enough to run the task.
         * @see PlacementStrategyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlacementStrategyType
         */
        Builder type(PlacementStrategyType type);

        /**
         * <p>
         * The field to apply the placement strategy against. For the <code>spread</code> placement strategy, valid
         * values are <code>instanceId</code> (or <code>host</code>, which has the same effect), or any platform or
         * custom attribute that's applied to a container instance, such as <code>attribute:ecs.availability-zone</code>
         * . For the <code>binpack</code> placement strategy, valid values are <code>cpu</code> and <code>memory</code>.
         * For the <code>random</code> placement strategy, this field is not used.
         * </p>
         * 
         * @param field
         *        The field to apply the placement strategy against. For the <code>spread</code> placement strategy,
         *        valid values are <code>instanceId</code> (or <code>host</code>, which has the same effect), or any
         *        platform or custom attribute that's applied to a container instance, such as
         *        <code>attribute:ecs.availability-zone</code>. For the <code>binpack</code> placement strategy, valid
         *        values are <code>cpu</code> and <code>memory</code>. For the <code>random</code> placement strategy,
         *        this field is not used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder field(String field);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String field;

        private BuilderImpl() {
        }

        private BuilderImpl(PlacementStrategy model) {
            type(model.type);
            field(model.field);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        @Transient
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        @Transient
        public final Builder type(PlacementStrategyType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getField() {
            return field;
        }

        public final void setField(String field) {
            this.field = field;
        }

        @Override
        @Transient
        public final Builder field(String field) {
            this.field = field;
            return this;
        }

        @Override
        public PlacementStrategy build() {
            return new PlacementStrategy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
