/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The tag specifications of an Amazon EBS volume.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EBSTagSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<EBSTagSpecification.Builder, EBSTagSpecification> {
    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("resourceType").getter(getter(EBSTagSpecification::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourceType").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("tags")
            .getter(getter(EBSTagSpecification::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> PROPAGATE_TAGS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("propagateTags").getter(getter(EBSTagSpecification::propagateTagsAsString))
            .setter(setter(Builder::propagateTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("propagateTags").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_TYPE_FIELD,
            TAGS_FIELD, PROPAGATE_TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String resourceType;

    private final List<Tag> tags;

    private final String propagateTags;

    private EBSTagSpecification(BuilderImpl builder) {
        this.resourceType = builder.resourceType;
        this.tags = builder.tags;
        this.propagateTags = builder.propagateTags;
    }

    /**
     * <p>
     * The type of volume resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link EBSResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of volume resource.
     * @see EBSResourceType
     */
    public final EBSResourceType resourceType() {
        return EBSResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The type of volume resource.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link EBSResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The type of volume resource.
     * @see EBSResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags applied to this Amazon EBS volume. <code>AmazonECSCreated</code> and <code>AmazonECSManaged</code> are
     * reserved tags that can't be used.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags applied to this Amazon EBS volume. <code>AmazonECSCreated</code> and
     *         <code>AmazonECSManaged</code> are reserved tags that can't be used.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Determines whether to propagate the tags from the task definition to &#x2028;the Amazon EBS volume. Tags can only
     * propagate to a <code>SERVICE</code> specified in &#x2028;<code>ServiceVolumeConfiguration</code>. If no value is
     * specified, the tags aren't &#x2028;propagated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #propagateTags}
     * will return {@link PropagateTags#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #propagateTagsAsString}.
     * </p>
     * 
     * @return Determines whether to propagate the tags from the task definition to &#x2028;the Amazon EBS volume. Tags
     *         can only propagate to a <code>SERVICE</code> specified in &#x2028;<code>ServiceVolumeConfiguration</code>
     *         . If no value is specified, the tags aren't &#x2028;propagated.
     * @see PropagateTags
     */
    public final PropagateTags propagateTags() {
        return PropagateTags.fromValue(propagateTags);
    }

    /**
     * <p>
     * Determines whether to propagate the tags from the task definition to &#x2028;the Amazon EBS volume. Tags can only
     * propagate to a <code>SERVICE</code> specified in &#x2028;<code>ServiceVolumeConfiguration</code>. If no value is
     * specified, the tags aren't &#x2028;propagated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #propagateTags}
     * will return {@link PropagateTags#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #propagateTagsAsString}.
     * </p>
     * 
     * @return Determines whether to propagate the tags from the task definition to &#x2028;the Amazon EBS volume. Tags
     *         can only propagate to a <code>SERVICE</code> specified in &#x2028;<code>ServiceVolumeConfiguration</code>
     *         . If no value is specified, the tags aren't &#x2028;propagated.
     * @see PropagateTags
     */
    public final String propagateTagsAsString() {
        return propagateTags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(propagateTagsAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EBSTagSpecification)) {
            return false;
        }
        EBSTagSpecification other = (EBSTagSpecification) obj;
        return Objects.equals(resourceTypeAsString(), other.resourceTypeAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(propagateTagsAsString(), other.propagateTagsAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EBSTagSpecification").add("ResourceType", resourceTypeAsString())
                .add("Tags", hasTags() ? tags() : null).add("PropagateTags", propagateTagsAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "resourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "propagateTags":
            return Optional.ofNullable(clazz.cast(propagateTagsAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EBSTagSpecification, T> g) {
        return obj -> g.apply((EBSTagSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EBSTagSpecification> {
        /**
         * <p>
         * The type of volume resource.
         * </p>
         * 
         * @param resourceType
         *        The type of volume resource.
         * @see EBSResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EBSResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The type of volume resource.
         * </p>
         * 
         * @param resourceType
         *        The type of volume resource.
         * @see EBSResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EBSResourceType
         */
        Builder resourceType(EBSResourceType resourceType);

        /**
         * <p>
         * The tags applied to this Amazon EBS volume. <code>AmazonECSCreated</code> and <code>AmazonECSManaged</code>
         * are reserved tags that can't be used.
         * </p>
         * 
         * @param tags
         *        The tags applied to this Amazon EBS volume. <code>AmazonECSCreated</code> and
         *        <code>AmazonECSManaged</code> are reserved tags that can't be used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags applied to this Amazon EBS volume. <code>AmazonECSCreated</code> and <code>AmazonECSManaged</code>
         * are reserved tags that can't be used.
         * </p>
         * 
         * @param tags
         *        The tags applied to this Amazon EBS volume. <code>AmazonECSCreated</code> and
         *        <code>AmazonECSManaged</code> are reserved tags that can't be used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags applied to this Amazon EBS volume. <code>AmazonECSCreated</code> and <code>AmazonECSManaged</code>
         * are reserved tags that can't be used.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ecs.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ecs.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ecs.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Determines whether to propagate the tags from the task definition to &#x2028;the Amazon EBS volume. Tags can
         * only propagate to a <code>SERVICE</code> specified in &#x2028;<code>ServiceVolumeConfiguration</code>. If no
         * value is specified, the tags aren't &#x2028;propagated.
         * </p>
         * 
         * @param propagateTags
         *        Determines whether to propagate the tags from the task definition to &#x2028;the Amazon EBS volume.
         *        Tags can only propagate to a <code>SERVICE</code> specified in &#x2028;
         *        <code>ServiceVolumeConfiguration</code>. If no value is specified, the tags aren't &#x2028;propagated.
         * @see PropagateTags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PropagateTags
         */
        Builder propagateTags(String propagateTags);

        /**
         * <p>
         * Determines whether to propagate the tags from the task definition to &#x2028;the Amazon EBS volume. Tags can
         * only propagate to a <code>SERVICE</code> specified in &#x2028;<code>ServiceVolumeConfiguration</code>. If no
         * value is specified, the tags aren't &#x2028;propagated.
         * </p>
         * 
         * @param propagateTags
         *        Determines whether to propagate the tags from the task definition to &#x2028;the Amazon EBS volume.
         *        Tags can only propagate to a <code>SERVICE</code> specified in &#x2028;
         *        <code>ServiceVolumeConfiguration</code>. If no value is specified, the tags aren't &#x2028;propagated.
         * @see PropagateTags
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PropagateTags
         */
        Builder propagateTags(PropagateTags propagateTags);
    }

    static final class BuilderImpl implements Builder {
        private String resourceType;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String propagateTags;

        private BuilderImpl() {
        }

        private BuilderImpl(EBSTagSpecification model) {
            resourceType(model.resourceType);
            tags(model.tags);
            propagateTags(model.propagateTags);
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(EBSResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getPropagateTags() {
            return propagateTags;
        }

        public final void setPropagateTags(String propagateTags) {
            this.propagateTags = propagateTags;
        }

        @Override
        public final Builder propagateTags(String propagateTags) {
            this.propagateTags = propagateTags;
            return this;
        }

        @Override
        public final Builder propagateTags(PropagateTags propagateTags) {
            this.propagateTags(propagateTags == null ? null : propagateTags.toString());
            return this;
        }

        @Override
        public EBSTagSpecification build() {
            return new EBSTagSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
