/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Amazon EKS Pod Identity associations provide the ability to manage credentials for your applications, similar to the
 * way that Amazon EC2 instance profiles provide credentials to Amazon EC2 instances.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PodIdentityAssociation implements SdkPojo, Serializable,
        ToCopyableBuilder<PodIdentityAssociation.Builder, PodIdentityAssociation> {
    private static final SdkField<String> CLUSTER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("clusterName").getter(getter(PodIdentityAssociation::clusterName)).setter(setter(Builder::clusterName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clusterName").build()).build();

    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("namespace").getter(getter(PodIdentityAssociation::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("namespace").build()).build();

    private static final SdkField<String> SERVICE_ACCOUNT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("serviceAccount").getter(getter(PodIdentityAssociation::serviceAccount))
            .setter(setter(Builder::serviceAccount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("serviceAccount").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(PodIdentityAssociation::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<String> ASSOCIATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("associationArn").getter(getter(PodIdentityAssociation::associationArn))
            .setter(setter(Builder::associationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("associationArn").build()).build();

    private static final SdkField<String> ASSOCIATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("associationId").getter(getter(PodIdentityAssociation::associationId))
            .setter(setter(Builder::associationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("associationId").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(PodIdentityAssociation::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(PodIdentityAssociation::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<Instant> MODIFIED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("modifiedAt").getter(getter(PodIdentityAssociation::modifiedAt)).setter(setter(Builder::modifiedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modifiedAt").build()).build();

    private static final SdkField<String> OWNER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ownerArn").getter(getter(PodIdentityAssociation::ownerArn)).setter(setter(Builder::ownerArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ownerArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLUSTER_NAME_FIELD,
            NAMESPACE_FIELD, SERVICE_ACCOUNT_FIELD, ROLE_ARN_FIELD, ASSOCIATION_ARN_FIELD, ASSOCIATION_ID_FIELD, TAGS_FIELD,
            CREATED_AT_FIELD, MODIFIED_AT_FIELD, OWNER_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String clusterName;

    private final String namespace;

    private final String serviceAccount;

    private final String roleArn;

    private final String associationArn;

    private final String associationId;

    private final Map<String, String> tags;

    private final Instant createdAt;

    private final Instant modifiedAt;

    private final String ownerArn;

    private PodIdentityAssociation(BuilderImpl builder) {
        this.clusterName = builder.clusterName;
        this.namespace = builder.namespace;
        this.serviceAccount = builder.serviceAccount;
        this.roleArn = builder.roleArn;
        this.associationArn = builder.associationArn;
        this.associationId = builder.associationId;
        this.tags = builder.tags;
        this.createdAt = builder.createdAt;
        this.modifiedAt = builder.modifiedAt;
        this.ownerArn = builder.ownerArn;
    }

    /**
     * <p>
     * The name of the cluster that the association is in.
     * </p>
     * 
     * @return The name of the cluster that the association is in.
     */
    public final String clusterName() {
        return clusterName;
    }

    /**
     * <p>
     * The name of the Kubernetes namespace inside the cluster to create the association in. The service account and the
     * pods that use the service account must be in this namespace.
     * </p>
     * 
     * @return The name of the Kubernetes namespace inside the cluster to create the association in. The service account
     *         and the pods that use the service account must be in this namespace.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * <p>
     * The name of the Kubernetes service account inside the cluster to associate the IAM credentials with.
     * </p>
     * 
     * @return The name of the Kubernetes service account inside the cluster to associate the IAM credentials with.
     */
    public final String serviceAccount() {
        return serviceAccount;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role to associate with the service account. The EKS Pod Identity agent
     * manages credentials to assume this role for applications in the containers in the pods that use this service
     * account.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role to associate with the service account. The EKS Pod
     *         Identity agent manages credentials to assume this role for applications in the containers in the pods
     *         that use this service account.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the association.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the association.
     */
    public final String associationArn() {
        return associationArn;
    }

    /**
     * <p>
     * The ID of the association.
     * </p>
     * 
     * @return The ID of the association.
     */
    public final String associationId() {
        return associationId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Metadata that assists with categorization and organization. Each tag consists of a key and an optional value. You
     * define both. Tags don't propagate to any other cluster or Amazon Web Services resources.
     * </p>
     * <p>
     * The following basic restrictions apply to tags:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Maximum number of tags per resource – 50
     * </p>
     * </li>
     * <li>
     * <p>
     * For each resource, each tag key must be unique, and each tag key can have only one value.
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum key length – 128 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * Maximum value length – 256 Unicode characters in UTF-8
     * </p>
     * </li>
     * <li>
     * <p>
     * If your tagging schema is used across multiple services and resources, remember that other services may have
     * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces representable
     * in UTF-8, and the following characters: + - = . _ : / @.
     * </p>
     * </li>
     * <li>
     * <p>
     * Tag keys and values are case-sensitive.
     * </p>
     * </li>
     * <li>
     * <p>
     * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix for
     * either keys or values as it is reserved for Amazon Web Services use. You cannot edit or delete tag keys or values
     * with this prefix. Tags with this prefix do not count against your tags per resource limit.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Metadata that assists with categorization and organization. Each tag consists of a key and an optional
     *         value. You define both. Tags don't propagate to any other cluster or Amazon Web Services resources.</p>
     *         <p>
     *         The following basic restrictions apply to tags:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Maximum number of tags per resource – 50
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For each resource, each tag key must be unique, and each tag key can have only one value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum key length – 128 Unicode characters in UTF-8
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Maximum value length – 256 Unicode characters in UTF-8
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If your tagging schema is used across multiple services and resources, remember that other services may
     *         have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
     *         representable in UTF-8, and the following characters: + - = . _ : / @.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Tag keys and values are case-sensitive.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a
     *         prefix for either keys or values as it is reserved for Amazon Web Services use. You cannot edit or delete
     *         tag keys or values with this prefix. Tags with this prefix do not count against your tags per resource
     *         limit.
     *         </p>
     *         </li>
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The timestamp that the association was created at.
     * </p>
     * 
     * @return The timestamp that the association was created at.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The most recent timestamp that the association was modified at
     * </p>
     * 
     * @return The most recent timestamp that the association was modified at
     */
    public final Instant modifiedAt() {
        return modifiedAt;
    }

    /**
     * <p>
     * If defined, the Pod Identity Association is owned by an Amazon EKS Addon.
     * </p>
     * 
     * @return If defined, the Pod Identity Association is owned by an Amazon EKS Addon.
     */
    public final String ownerArn() {
        return ownerArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(clusterName());
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(serviceAccount());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(associationArn());
        hashCode = 31 * hashCode + Objects.hashCode(associationId());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(modifiedAt());
        hashCode = 31 * hashCode + Objects.hashCode(ownerArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PodIdentityAssociation)) {
            return false;
        }
        PodIdentityAssociation other = (PodIdentityAssociation) obj;
        return Objects.equals(clusterName(), other.clusterName()) && Objects.equals(namespace(), other.namespace())
                && Objects.equals(serviceAccount(), other.serviceAccount()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(associationArn(), other.associationArn())
                && Objects.equals(associationId(), other.associationId()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(modifiedAt(), other.modifiedAt()) && Objects.equals(ownerArn(), other.ownerArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PodIdentityAssociation").add("ClusterName", clusterName()).add("Namespace", namespace())
                .add("ServiceAccount", serviceAccount()).add("RoleArn", roleArn()).add("AssociationArn", associationArn())
                .add("AssociationId", associationId()).add("Tags", hasTags() ? tags() : null).add("CreatedAt", createdAt())
                .add("ModifiedAt", modifiedAt()).add("OwnerArn", ownerArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "clusterName":
            return Optional.ofNullable(clazz.cast(clusterName()));
        case "namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "serviceAccount":
            return Optional.ofNullable(clazz.cast(serviceAccount()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "associationArn":
            return Optional.ofNullable(clazz.cast(associationArn()));
        case "associationId":
            return Optional.ofNullable(clazz.cast(associationId()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "modifiedAt":
            return Optional.ofNullable(clazz.cast(modifiedAt()));
        case "ownerArn":
            return Optional.ofNullable(clazz.cast(ownerArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PodIdentityAssociation, T> g) {
        return obj -> g.apply((PodIdentityAssociation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PodIdentityAssociation> {
        /**
         * <p>
         * The name of the cluster that the association is in.
         * </p>
         * 
         * @param clusterName
         *        The name of the cluster that the association is in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterName(String clusterName);

        /**
         * <p>
         * The name of the Kubernetes namespace inside the cluster to create the association in. The service account and
         * the pods that use the service account must be in this namespace.
         * </p>
         * 
         * @param namespace
         *        The name of the Kubernetes namespace inside the cluster to create the association in. The service
         *        account and the pods that use the service account must be in this namespace.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The name of the Kubernetes service account inside the cluster to associate the IAM credentials with.
         * </p>
         * 
         * @param serviceAccount
         *        The name of the Kubernetes service account inside the cluster to associate the IAM credentials with.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceAccount(String serviceAccount);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role to associate with the service account. The EKS Pod Identity
         * agent manages credentials to assume this role for applications in the containers in the pods that use this
         * service account.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) of the IAM role to associate with the service account. The EKS Pod
         *        Identity agent manages credentials to assume this role for applications in the containers in the pods
         *        that use this service account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the association.
         * </p>
         * 
         * @param associationArn
         *        The Amazon Resource Name (ARN) of the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationArn(String associationArn);

        /**
         * <p>
         * The ID of the association.
         * </p>
         * 
         * @param associationId
         *        The ID of the association.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associationId(String associationId);

        /**
         * <p>
         * Metadata that assists with categorization and organization. Each tag consists of a key and an optional value.
         * You define both. Tags don't propagate to any other cluster or Amazon Web Services resources.
         * </p>
         * <p>
         * The following basic restrictions apply to tags:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Maximum number of tags per resource – 50
         * </p>
         * </li>
         * <li>
         * <p>
         * For each resource, each tag key must be unique, and each tag key can have only one value.
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum key length – 128 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * Maximum value length – 256 Unicode characters in UTF-8
         * </p>
         * </li>
         * <li>
         * <p>
         * If your tagging schema is used across multiple services and resources, remember that other services may have
         * restrictions on allowed characters. Generally allowed characters are: letters, numbers, and spaces
         * representable in UTF-8, and the following characters: + - = . _ : / @.
         * </p>
         * </li>
         * <li>
         * <p>
         * Tag keys and values are case-sensitive.
         * </p>
         * </li>
         * <li>
         * <p>
         * Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a prefix
         * for either keys or values as it is reserved for Amazon Web Services use. You cannot edit or delete tag keys
         * or values with this prefix. Tags with this prefix do not count against your tags per resource limit.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tags
         *        Metadata that assists with categorization and organization. Each tag consists of a key and an optional
         *        value. You define both. Tags don't propagate to any other cluster or Amazon Web Services
         *        resources.</p>
         *        <p>
         *        The following basic restrictions apply to tags:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Maximum number of tags per resource – 50
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For each resource, each tag key must be unique, and each tag key can have only one value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum key length – 128 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Maximum value length – 256 Unicode characters in UTF-8
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If your tagging schema is used across multiple services and resources, remember that other services
         *        may have restrictions on allowed characters. Generally allowed characters are: letters, numbers, and
         *        spaces representable in UTF-8, and the following characters: + - = . _ : / @.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Tag keys and values are case-sensitive.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Do not use <code>aws:</code>, <code>AWS:</code>, or any upper or lowercase combination of such as a
         *        prefix for either keys or values as it is reserved for Amazon Web Services use. You cannot edit or
         *        delete tag keys or values with this prefix. Tags with this prefix do not count against your tags per
         *        resource limit.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The timestamp that the association was created at.
         * </p>
         * 
         * @param createdAt
         *        The timestamp that the association was created at.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The most recent timestamp that the association was modified at
         * </p>
         * 
         * @param modifiedAt
         *        The most recent timestamp that the association was modified at
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifiedAt(Instant modifiedAt);

        /**
         * <p>
         * If defined, the Pod Identity Association is owned by an Amazon EKS Addon.
         * </p>
         * 
         * @param ownerArn
         *        If defined, the Pod Identity Association is owned by an Amazon EKS Addon.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerArn(String ownerArn);
    }

    static final class BuilderImpl implements Builder {
        private String clusterName;

        private String namespace;

        private String serviceAccount;

        private String roleArn;

        private String associationArn;

        private String associationId;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private Instant createdAt;

        private Instant modifiedAt;

        private String ownerArn;

        private BuilderImpl() {
        }

        private BuilderImpl(PodIdentityAssociation model) {
            clusterName(model.clusterName);
            namespace(model.namespace);
            serviceAccount(model.serviceAccount);
            roleArn(model.roleArn);
            associationArn(model.associationArn);
            associationId(model.associationId);
            tags(model.tags);
            createdAt(model.createdAt);
            modifiedAt(model.modifiedAt);
            ownerArn(model.ownerArn);
        }

        public final String getClusterName() {
            return clusterName;
        }

        public final void setClusterName(String clusterName) {
            this.clusterName = clusterName;
        }

        @Override
        public final Builder clusterName(String clusterName) {
            this.clusterName = clusterName;
            return this;
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final String getServiceAccount() {
            return serviceAccount;
        }

        public final void setServiceAccount(String serviceAccount) {
            this.serviceAccount = serviceAccount;
        }

        @Override
        public final Builder serviceAccount(String serviceAccount) {
            this.serviceAccount = serviceAccount;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final String getAssociationArn() {
            return associationArn;
        }

        public final void setAssociationArn(String associationArn) {
            this.associationArn = associationArn;
        }

        @Override
        public final Builder associationArn(String associationArn) {
            this.associationArn = associationArn;
            return this;
        }

        public final String getAssociationId() {
            return associationId;
        }

        public final void setAssociationId(String associationId) {
            this.associationId = associationId;
        }

        @Override
        public final Builder associationId(String associationId) {
            this.associationId = associationId;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getModifiedAt() {
            return modifiedAt;
        }

        public final void setModifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
        }

        @Override
        public final Builder modifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
            return this;
        }

        public final String getOwnerArn() {
            return ownerArn;
        }

        public final void setOwnerArn(String ownerArn) {
            this.ownerArn = ownerArn;
        }

        @Override
        public final Builder ownerArn(String ownerArn) {
            this.ownerArn = ownerArn;
            return this;
        }

        @Override
        public PodIdentityAssociation build() {
            return new PodIdentityAssociation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
