/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticsearch.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Status of the advanced options for the specified Elasticsearch domain. Currently, the following advanced options are
 * available:
 * </p>
 * <ul>
 * <li>Option to allow references to indices in an HTTP request body. Must be <code>false</code> when configuring access
 * to individual sub-resources. By default, the value is <code>true</code>. See <a href=
 * "http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomain-configure-advanced-options"
 * target="_blank">Configuration Advanced Options</a> for more information.</li>
 * <li>Option to specify the percentage of heap space that is allocated to field data. By default, this setting is
 * unbounded.</li>
 * </ul>
 * <p>
 * For more information, see <a href=
 * "http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomain-configure-advanced-options"
 * >Configuring Advanced Options</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AdvancedOptionsStatus implements SdkPojo, Serializable,
        ToCopyableBuilder<AdvancedOptionsStatus.Builder, AdvancedOptionsStatus> {
    private static final SdkField<Map<String, String>> OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Options")
            .getter(getter(AdvancedOptionsStatus::options))
            .setter(setter(Builder::options))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Options").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<OptionStatus> STATUS_FIELD = SdkField.<OptionStatus> builder(MarshallingType.SDK_POJO)
            .memberName("Status").getter(getter(AdvancedOptionsStatus::status)).setter(setter(Builder::status))
            .constructor(OptionStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OPTIONS_FIELD, STATUS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Map<String, String> options;

    private final OptionStatus status;

    private AdvancedOptionsStatus(BuilderImpl builder) {
        this.options = builder.options;
        this.status = builder.status;
    }

    /**
     * For responses, this returns true if the service returned a value for the Options property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasOptions() {
        return options != null && !(options instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the status of advanced options for the specified Elasticsearch domain.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOptions} method.
     * </p>
     * 
     * @return Specifies the status of advanced options for the specified Elasticsearch domain.
     */
    public final Map<String, String> options() {
        return options;
    }

    /**
     * <p>
     * Specifies the status of <code>OptionStatus</code> for advanced options for the specified Elasticsearch domain.
     * </p>
     * 
     * @return Specifies the status of <code>OptionStatus</code> for advanced options for the specified Elasticsearch
     *         domain.
     */
    public final OptionStatus status() {
        return status;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasOptions() ? options() : null);
        hashCode = 31 * hashCode + Objects.hashCode(status());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AdvancedOptionsStatus)) {
            return false;
        }
        AdvancedOptionsStatus other = (AdvancedOptionsStatus) obj;
        return hasOptions() == other.hasOptions() && Objects.equals(options(), other.options())
                && Objects.equals(status(), other.status());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AdvancedOptionsStatus").add("Options", hasOptions() ? options() : null).add("Status", status())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Options":
            return Optional.ofNullable(clazz.cast(options()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AdvancedOptionsStatus, T> g) {
        return obj -> g.apply((AdvancedOptionsStatus) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AdvancedOptionsStatus> {
        /**
         * <p>
         * Specifies the status of advanced options for the specified Elasticsearch domain.
         * </p>
         * 
         * @param options
         *        Specifies the status of advanced options for the specified Elasticsearch domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder options(Map<String, String> options);

        /**
         * <p>
         * Specifies the status of <code>OptionStatus</code> for advanced options for the specified Elasticsearch
         * domain.
         * </p>
         * 
         * @param status
         *        Specifies the status of <code>OptionStatus</code> for advanced options for the specified Elasticsearch
         *        domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(OptionStatus status);

        /**
         * <p>
         * Specifies the status of <code>OptionStatus</code> for advanced options for the specified Elasticsearch
         * domain.
         * </p>
         * This is a convenience that creates an instance of the {@link OptionStatus.Builder} avoiding the need to
         * create one manually via {@link OptionStatus#builder()}.
         *
         * When the {@link Consumer} completes, {@link OptionStatus.Builder#build()} is called immediately and its
         * result is passed to {@link #status(OptionStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link OptionStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(OptionStatus)
         */
        default Builder status(Consumer<OptionStatus.Builder> status) {
            return status(OptionStatus.builder().applyMutation(status).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Map<String, String> options = DefaultSdkAutoConstructMap.getInstance();

        private OptionStatus status;

        private BuilderImpl() {
        }

        private BuilderImpl(AdvancedOptionsStatus model) {
            options(model.options);
            status(model.status);
        }

        public final Map<String, String> getOptions() {
            if (options instanceof SdkAutoConstructMap) {
                return null;
            }
            return options;
        }

        public final void setOptions(Map<String, String> options) {
            this.options = AdvancedOptionsCopier.copy(options);
        }

        @Override
        @Transient
        public final Builder options(Map<String, String> options) {
            this.options = AdvancedOptionsCopier.copy(options);
            return this;
        }

        public final OptionStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        public final void setStatus(OptionStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        @Override
        @Transient
        public final Builder status(OptionStatus status) {
            this.status = status;
            return this;
        }

        @Override
        public AdvancedOptionsStatus build() {
            return new AdvancedOptionsStatus(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
