/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emrserverless.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a job run. A job run is a unit of work, such as a Spark JAR, Hive query, or SparkSQL query, that
 * you submit to an EMR Serverless application.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobRun implements SdkPojo, Serializable, ToCopyableBuilder<JobRun.Builder, JobRun> {
    private static final SdkField<String> APPLICATION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("applicationId").getter(getter(JobRun::applicationId)).setter(setter(Builder::applicationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("applicationId").build()).build();

    private static final SdkField<String> JOB_RUN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("jobRunId").getter(getter(JobRun::jobRunId)).setter(setter(Builder::jobRunId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobRunId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(JobRun::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("arn")
            .getter(getter(JobRun::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<String> CREATED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("createdBy").getter(getter(JobRun::createdBy)).setter(setter(Builder::createdBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdBy").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt").getter(getter(JobRun::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt").getter(getter(JobRun::updatedAt)).setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build()).build();

    private static final SdkField<String> EXECUTION_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("executionRole").getter(getter(JobRun::executionRole)).setter(setter(Builder::executionRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("executionRole").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(JobRun::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<String> STATE_DETAILS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("stateDetails").getter(getter(JobRun::stateDetails)).setter(setter(Builder::stateDetails))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("stateDetails").build()).build();

    private static final SdkField<String> RELEASE_LABEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("releaseLabel").getter(getter(JobRun::releaseLabel)).setter(setter(Builder::releaseLabel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("releaseLabel").build()).build();

    private static final SdkField<ConfigurationOverrides> CONFIGURATION_OVERRIDES_FIELD = SdkField
            .<ConfigurationOverrides> builder(MarshallingType.SDK_POJO).memberName("configurationOverrides")
            .getter(getter(JobRun::configurationOverrides)).setter(setter(Builder::configurationOverrides))
            .constructor(ConfigurationOverrides::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configurationOverrides").build())
            .build();

    private static final SdkField<JobDriver> JOB_DRIVER_FIELD = SdkField.<JobDriver> builder(MarshallingType.SDK_POJO)
            .memberName("jobDriver").getter(getter(JobRun::jobDriver)).setter(setter(Builder::jobDriver))
            .constructor(JobDriver::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jobDriver").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(JobRun::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<TotalResourceUtilization> TOTAL_RESOURCE_UTILIZATION_FIELD = SdkField
            .<TotalResourceUtilization> builder(MarshallingType.SDK_POJO).memberName("totalResourceUtilization")
            .getter(getter(JobRun::totalResourceUtilization)).setter(setter(Builder::totalResourceUtilization))
            .constructor(TotalResourceUtilization::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("totalResourceUtilization").build())
            .build();

    private static final SdkField<NetworkConfiguration> NETWORK_CONFIGURATION_FIELD = SdkField
            .<NetworkConfiguration> builder(MarshallingType.SDK_POJO).memberName("networkConfiguration")
            .getter(getter(JobRun::networkConfiguration)).setter(setter(Builder::networkConfiguration))
            .constructor(NetworkConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkConfiguration").build())
            .build();

    private static final SdkField<Integer> TOTAL_EXECUTION_DURATION_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("totalExecutionDurationSeconds")
            .getter(getter(JobRun::totalExecutionDurationSeconds))
            .setter(setter(Builder::totalExecutionDurationSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("totalExecutionDurationSeconds")
                    .build()).build();

    private static final SdkField<Long> EXECUTION_TIMEOUT_MINUTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("executionTimeoutMinutes").getter(getter(JobRun::executionTimeoutMinutes))
            .setter(setter(Builder::executionTimeoutMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("executionTimeoutMinutes").build())
            .build();

    private static final SdkField<ResourceUtilization> BILLED_RESOURCE_UTILIZATION_FIELD = SdkField
            .<ResourceUtilization> builder(MarshallingType.SDK_POJO).memberName("billedResourceUtilization")
            .getter(getter(JobRun::billedResourceUtilization)).setter(setter(Builder::billedResourceUtilization))
            .constructor(ResourceUtilization::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("billedResourceUtilization").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APPLICATION_ID_FIELD,
            JOB_RUN_ID_FIELD, NAME_FIELD, ARN_FIELD, CREATED_BY_FIELD, CREATED_AT_FIELD, UPDATED_AT_FIELD, EXECUTION_ROLE_FIELD,
            STATE_FIELD, STATE_DETAILS_FIELD, RELEASE_LABEL_FIELD, CONFIGURATION_OVERRIDES_FIELD, JOB_DRIVER_FIELD, TAGS_FIELD,
            TOTAL_RESOURCE_UTILIZATION_FIELD, NETWORK_CONFIGURATION_FIELD, TOTAL_EXECUTION_DURATION_SECONDS_FIELD,
            EXECUTION_TIMEOUT_MINUTES_FIELD, BILLED_RESOURCE_UTILIZATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String applicationId;

    private final String jobRunId;

    private final String name;

    private final String arn;

    private final String createdBy;

    private final Instant createdAt;

    private final Instant updatedAt;

    private final String executionRole;

    private final String state;

    private final String stateDetails;

    private final String releaseLabel;

    private final ConfigurationOverrides configurationOverrides;

    private final JobDriver jobDriver;

    private final Map<String, String> tags;

    private final TotalResourceUtilization totalResourceUtilization;

    private final NetworkConfiguration networkConfiguration;

    private final Integer totalExecutionDurationSeconds;

    private final Long executionTimeoutMinutes;

    private final ResourceUtilization billedResourceUtilization;

    private JobRun(BuilderImpl builder) {
        this.applicationId = builder.applicationId;
        this.jobRunId = builder.jobRunId;
        this.name = builder.name;
        this.arn = builder.arn;
        this.createdBy = builder.createdBy;
        this.createdAt = builder.createdAt;
        this.updatedAt = builder.updatedAt;
        this.executionRole = builder.executionRole;
        this.state = builder.state;
        this.stateDetails = builder.stateDetails;
        this.releaseLabel = builder.releaseLabel;
        this.configurationOverrides = builder.configurationOverrides;
        this.jobDriver = builder.jobDriver;
        this.tags = builder.tags;
        this.totalResourceUtilization = builder.totalResourceUtilization;
        this.networkConfiguration = builder.networkConfiguration;
        this.totalExecutionDurationSeconds = builder.totalExecutionDurationSeconds;
        this.executionTimeoutMinutes = builder.executionTimeoutMinutes;
        this.billedResourceUtilization = builder.billedResourceUtilization;
    }

    /**
     * <p>
     * The ID of the application the job is running on.
     * </p>
     * 
     * @return The ID of the application the job is running on.
     */
    public final String applicationId() {
        return applicationId;
    }

    /**
     * <p>
     * The ID of the job run.
     * </p>
     * 
     * @return The ID of the job run.
     */
    public final String jobRunId() {
        return jobRunId;
    }

    /**
     * <p>
     * The optional job run name. This doesn't have to be unique.
     * </p>
     * 
     * @return The optional job run name. This doesn't have to be unique.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The execution role ARN of the job run.
     * </p>
     * 
     * @return The execution role ARN of the job run.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The user who created the job run.
     * </p>
     * 
     * @return The user who created the job run.
     */
    public final String createdBy() {
        return createdBy;
    }

    /**
     * <p>
     * The date and time when the job run was created.
     * </p>
     * 
     * @return The date and time when the job run was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The date and time when the job run was updated.
     * </p>
     * 
     * @return The date and time when the job run was updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * <p>
     * The execution role ARN of the job run.
     * </p>
     * 
     * @return The execution role ARN of the job run.
     */
    public final String executionRole() {
        return executionRole;
    }

    /**
     * <p>
     * The state of the job run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link JobRunState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the job run.
     * @see JobRunState
     */
    public final JobRunState state() {
        return JobRunState.fromValue(state);
    }

    /**
     * <p>
     * The state of the job run.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link JobRunState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the job run.
     * @see JobRunState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The state details of the job run.
     * </p>
     * 
     * @return The state details of the job run.
     */
    public final String stateDetails() {
        return stateDetails;
    }

    /**
     * <p>
     * The EMR release associated with the application your job is running on.
     * </p>
     * 
     * @return The EMR release associated with the application your job is running on.
     */
    public final String releaseLabel() {
        return releaseLabel;
    }

    /**
     * <p>
     * The configuration settings that are used to override default configuration.
     * </p>
     * 
     * @return The configuration settings that are used to override default configuration.
     */
    public final ConfigurationOverrides configurationOverrides() {
        return configurationOverrides;
    }

    /**
     * <p>
     * The job driver for the job run.
     * </p>
     * 
     * @return The job driver for the job run.
     */
    public final JobDriver jobDriver() {
        return jobDriver;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The tags assigned to the job run.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags assigned to the job run.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * The aggregate vCPU, memory, and storage resources used from the time the job starts to execute, until the time
     * the job terminates, rounded up to the nearest second.
     * </p>
     * 
     * @return The aggregate vCPU, memory, and storage resources used from the time the job starts to execute, until the
     *         time the job terminates, rounded up to the nearest second.
     */
    public final TotalResourceUtilization totalResourceUtilization() {
        return totalResourceUtilization;
    }

    /**
     * Returns the value of the NetworkConfiguration property for this object.
     * 
     * @return The value of the NetworkConfiguration property for this object.
     */
    public final NetworkConfiguration networkConfiguration() {
        return networkConfiguration;
    }

    /**
     * <p>
     * The job run total execution duration in seconds. This field is only available for job runs in a
     * <code>COMPLETED</code>, <code>FAILED</code>, or <code>CANCELLED</code> state.
     * </p>
     * 
     * @return The job run total execution duration in seconds. This field is only available for job runs in a
     *         <code>COMPLETED</code>, <code>FAILED</code>, or <code>CANCELLED</code> state.
     */
    public final Integer totalExecutionDurationSeconds() {
        return totalExecutionDurationSeconds;
    }

    /**
     * <p>
     * Returns the job run timeout value from the <code>StartJobRun</code> call. If no timeout was specified, then it
     * returns the default timeout of 720 minutes.
     * </p>
     * 
     * @return Returns the job run timeout value from the <code>StartJobRun</code> call. If no timeout was specified,
     *         then it returns the default timeout of 720 minutes.
     */
    public final Long executionTimeoutMinutes() {
        return executionTimeoutMinutes;
    }

    /**
     * <p>
     * The aggregate vCPU, memory, and storage that AWS has billed for the job run. The billed resources include a
     * 1-minute minimum usage for workers, plus additional storage over 20 GB per worker. Note that billed resources do
     * not include usage for idle pre-initialized workers.
     * </p>
     * 
     * @return The aggregate vCPU, memory, and storage that AWS has billed for the job run. The billed resources include
     *         a 1-minute minimum usage for workers, plus additional storage over 20 GB per worker. Note that billed
     *         resources do not include usage for idle pre-initialized workers.
     */
    public final ResourceUtilization billedResourceUtilization() {
        return billedResourceUtilization;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(applicationId());
        hashCode = 31 * hashCode + Objects.hashCode(jobRunId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(createdBy());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(executionRole());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateDetails());
        hashCode = 31 * hashCode + Objects.hashCode(releaseLabel());
        hashCode = 31 * hashCode + Objects.hashCode(configurationOverrides());
        hashCode = 31 * hashCode + Objects.hashCode(jobDriver());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(totalResourceUtilization());
        hashCode = 31 * hashCode + Objects.hashCode(networkConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(totalExecutionDurationSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(executionTimeoutMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(billedResourceUtilization());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobRun)) {
            return false;
        }
        JobRun other = (JobRun) obj;
        return Objects.equals(applicationId(), other.applicationId()) && Objects.equals(jobRunId(), other.jobRunId())
                && Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(createdBy(), other.createdBy()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(updatedAt(), other.updatedAt()) && Objects.equals(executionRole(), other.executionRole())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(stateDetails(), other.stateDetails())
                && Objects.equals(releaseLabel(), other.releaseLabel())
                && Objects.equals(configurationOverrides(), other.configurationOverrides())
                && Objects.equals(jobDriver(), other.jobDriver()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(totalResourceUtilization(), other.totalResourceUtilization())
                && Objects.equals(networkConfiguration(), other.networkConfiguration())
                && Objects.equals(totalExecutionDurationSeconds(), other.totalExecutionDurationSeconds())
                && Objects.equals(executionTimeoutMinutes(), other.executionTimeoutMinutes())
                && Objects.equals(billedResourceUtilization(), other.billedResourceUtilization());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JobRun").add("ApplicationId", applicationId()).add("JobRunId", jobRunId()).add("Name", name())
                .add("Arn", arn()).add("CreatedBy", createdBy()).add("CreatedAt", createdAt()).add("UpdatedAt", updatedAt())
                .add("ExecutionRole", executionRole()).add("State", stateAsString()).add("StateDetails", stateDetails())
                .add("ReleaseLabel", releaseLabel()).add("ConfigurationOverrides", configurationOverrides())
                .add("JobDriver", jobDriver()).add("Tags", hasTags() ? tags() : null)
                .add("TotalResourceUtilization", totalResourceUtilization()).add("NetworkConfiguration", networkConfiguration())
                .add("TotalExecutionDurationSeconds", totalExecutionDurationSeconds())
                .add("ExecutionTimeoutMinutes", executionTimeoutMinutes())
                .add("BilledResourceUtilization", billedResourceUtilization()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "applicationId":
            return Optional.ofNullable(clazz.cast(applicationId()));
        case "jobRunId":
            return Optional.ofNullable(clazz.cast(jobRunId()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "createdBy":
            return Optional.ofNullable(clazz.cast(createdBy()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "executionRole":
            return Optional.ofNullable(clazz.cast(executionRole()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "stateDetails":
            return Optional.ofNullable(clazz.cast(stateDetails()));
        case "releaseLabel":
            return Optional.ofNullable(clazz.cast(releaseLabel()));
        case "configurationOverrides":
            return Optional.ofNullable(clazz.cast(configurationOverrides()));
        case "jobDriver":
            return Optional.ofNullable(clazz.cast(jobDriver()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "totalResourceUtilization":
            return Optional.ofNullable(clazz.cast(totalResourceUtilization()));
        case "networkConfiguration":
            return Optional.ofNullable(clazz.cast(networkConfiguration()));
        case "totalExecutionDurationSeconds":
            return Optional.ofNullable(clazz.cast(totalExecutionDurationSeconds()));
        case "executionTimeoutMinutes":
            return Optional.ofNullable(clazz.cast(executionTimeoutMinutes()));
        case "billedResourceUtilization":
            return Optional.ofNullable(clazz.cast(billedResourceUtilization()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<JobRun, T> g) {
        return obj -> g.apply((JobRun) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobRun> {
        /**
         * <p>
         * The ID of the application the job is running on.
         * </p>
         * 
         * @param applicationId
         *        The ID of the application the job is running on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationId(String applicationId);

        /**
         * <p>
         * The ID of the job run.
         * </p>
         * 
         * @param jobRunId
         *        The ID of the job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobRunId(String jobRunId);

        /**
         * <p>
         * The optional job run name. This doesn't have to be unique.
         * </p>
         * 
         * @param name
         *        The optional job run name. This doesn't have to be unique.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The execution role ARN of the job run.
         * </p>
         * 
         * @param arn
         *        The execution role ARN of the job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The user who created the job run.
         * </p>
         * 
         * @param createdBy
         *        The user who created the job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBy(String createdBy);

        /**
         * <p>
         * The date and time when the job run was created.
         * </p>
         * 
         * @param createdAt
         *        The date and time when the job run was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The date and time when the job run was updated.
         * </p>
         * 
         * @param updatedAt
         *        The date and time when the job run was updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * The execution role ARN of the job run.
         * </p>
         * 
         * @param executionRole
         *        The execution role ARN of the job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRole(String executionRole);

        /**
         * <p>
         * The state of the job run.
         * </p>
         * 
         * @param state
         *        The state of the job run.
         * @see JobRunState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobRunState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the job run.
         * </p>
         * 
         * @param state
         *        The state of the job run.
         * @see JobRunState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobRunState
         */
        Builder state(JobRunState state);

        /**
         * <p>
         * The state details of the job run.
         * </p>
         * 
         * @param stateDetails
         *        The state details of the job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateDetails(String stateDetails);

        /**
         * <p>
         * The EMR release associated with the application your job is running on.
         * </p>
         * 
         * @param releaseLabel
         *        The EMR release associated with the application your job is running on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder releaseLabel(String releaseLabel);

        /**
         * <p>
         * The configuration settings that are used to override default configuration.
         * </p>
         * 
         * @param configurationOverrides
         *        The configuration settings that are used to override default configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurationOverrides(ConfigurationOverrides configurationOverrides);

        /**
         * <p>
         * The configuration settings that are used to override default configuration.
         * </p>
         * This is a convenience method that creates an instance of the {@link ConfigurationOverrides.Builder} avoiding
         * the need to create one manually via {@link ConfigurationOverrides#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ConfigurationOverrides.Builder#build()} is called immediately and
         * its result is passed to {@link #configurationOverrides(ConfigurationOverrides)}.
         * 
         * @param configurationOverrides
         *        a consumer that will call methods on {@link ConfigurationOverrides.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configurationOverrides(ConfigurationOverrides)
         */
        default Builder configurationOverrides(Consumer<ConfigurationOverrides.Builder> configurationOverrides) {
            return configurationOverrides(ConfigurationOverrides.builder().applyMutation(configurationOverrides).build());
        }

        /**
         * <p>
         * The job driver for the job run.
         * </p>
         * 
         * @param jobDriver
         *        The job driver for the job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jobDriver(JobDriver jobDriver);

        /**
         * <p>
         * The job driver for the job run.
         * </p>
         * This is a convenience method that creates an instance of the {@link JobDriver.Builder} avoiding the need to
         * create one manually via {@link JobDriver#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JobDriver.Builder#build()} is called immediately and its result
         * is passed to {@link #jobDriver(JobDriver)}.
         * 
         * @param jobDriver
         *        a consumer that will call methods on {@link JobDriver.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jobDriver(JobDriver)
         */
        default Builder jobDriver(Consumer<JobDriver.Builder> jobDriver) {
            return jobDriver(JobDriver.builder().applyMutation(jobDriver).build());
        }

        /**
         * <p>
         * The tags assigned to the job run.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the job run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The aggregate vCPU, memory, and storage resources used from the time the job starts to execute, until the
         * time the job terminates, rounded up to the nearest second.
         * </p>
         * 
         * @param totalResourceUtilization
         *        The aggregate vCPU, memory, and storage resources used from the time the job starts to execute, until
         *        the time the job terminates, rounded up to the nearest second.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalResourceUtilization(TotalResourceUtilization totalResourceUtilization);

        /**
         * <p>
         * The aggregate vCPU, memory, and storage resources used from the time the job starts to execute, until the
         * time the job terminates, rounded up to the nearest second.
         * </p>
         * This is a convenience method that creates an instance of the {@link TotalResourceUtilization.Builder}
         * avoiding the need to create one manually via {@link TotalResourceUtilization#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TotalResourceUtilization.Builder#build()} is called immediately
         * and its result is passed to {@link #totalResourceUtilization(TotalResourceUtilization)}.
         * 
         * @param totalResourceUtilization
         *        a consumer that will call methods on {@link TotalResourceUtilization.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #totalResourceUtilization(TotalResourceUtilization)
         */
        default Builder totalResourceUtilization(Consumer<TotalResourceUtilization.Builder> totalResourceUtilization) {
            return totalResourceUtilization(TotalResourceUtilization.builder().applyMutation(totalResourceUtilization).build());
        }

        /**
         * Sets the value of the NetworkConfiguration property for this object.
         *
         * @param networkConfiguration
         *        The new value for the NetworkConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkConfiguration(NetworkConfiguration networkConfiguration);

        /**
         * Sets the value of the NetworkConfiguration property for this object.
         *
         * This is a convenience method that creates an instance of the {@link NetworkConfiguration.Builder} avoiding
         * the need to create one manually via {@link NetworkConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NetworkConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #networkConfiguration(NetworkConfiguration)}.
         * 
         * @param networkConfiguration
         *        a consumer that will call methods on {@link NetworkConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkConfiguration(NetworkConfiguration)
         */
        default Builder networkConfiguration(Consumer<NetworkConfiguration.Builder> networkConfiguration) {
            return networkConfiguration(NetworkConfiguration.builder().applyMutation(networkConfiguration).build());
        }

        /**
         * <p>
         * The job run total execution duration in seconds. This field is only available for job runs in a
         * <code>COMPLETED</code>, <code>FAILED</code>, or <code>CANCELLED</code> state.
         * </p>
         * 
         * @param totalExecutionDurationSeconds
         *        The job run total execution duration in seconds. This field is only available for job runs in a
         *        <code>COMPLETED</code>, <code>FAILED</code>, or <code>CANCELLED</code> state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalExecutionDurationSeconds(Integer totalExecutionDurationSeconds);

        /**
         * <p>
         * Returns the job run timeout value from the <code>StartJobRun</code> call. If no timeout was specified, then
         * it returns the default timeout of 720 minutes.
         * </p>
         * 
         * @param executionTimeoutMinutes
         *        Returns the job run timeout value from the <code>StartJobRun</code> call. If no timeout was specified,
         *        then it returns the default timeout of 720 minutes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionTimeoutMinutes(Long executionTimeoutMinutes);

        /**
         * <p>
         * The aggregate vCPU, memory, and storage that AWS has billed for the job run. The billed resources include a
         * 1-minute minimum usage for workers, plus additional storage over 20 GB per worker. Note that billed resources
         * do not include usage for idle pre-initialized workers.
         * </p>
         * 
         * @param billedResourceUtilization
         *        The aggregate vCPU, memory, and storage that AWS has billed for the job run. The billed resources
         *        include a 1-minute minimum usage for workers, plus additional storage over 20 GB per worker. Note that
         *        billed resources do not include usage for idle pre-initialized workers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder billedResourceUtilization(ResourceUtilization billedResourceUtilization);

        /**
         * <p>
         * The aggregate vCPU, memory, and storage that AWS has billed for the job run. The billed resources include a
         * 1-minute minimum usage for workers, plus additional storage over 20 GB per worker. Note that billed resources
         * do not include usage for idle pre-initialized workers.
         * </p>
         * This is a convenience method that creates an instance of the {@link ResourceUtilization.Builder} avoiding the
         * need to create one manually via {@link ResourceUtilization#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResourceUtilization.Builder#build()} is called immediately and
         * its result is passed to {@link #billedResourceUtilization(ResourceUtilization)}.
         * 
         * @param billedResourceUtilization
         *        a consumer that will call methods on {@link ResourceUtilization.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #billedResourceUtilization(ResourceUtilization)
         */
        default Builder billedResourceUtilization(Consumer<ResourceUtilization.Builder> billedResourceUtilization) {
            return billedResourceUtilization(ResourceUtilization.builder().applyMutation(billedResourceUtilization).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String applicationId;

        private String jobRunId;

        private String name;

        private String arn;

        private String createdBy;

        private Instant createdAt;

        private Instant updatedAt;

        private String executionRole;

        private String state;

        private String stateDetails;

        private String releaseLabel;

        private ConfigurationOverrides configurationOverrides;

        private JobDriver jobDriver;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private TotalResourceUtilization totalResourceUtilization;

        private NetworkConfiguration networkConfiguration;

        private Integer totalExecutionDurationSeconds;

        private Long executionTimeoutMinutes;

        private ResourceUtilization billedResourceUtilization;

        private BuilderImpl() {
        }

        private BuilderImpl(JobRun model) {
            applicationId(model.applicationId);
            jobRunId(model.jobRunId);
            name(model.name);
            arn(model.arn);
            createdBy(model.createdBy);
            createdAt(model.createdAt);
            updatedAt(model.updatedAt);
            executionRole(model.executionRole);
            state(model.state);
            stateDetails(model.stateDetails);
            releaseLabel(model.releaseLabel);
            configurationOverrides(model.configurationOverrides);
            jobDriver(model.jobDriver);
            tags(model.tags);
            totalResourceUtilization(model.totalResourceUtilization);
            networkConfiguration(model.networkConfiguration);
            totalExecutionDurationSeconds(model.totalExecutionDurationSeconds);
            executionTimeoutMinutes(model.executionTimeoutMinutes);
            billedResourceUtilization(model.billedResourceUtilization);
        }

        public final String getApplicationId() {
            return applicationId;
        }

        public final void setApplicationId(String applicationId) {
            this.applicationId = applicationId;
        }

        @Override
        public final Builder applicationId(String applicationId) {
            this.applicationId = applicationId;
            return this;
        }

        public final String getJobRunId() {
            return jobRunId;
        }

        public final void setJobRunId(String jobRunId) {
            this.jobRunId = jobRunId;
        }

        @Override
        public final Builder jobRunId(String jobRunId) {
            this.jobRunId = jobRunId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getCreatedBy() {
            return createdBy;
        }

        public final void setCreatedBy(String createdBy) {
            this.createdBy = createdBy;
        }

        @Override
        public final Builder createdBy(String createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final String getExecutionRole() {
            return executionRole;
        }

        public final void setExecutionRole(String executionRole) {
            this.executionRole = executionRole;
        }

        @Override
        public final Builder executionRole(String executionRole) {
            this.executionRole = executionRole;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(JobRunState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getStateDetails() {
            return stateDetails;
        }

        public final void setStateDetails(String stateDetails) {
            this.stateDetails = stateDetails;
        }

        @Override
        public final Builder stateDetails(String stateDetails) {
            this.stateDetails = stateDetails;
            return this;
        }

        public final String getReleaseLabel() {
            return releaseLabel;
        }

        public final void setReleaseLabel(String releaseLabel) {
            this.releaseLabel = releaseLabel;
        }

        @Override
        public final Builder releaseLabel(String releaseLabel) {
            this.releaseLabel = releaseLabel;
            return this;
        }

        public final ConfigurationOverrides.Builder getConfigurationOverrides() {
            return configurationOverrides != null ? configurationOverrides.toBuilder() : null;
        }

        public final void setConfigurationOverrides(ConfigurationOverrides.BuilderImpl configurationOverrides) {
            this.configurationOverrides = configurationOverrides != null ? configurationOverrides.build() : null;
        }

        @Override
        public final Builder configurationOverrides(ConfigurationOverrides configurationOverrides) {
            this.configurationOverrides = configurationOverrides;
            return this;
        }

        public final JobDriver.Builder getJobDriver() {
            return jobDriver != null ? jobDriver.toBuilder() : null;
        }

        public final void setJobDriver(JobDriver.BuilderImpl jobDriver) {
            this.jobDriver = jobDriver != null ? jobDriver.build() : null;
        }

        @Override
        public final Builder jobDriver(JobDriver jobDriver) {
            this.jobDriver = jobDriver;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final TotalResourceUtilization.Builder getTotalResourceUtilization() {
            return totalResourceUtilization != null ? totalResourceUtilization.toBuilder() : null;
        }

        public final void setTotalResourceUtilization(TotalResourceUtilization.BuilderImpl totalResourceUtilization) {
            this.totalResourceUtilization = totalResourceUtilization != null ? totalResourceUtilization.build() : null;
        }

        @Override
        public final Builder totalResourceUtilization(TotalResourceUtilization totalResourceUtilization) {
            this.totalResourceUtilization = totalResourceUtilization;
            return this;
        }

        public final NetworkConfiguration.Builder getNetworkConfiguration() {
            return networkConfiguration != null ? networkConfiguration.toBuilder() : null;
        }

        public final void setNetworkConfiguration(NetworkConfiguration.BuilderImpl networkConfiguration) {
            this.networkConfiguration = networkConfiguration != null ? networkConfiguration.build() : null;
        }

        @Override
        public final Builder networkConfiguration(NetworkConfiguration networkConfiguration) {
            this.networkConfiguration = networkConfiguration;
            return this;
        }

        public final Integer getTotalExecutionDurationSeconds() {
            return totalExecutionDurationSeconds;
        }

        public final void setTotalExecutionDurationSeconds(Integer totalExecutionDurationSeconds) {
            this.totalExecutionDurationSeconds = totalExecutionDurationSeconds;
        }

        @Override
        public final Builder totalExecutionDurationSeconds(Integer totalExecutionDurationSeconds) {
            this.totalExecutionDurationSeconds = totalExecutionDurationSeconds;
            return this;
        }

        public final Long getExecutionTimeoutMinutes() {
            return executionTimeoutMinutes;
        }

        public final void setExecutionTimeoutMinutes(Long executionTimeoutMinutes) {
            this.executionTimeoutMinutes = executionTimeoutMinutes;
        }

        @Override
        public final Builder executionTimeoutMinutes(Long executionTimeoutMinutes) {
            this.executionTimeoutMinutes = executionTimeoutMinutes;
            return this;
        }

        public final ResourceUtilization.Builder getBilledResourceUtilization() {
            return billedResourceUtilization != null ? billedResourceUtilization.toBuilder() : null;
        }

        public final void setBilledResourceUtilization(ResourceUtilization.BuilderImpl billedResourceUtilization) {
            this.billedResourceUtilization = billedResourceUtilization != null ? billedResourceUtilization.build() : null;
        }

        @Override
        public final Builder billedResourceUtilization(ResourceUtilization billedResourceUtilization) {
            this.billedResourceUtilization = billedResourceUtilization;
            return this;
        }

        @Override
        public JobRun build() {
            return new JobRun(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
