/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.firehose.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A serializer to use for converting data to the Parquet format before storing it in Amazon S3. For more information,
 * see <a href="https://parquet.apache.org/documentation/latest/">Apache Parquet</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ParquetSerDe implements SdkPojo, Serializable, ToCopyableBuilder<ParquetSerDe.Builder, ParquetSerDe> {
    private static final SdkField<Integer> BLOCK_SIZE_BYTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("BlockSizeBytes").getter(getter(ParquetSerDe::blockSizeBytes)).setter(setter(Builder::blockSizeBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BlockSizeBytes").build()).build();

    private static final SdkField<Integer> PAGE_SIZE_BYTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("PageSizeBytes").getter(getter(ParquetSerDe::pageSizeBytes)).setter(setter(Builder::pageSizeBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PageSizeBytes").build()).build();

    private static final SdkField<String> COMPRESSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Compression").getter(getter(ParquetSerDe::compressionAsString)).setter(setter(Builder::compression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Compression").build()).build();

    private static final SdkField<Boolean> ENABLE_DICTIONARY_COMPRESSION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableDictionaryCompression")
            .getter(getter(ParquetSerDe::enableDictionaryCompression))
            .setter(setter(Builder::enableDictionaryCompression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableDictionaryCompression")
                    .build()).build();

    private static final SdkField<Integer> MAX_PADDING_BYTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxPaddingBytes").getter(getter(ParquetSerDe::maxPaddingBytes)).setter(setter(Builder::maxPaddingBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxPaddingBytes").build()).build();

    private static final SdkField<String> WRITER_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("WriterVersion").getter(getter(ParquetSerDe::writerVersionAsString))
            .setter(setter(Builder::writerVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WriterVersion").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BLOCK_SIZE_BYTES_FIELD,
            PAGE_SIZE_BYTES_FIELD, COMPRESSION_FIELD, ENABLE_DICTIONARY_COMPRESSION_FIELD, MAX_PADDING_BYTES_FIELD,
            WRITER_VERSION_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer blockSizeBytes;

    private final Integer pageSizeBytes;

    private final String compression;

    private final Boolean enableDictionaryCompression;

    private final Integer maxPaddingBytes;

    private final String writerVersion;

    private ParquetSerDe(BuilderImpl builder) {
        this.blockSizeBytes = builder.blockSizeBytes;
        this.pageSizeBytes = builder.pageSizeBytes;
        this.compression = builder.compression;
        this.enableDictionaryCompression = builder.enableDictionaryCompression;
        this.maxPaddingBytes = builder.maxPaddingBytes;
        this.writerVersion = builder.writerVersion;
    }

    /**
     * <p>
     * The Hadoop Distributed File System (HDFS) block size. This is useful if you intend to copy the data from Amazon
     * S3 to HDFS before querying. The default is 256 MiB and the minimum is 64 MiB. Firehose uses this value for
     * padding calculations.
     * </p>
     * 
     * @return The Hadoop Distributed File System (HDFS) block size. This is useful if you intend to copy the data from
     *         Amazon S3 to HDFS before querying. The default is 256 MiB and the minimum is 64 MiB. Firehose uses this
     *         value for padding calculations.
     */
    public final Integer blockSizeBytes() {
        return blockSizeBytes;
    }

    /**
     * <p>
     * The Parquet page size. Column chunks are divided into pages. A page is conceptually an indivisible unit (in terms
     * of compression and encoding). The minimum value is 64 KiB and the default is 1 MiB.
     * </p>
     * 
     * @return The Parquet page size. Column chunks are divided into pages. A page is conceptually an indivisible unit
     *         (in terms of compression and encoding). The minimum value is 64 KiB and the default is 1 MiB.
     */
    public final Integer pageSizeBytes() {
        return pageSizeBytes;
    }

    /**
     * <p>
     * The compression code to use over data blocks. The possible values are <code>UNCOMPRESSED</code>,
     * <code>SNAPPY</code>, and <code>GZIP</code>, with the default being <code>SNAPPY</code>. Use <code>SNAPPY</code>
     * for higher decompression speed. Use <code>GZIP</code> if the compression ratio is more important than speed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #compression} will
     * return {@link ParquetCompression#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #compressionAsString}.
     * </p>
     * 
     * @return The compression code to use over data blocks. The possible values are <code>UNCOMPRESSED</code>,
     *         <code>SNAPPY</code>, and <code>GZIP</code>, with the default being <code>SNAPPY</code>. Use
     *         <code>SNAPPY</code> for higher decompression speed. Use <code>GZIP</code> if the compression ratio is
     *         more important than speed.
     * @see ParquetCompression
     */
    public final ParquetCompression compression() {
        return ParquetCompression.fromValue(compression);
    }

    /**
     * <p>
     * The compression code to use over data blocks. The possible values are <code>UNCOMPRESSED</code>,
     * <code>SNAPPY</code>, and <code>GZIP</code>, with the default being <code>SNAPPY</code>. Use <code>SNAPPY</code>
     * for higher decompression speed. Use <code>GZIP</code> if the compression ratio is more important than speed.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #compression} will
     * return {@link ParquetCompression#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #compressionAsString}.
     * </p>
     * 
     * @return The compression code to use over data blocks. The possible values are <code>UNCOMPRESSED</code>,
     *         <code>SNAPPY</code>, and <code>GZIP</code>, with the default being <code>SNAPPY</code>. Use
     *         <code>SNAPPY</code> for higher decompression speed. Use <code>GZIP</code> if the compression ratio is
     *         more important than speed.
     * @see ParquetCompression
     */
    public final String compressionAsString() {
        return compression;
    }

    /**
     * <p>
     * Indicates whether to enable dictionary compression.
     * </p>
     * 
     * @return Indicates whether to enable dictionary compression.
     */
    public final Boolean enableDictionaryCompression() {
        return enableDictionaryCompression;
    }

    /**
     * <p>
     * The maximum amount of padding to apply. This is useful if you intend to copy the data from Amazon S3 to HDFS
     * before querying. The default is 0.
     * </p>
     * 
     * @return The maximum amount of padding to apply. This is useful if you intend to copy the data from Amazon S3 to
     *         HDFS before querying. The default is 0.
     */
    public final Integer maxPaddingBytes() {
        return maxPaddingBytes;
    }

    /**
     * <p>
     * Indicates the version of row format to output. The possible values are <code>V1</code> and <code>V2</code>. The
     * default is <code>V1</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #writerVersion}
     * will return {@link ParquetWriterVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #writerVersionAsString}.
     * </p>
     * 
     * @return Indicates the version of row format to output. The possible values are <code>V1</code> and
     *         <code>V2</code>. The default is <code>V1</code>.
     * @see ParquetWriterVersion
     */
    public final ParquetWriterVersion writerVersion() {
        return ParquetWriterVersion.fromValue(writerVersion);
    }

    /**
     * <p>
     * Indicates the version of row format to output. The possible values are <code>V1</code> and <code>V2</code>. The
     * default is <code>V1</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #writerVersion}
     * will return {@link ParquetWriterVersion#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #writerVersionAsString}.
     * </p>
     * 
     * @return Indicates the version of row format to output. The possible values are <code>V1</code> and
     *         <code>V2</code>. The default is <code>V1</code>.
     * @see ParquetWriterVersion
     */
    public final String writerVersionAsString() {
        return writerVersion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(blockSizeBytes());
        hashCode = 31 * hashCode + Objects.hashCode(pageSizeBytes());
        hashCode = 31 * hashCode + Objects.hashCode(compressionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(enableDictionaryCompression());
        hashCode = 31 * hashCode + Objects.hashCode(maxPaddingBytes());
        hashCode = 31 * hashCode + Objects.hashCode(writerVersionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ParquetSerDe)) {
            return false;
        }
        ParquetSerDe other = (ParquetSerDe) obj;
        return Objects.equals(blockSizeBytes(), other.blockSizeBytes()) && Objects.equals(pageSizeBytes(), other.pageSizeBytes())
                && Objects.equals(compressionAsString(), other.compressionAsString())
                && Objects.equals(enableDictionaryCompression(), other.enableDictionaryCompression())
                && Objects.equals(maxPaddingBytes(), other.maxPaddingBytes())
                && Objects.equals(writerVersionAsString(), other.writerVersionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ParquetSerDe").add("BlockSizeBytes", blockSizeBytes()).add("PageSizeBytes", pageSizeBytes())
                .add("Compression", compressionAsString()).add("EnableDictionaryCompression", enableDictionaryCompression())
                .add("MaxPaddingBytes", maxPaddingBytes()).add("WriterVersion", writerVersionAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BlockSizeBytes":
            return Optional.ofNullable(clazz.cast(blockSizeBytes()));
        case "PageSizeBytes":
            return Optional.ofNullable(clazz.cast(pageSizeBytes()));
        case "Compression":
            return Optional.ofNullable(clazz.cast(compressionAsString()));
        case "EnableDictionaryCompression":
            return Optional.ofNullable(clazz.cast(enableDictionaryCompression()));
        case "MaxPaddingBytes":
            return Optional.ofNullable(clazz.cast(maxPaddingBytes()));
        case "WriterVersion":
            return Optional.ofNullable(clazz.cast(writerVersionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ParquetSerDe, T> g) {
        return obj -> g.apply((ParquetSerDe) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ParquetSerDe> {
        /**
         * <p>
         * The Hadoop Distributed File System (HDFS) block size. This is useful if you intend to copy the data from
         * Amazon S3 to HDFS before querying. The default is 256 MiB and the minimum is 64 MiB. Firehose uses this value
         * for padding calculations.
         * </p>
         * 
         * @param blockSizeBytes
         *        The Hadoop Distributed File System (HDFS) block size. This is useful if you intend to copy the data
         *        from Amazon S3 to HDFS before querying. The default is 256 MiB and the minimum is 64 MiB. Firehose
         *        uses this value for padding calculations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blockSizeBytes(Integer blockSizeBytes);

        /**
         * <p>
         * The Parquet page size. Column chunks are divided into pages. A page is conceptually an indivisible unit (in
         * terms of compression and encoding). The minimum value is 64 KiB and the default is 1 MiB.
         * </p>
         * 
         * @param pageSizeBytes
         *        The Parquet page size. Column chunks are divided into pages. A page is conceptually an indivisible
         *        unit (in terms of compression and encoding). The minimum value is 64 KiB and the default is 1 MiB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pageSizeBytes(Integer pageSizeBytes);

        /**
         * <p>
         * The compression code to use over data blocks. The possible values are <code>UNCOMPRESSED</code>,
         * <code>SNAPPY</code>, and <code>GZIP</code>, with the default being <code>SNAPPY</code>. Use
         * <code>SNAPPY</code> for higher decompression speed. Use <code>GZIP</code> if the compression ratio is more
         * important than speed.
         * </p>
         * 
         * @param compression
         *        The compression code to use over data blocks. The possible values are <code>UNCOMPRESSED</code>,
         *        <code>SNAPPY</code>, and <code>GZIP</code>, with the default being <code>SNAPPY</code>. Use
         *        <code>SNAPPY</code> for higher decompression speed. Use <code>GZIP</code> if the compression ratio is
         *        more important than speed.
         * @see ParquetCompression
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParquetCompression
         */
        Builder compression(String compression);

        /**
         * <p>
         * The compression code to use over data blocks. The possible values are <code>UNCOMPRESSED</code>,
         * <code>SNAPPY</code>, and <code>GZIP</code>, with the default being <code>SNAPPY</code>. Use
         * <code>SNAPPY</code> for higher decompression speed. Use <code>GZIP</code> if the compression ratio is more
         * important than speed.
         * </p>
         * 
         * @param compression
         *        The compression code to use over data blocks. The possible values are <code>UNCOMPRESSED</code>,
         *        <code>SNAPPY</code>, and <code>GZIP</code>, with the default being <code>SNAPPY</code>. Use
         *        <code>SNAPPY</code> for higher decompression speed. Use <code>GZIP</code> if the compression ratio is
         *        more important than speed.
         * @see ParquetCompression
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParquetCompression
         */
        Builder compression(ParquetCompression compression);

        /**
         * <p>
         * Indicates whether to enable dictionary compression.
         * </p>
         * 
         * @param enableDictionaryCompression
         *        Indicates whether to enable dictionary compression.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableDictionaryCompression(Boolean enableDictionaryCompression);

        /**
         * <p>
         * The maximum amount of padding to apply. This is useful if you intend to copy the data from Amazon S3 to HDFS
         * before querying. The default is 0.
         * </p>
         * 
         * @param maxPaddingBytes
         *        The maximum amount of padding to apply. This is useful if you intend to copy the data from Amazon S3
         *        to HDFS before querying. The default is 0.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxPaddingBytes(Integer maxPaddingBytes);

        /**
         * <p>
         * Indicates the version of row format to output. The possible values are <code>V1</code> and <code>V2</code>.
         * The default is <code>V1</code>.
         * </p>
         * 
         * @param writerVersion
         *        Indicates the version of row format to output. The possible values are <code>V1</code> and
         *        <code>V2</code>. The default is <code>V1</code>.
         * @see ParquetWriterVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParquetWriterVersion
         */
        Builder writerVersion(String writerVersion);

        /**
         * <p>
         * Indicates the version of row format to output. The possible values are <code>V1</code> and <code>V2</code>.
         * The default is <code>V1</code>.
         * </p>
         * 
         * @param writerVersion
         *        Indicates the version of row format to output. The possible values are <code>V1</code> and
         *        <code>V2</code>. The default is <code>V1</code>.
         * @see ParquetWriterVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ParquetWriterVersion
         */
        Builder writerVersion(ParquetWriterVersion writerVersion);
    }

    static final class BuilderImpl implements Builder {
        private Integer blockSizeBytes;

        private Integer pageSizeBytes;

        private String compression;

        private Boolean enableDictionaryCompression;

        private Integer maxPaddingBytes;

        private String writerVersion;

        private BuilderImpl() {
        }

        private BuilderImpl(ParquetSerDe model) {
            blockSizeBytes(model.blockSizeBytes);
            pageSizeBytes(model.pageSizeBytes);
            compression(model.compression);
            enableDictionaryCompression(model.enableDictionaryCompression);
            maxPaddingBytes(model.maxPaddingBytes);
            writerVersion(model.writerVersion);
        }

        public final Integer getBlockSizeBytes() {
            return blockSizeBytes;
        }

        public final void setBlockSizeBytes(Integer blockSizeBytes) {
            this.blockSizeBytes = blockSizeBytes;
        }

        @Override
        public final Builder blockSizeBytes(Integer blockSizeBytes) {
            this.blockSizeBytes = blockSizeBytes;
            return this;
        }

        public final Integer getPageSizeBytes() {
            return pageSizeBytes;
        }

        public final void setPageSizeBytes(Integer pageSizeBytes) {
            this.pageSizeBytes = pageSizeBytes;
        }

        @Override
        public final Builder pageSizeBytes(Integer pageSizeBytes) {
            this.pageSizeBytes = pageSizeBytes;
            return this;
        }

        public final String getCompression() {
            return compression;
        }

        public final void setCompression(String compression) {
            this.compression = compression;
        }

        @Override
        public final Builder compression(String compression) {
            this.compression = compression;
            return this;
        }

        @Override
        public final Builder compression(ParquetCompression compression) {
            this.compression(compression == null ? null : compression.toString());
            return this;
        }

        public final Boolean getEnableDictionaryCompression() {
            return enableDictionaryCompression;
        }

        public final void setEnableDictionaryCompression(Boolean enableDictionaryCompression) {
            this.enableDictionaryCompression = enableDictionaryCompression;
        }

        @Override
        public final Builder enableDictionaryCompression(Boolean enableDictionaryCompression) {
            this.enableDictionaryCompression = enableDictionaryCompression;
            return this;
        }

        public final Integer getMaxPaddingBytes() {
            return maxPaddingBytes;
        }

        public final void setMaxPaddingBytes(Integer maxPaddingBytes) {
            this.maxPaddingBytes = maxPaddingBytes;
        }

        @Override
        public final Builder maxPaddingBytes(Integer maxPaddingBytes) {
            this.maxPaddingBytes = maxPaddingBytes;
            return this;
        }

        public final String getWriterVersion() {
            return writerVersion;
        }

        public final void setWriterVersion(String writerVersion) {
            this.writerVersion = writerVersion;
        }

        @Override
        public final Builder writerVersion(String writerVersion) {
            this.writerVersion = writerVersion;
            return this;
        }

        @Override
        public final Builder writerVersion(ParquetWriterVersion writerVersion) {
            this.writerVersion(writerVersion == null ? null : writerVersion.toString());
            return this;
        }

        @Override
        public ParquetSerDe build() {
            return new ParquetSerDe(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
