/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies a transform that combines the rows from two or more datasets into a single result.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Union implements SdkPojo, Serializable, ToCopyableBuilder<Union.Builder, Union> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Union::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<List<String>> INPUTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Inputs")
            .getter(getter(Union::inputs))
            .setter(setter(Builder::inputs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Inputs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> UNION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UnionType").getter(getter(Union::unionTypeAsString)).setter(setter(Builder::unionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UnionType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, INPUTS_FIELD,
            UNION_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Name", NAME_FIELD);
                    put("Inputs", INPUTS_FIELD);
                    put("UnionType", UNION_TYPE_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<String> inputs;

    private final String unionType;

    private Union(BuilderImpl builder) {
        this.name = builder.name;
        this.inputs = builder.inputs;
        this.unionType = builder.unionType;
    }

    /**
     * <p>
     * The name of the transform node.
     * </p>
     * 
     * @return The name of the transform node.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Inputs property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasInputs() {
        return inputs != null && !(inputs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The node ID inputs to the transform.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInputs} method.
     * </p>
     * 
     * @return The node ID inputs to the transform.
     */
    public final List<String> inputs() {
        return inputs;
    }

    /**
     * <p>
     * Indicates the type of Union transform.
     * </p>
     * <p>
     * Specify <code>ALL</code> to join all rows from data sources to the resulting DynamicFrame. The resulting union
     * does not remove duplicate rows.
     * </p>
     * <p>
     * Specify <code>DISTINCT</code> to remove duplicate rows in the resulting DynamicFrame.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unionType} will
     * return {@link UnionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unionTypeAsString}.
     * </p>
     * 
     * @return Indicates the type of Union transform. </p>
     *         <p>
     *         Specify <code>ALL</code> to join all rows from data sources to the resulting DynamicFrame. The resulting
     *         union does not remove duplicate rows.
     *         </p>
     *         <p>
     *         Specify <code>DISTINCT</code> to remove duplicate rows in the resulting DynamicFrame.
     * @see UnionType
     */
    public final UnionType unionType() {
        return UnionType.fromValue(unionType);
    }

    /**
     * <p>
     * Indicates the type of Union transform.
     * </p>
     * <p>
     * Specify <code>ALL</code> to join all rows from data sources to the resulting DynamicFrame. The resulting union
     * does not remove duplicate rows.
     * </p>
     * <p>
     * Specify <code>DISTINCT</code> to remove duplicate rows in the resulting DynamicFrame.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unionType} will
     * return {@link UnionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unionTypeAsString}.
     * </p>
     * 
     * @return Indicates the type of Union transform. </p>
     *         <p>
     *         Specify <code>ALL</code> to join all rows from data sources to the resulting DynamicFrame. The resulting
     *         union does not remove duplicate rows.
     *         </p>
     *         <p>
     *         Specify <code>DISTINCT</code> to remove duplicate rows in the resulting DynamicFrame.
     * @see UnionType
     */
    public final String unionTypeAsString() {
        return unionType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasInputs() ? inputs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(unionTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Union)) {
            return false;
        }
        Union other = (Union) obj;
        return Objects.equals(name(), other.name()) && hasInputs() == other.hasInputs()
                && Objects.equals(inputs(), other.inputs()) && Objects.equals(unionTypeAsString(), other.unionTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Union").add("Name", name()).add("Inputs", hasInputs() ? inputs() : null)
                .add("UnionType", unionTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Inputs":
            return Optional.ofNullable(clazz.cast(inputs()));
        case "UnionType":
            return Optional.ofNullable(clazz.cast(unionTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<Union, T> g) {
        return obj -> g.apply((Union) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Union> {
        /**
         * <p>
         * The name of the transform node.
         * </p>
         * 
         * @param name
         *        The name of the transform node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The node ID inputs to the transform.
         * </p>
         * 
         * @param inputs
         *        The node ID inputs to the transform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputs(Collection<String> inputs);

        /**
         * <p>
         * The node ID inputs to the transform.
         * </p>
         * 
         * @param inputs
         *        The node ID inputs to the transform.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputs(String... inputs);

        /**
         * <p>
         * Indicates the type of Union transform.
         * </p>
         * <p>
         * Specify <code>ALL</code> to join all rows from data sources to the resulting DynamicFrame. The resulting
         * union does not remove duplicate rows.
         * </p>
         * <p>
         * Specify <code>DISTINCT</code> to remove duplicate rows in the resulting DynamicFrame.
         * </p>
         * 
         * @param unionType
         *        Indicates the type of Union transform. </p>
         *        <p>
         *        Specify <code>ALL</code> to join all rows from data sources to the resulting DynamicFrame. The
         *        resulting union does not remove duplicate rows.
         *        </p>
         *        <p>
         *        Specify <code>DISTINCT</code> to remove duplicate rows in the resulting DynamicFrame.
         * @see UnionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UnionType
         */
        Builder unionType(String unionType);

        /**
         * <p>
         * Indicates the type of Union transform.
         * </p>
         * <p>
         * Specify <code>ALL</code> to join all rows from data sources to the resulting DynamicFrame. The resulting
         * union does not remove duplicate rows.
         * </p>
         * <p>
         * Specify <code>DISTINCT</code> to remove duplicate rows in the resulting DynamicFrame.
         * </p>
         * 
         * @param unionType
         *        Indicates the type of Union transform. </p>
         *        <p>
         *        Specify <code>ALL</code> to join all rows from data sources to the resulting DynamicFrame. The
         *        resulting union does not remove duplicate rows.
         *        </p>
         *        <p>
         *        Specify <code>DISTINCT</code> to remove duplicate rows in the resulting DynamicFrame.
         * @see UnionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UnionType
         */
        Builder unionType(UnionType unionType);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<String> inputs = DefaultSdkAutoConstructList.getInstance();

        private String unionType;

        private BuilderImpl() {
        }

        private BuilderImpl(Union model) {
            name(model.name);
            inputs(model.inputs);
            unionType(model.unionType);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Collection<String> getInputs() {
            if (inputs instanceof SdkAutoConstructList) {
                return null;
            }
            return inputs;
        }

        public final void setInputs(Collection<String> inputs) {
            this.inputs = TwoInputsCopier.copy(inputs);
        }

        @Override
        public final Builder inputs(Collection<String> inputs) {
            this.inputs = TwoInputsCopier.copy(inputs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inputs(String... inputs) {
            inputs(Arrays.asList(inputs));
            return this;
        }

        public final String getUnionType() {
            return unionType;
        }

        public final void setUnionType(String unionType) {
            this.unionType = unionType;
        }

        @Override
        public final Builder unionType(String unionType) {
            this.unionType = unionType;
            return this;
        }

        @Override
        public final Builder unionType(UnionType unionType) {
            this.unionType(unionType == null ? null : unionType.toString());
            return this;
        }

        @Override
        public Union build() {
            return new Union(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
