/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kafka.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 *
 * <p>
 * Contains source Kafka versions and compatible target Kafka versions.
 * </p>
 * 
 */
@Generated("software.amazon.awssdk:codegen")
public final class CompatibleKafkaVersion implements SdkPojo, Serializable,
        ToCopyableBuilder<CompatibleKafkaVersion.Builder, CompatibleKafkaVersion> {
    private static final SdkField<String> SOURCE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceVersion").getter(getter(CompatibleKafkaVersion::sourceVersion))
            .setter(setter(Builder::sourceVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sourceVersion").build()).build();

    private static final SdkField<List<String>> TARGET_VERSIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("TargetVersions")
            .getter(getter(CompatibleKafkaVersion::targetVersions))
            .setter(setter(Builder::targetVersions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("targetVersions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_VERSION_FIELD,
            TARGET_VERSIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String sourceVersion;

    private final List<String> targetVersions;

    private CompatibleKafkaVersion(BuilderImpl builder) {
        this.sourceVersion = builder.sourceVersion;
        this.targetVersions = builder.targetVersions;
    }

    /**
     *
     * <p>
     * A Kafka version.
     * </p>
     * 
     * @return <p>
     *         A Kafka version.
     *         </p>
     */
    public final String sourceVersion() {
        return sourceVersion;
    }

    /**
     * Returns true if the TargetVersions property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTargetVersions() {
        return targetVersions != null && !(targetVersions instanceof SdkAutoConstructList);
    }

    /**
     *
     * <p>
     * A list of Kafka versions.
     * </p>
     * 
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTargetVersions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return <p>
     *         A list of Kafka versions.
     *         </p>
     */
    public final List<String> targetVersions() {
        return targetVersions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sourceVersion());
        hashCode = 31 * hashCode + Objects.hashCode(hasTargetVersions() ? targetVersions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CompatibleKafkaVersion)) {
            return false;
        }
        CompatibleKafkaVersion other = (CompatibleKafkaVersion) obj;
        return Objects.equals(sourceVersion(), other.sourceVersion()) && hasTargetVersions() == other.hasTargetVersions()
                && Objects.equals(targetVersions(), other.targetVersions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CompatibleKafkaVersion").add("SourceVersion", sourceVersion())
                .add("TargetVersions", hasTargetVersions() ? targetVersions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SourceVersion":
            return Optional.ofNullable(clazz.cast(sourceVersion()));
        case "TargetVersions":
            return Optional.ofNullable(clazz.cast(targetVersions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CompatibleKafkaVersion, T> g) {
        return obj -> g.apply((CompatibleKafkaVersion) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CompatibleKafkaVersion> {
        /**
         *
         * <p>
         * A Kafka version.
         * </p>
         * 
         * @param sourceVersion
         *        <p>
         *        A Kafka version.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceVersion(String sourceVersion);

        /**
         *
         * <p>
         * A list of Kafka versions.
         * </p>
         * 
         * @param targetVersions
         *        <p>
         *        A list of Kafka versions.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetVersions(Collection<String> targetVersions);

        /**
         *
         * <p>
         * A list of Kafka versions.
         * </p>
         * 
         * @param targetVersions
         *        <p>
         *        A list of Kafka versions.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetVersions(String... targetVersions);
    }

    static final class BuilderImpl implements Builder {
        private String sourceVersion;

        private List<String> targetVersions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CompatibleKafkaVersion model) {
            sourceVersion(model.sourceVersion);
            targetVersions(model.targetVersions);
        }

        public final String getSourceVersion() {
            return sourceVersion;
        }

        @Override
        public final Builder sourceVersion(String sourceVersion) {
            this.sourceVersion = sourceVersion;
            return this;
        }

        public final void setSourceVersion(String sourceVersion) {
            this.sourceVersion = sourceVersion;
        }

        public final Collection<String> getTargetVersions() {
            if (targetVersions instanceof SdkAutoConstructList) {
                return null;
            }
            return targetVersions;
        }

        @Override
        public final Builder targetVersions(Collection<String> targetVersions) {
            this.targetVersions = ___listOf__stringCopier.copy(targetVersions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder targetVersions(String... targetVersions) {
            targetVersions(Arrays.asList(targetVersions));
            return this;
        }

        public final void setTargetVersions(Collection<String> targetVersions) {
            this.targetVersions = ___listOf__stringCopier.copy(targetVersions);
        }

        @Override
        public CompatibleKafkaVersion build() {
            return new CompatibleKafkaVersion(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
