/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kafka.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.kafka.KafkaClient;
import software.amazon.awssdk.services.kafka.internal.UserAgentUtils;
import software.amazon.awssdk.services.kafka.model.ClusterOperationV2Summary;
import software.amazon.awssdk.services.kafka.model.ListClusterOperationsV2Request;
import software.amazon.awssdk.services.kafka.model.ListClusterOperationsV2Response;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.kafka.KafkaClient#listClusterOperationsV2Paginator(software.amazon.awssdk.services.kafka.model.ListClusterOperationsV2Request)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.kafka.model.ListClusterOperationsV2Response} that can be used to iterate
 * through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.kafka.paginators.ListClusterOperationsV2Iterable responses = client.listClusterOperationsV2Paginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.kafka.paginators.ListClusterOperationsV2Iterable responses = client
 *             .listClusterOperationsV2Paginator(request);
 *     for (software.amazon.awssdk.services.kafka.model.ListClusterOperationsV2Response response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.kafka.paginators.ListClusterOperationsV2Iterable responses = client.listClusterOperationsV2Paginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the paginator.
 * It only limits the number of results in each page.</b>
 * </p>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listClusterOperationsV2(software.amazon.awssdk.services.kafka.model.ListClusterOperationsV2Request)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListClusterOperationsV2Iterable implements SdkIterable<ListClusterOperationsV2Response> {
    private final KafkaClient client;

    private final ListClusterOperationsV2Request firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListClusterOperationsV2Iterable(KafkaClient client, ListClusterOperationsV2Request firstRequest) {
        this.client = client;
        this.firstRequest = UserAgentUtils.applyPaginatorUserAgent(firstRequest);
        this.nextPageFetcher = new ListClusterOperationsV2ResponseFetcher();
    }

    @Override
    public Iterator<ListClusterOperationsV2Response> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    /**
     * Returns an iterable to iterate through the paginated
     * {@link ListClusterOperationsV2Response#clusterOperationInfoList()} member. The returned iterable is used to
     * iterate through the results across all response pages and not a single page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public final SdkIterable<ClusterOperationV2Summary> clusterOperationInfoList() {
        Function<ListClusterOperationsV2Response, Iterator<ClusterOperationV2Summary>> getIterator = response -> {
            if (response != null && response.clusterOperationInfoList() != null) {
                return response.clusterOperationInfoList().iterator();
            }
            return Collections.emptyIterator();
        };
        return PaginatedItemsIterable.<ListClusterOperationsV2Response, ClusterOperationV2Summary> builder().pagesIterable(this)
                .itemIteratorFunction(getIterator).build();
    }

    private class ListClusterOperationsV2ResponseFetcher implements SyncPageFetcher<ListClusterOperationsV2Response> {
        @Override
        public boolean hasNextPage(ListClusterOperationsV2Response previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public ListClusterOperationsV2Response nextPage(ListClusterOperationsV2Response previousPage) {
            if (previousPage == null) {
                return client.listClusterOperationsV2(firstRequest);
            }
            return client.listClusterOperationsV2(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
