/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.kms.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateKeyRequest extends KmsRequest implements ToCopyableBuilder<CreateKeyRequest.Builder, CreateKeyRequest> {
    private static final SdkField<String> POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Policy")
            .getter(getter(CreateKeyRequest::policy)).setter(setter(Builder::policy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Policy").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(CreateKeyRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> KEY_USAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KeyUsage").getter(getter(CreateKeyRequest::keyUsageAsString)).setter(setter(Builder::keyUsage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyUsage").build()).build();

    private static final SdkField<String> CUSTOMER_MASTER_KEY_SPEC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomerMasterKeySpec").getter(getter(CreateKeyRequest::customerMasterKeySpecAsString))
            .setter(setter(Builder::customerMasterKeySpec))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerMasterKeySpec").build())
            .build();

    private static final SdkField<String> ORIGIN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Origin")
            .getter(getter(CreateKeyRequest::originAsString)).setter(setter(Builder::origin))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Origin").build()).build();

    private static final SdkField<String> CUSTOM_KEY_STORE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomKeyStoreId").getter(getter(CreateKeyRequest::customKeyStoreId))
            .setter(setter(Builder::customKeyStoreId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomKeyStoreId").build()).build();

    private static final SdkField<Boolean> BYPASS_POLICY_LOCKOUT_SAFETY_CHECK_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("BypassPolicyLockoutSafetyCheck")
            .getter(getter(CreateKeyRequest::bypassPolicyLockoutSafetyCheck))
            .setter(setter(Builder::bypassPolicyLockoutSafetyCheck))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BypassPolicyLockoutSafetyCheck")
                    .build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateKeyRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> MULTI_REGION_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("MultiRegion").getter(getter(CreateKeyRequest::multiRegion)).setter(setter(Builder::multiRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MultiRegion").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(POLICY_FIELD,
            DESCRIPTION_FIELD, KEY_USAGE_FIELD, CUSTOMER_MASTER_KEY_SPEC_FIELD, ORIGIN_FIELD, CUSTOM_KEY_STORE_ID_FIELD,
            BYPASS_POLICY_LOCKOUT_SAFETY_CHECK_FIELD, TAGS_FIELD, MULTI_REGION_FIELD));

    private final String policy;

    private final String description;

    private final String keyUsage;

    private final String customerMasterKeySpec;

    private final String origin;

    private final String customKeyStoreId;

    private final Boolean bypassPolicyLockoutSafetyCheck;

    private final List<Tag> tags;

    private final Boolean multiRegion;

    private CreateKeyRequest(BuilderImpl builder) {
        super(builder);
        this.policy = builder.policy;
        this.description = builder.description;
        this.keyUsage = builder.keyUsage;
        this.customerMasterKeySpec = builder.customerMasterKeySpec;
        this.origin = builder.origin;
        this.customKeyStoreId = builder.customKeyStoreId;
        this.bypassPolicyLockoutSafetyCheck = builder.bypassPolicyLockoutSafetyCheck;
        this.tags = builder.tags;
        this.multiRegion = builder.multiRegion;
    }

    /**
     * <p>
     * The key policy to attach to the CMK.
     * </p>
     * <p>
     * If you provide a key policy, it must meet the following criteria:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you don't set <code>BypassPolicyLockoutSafetyCheck</code> to true, the key policy must allow the principal
     * that is making the <code>CreateKey</code> request to make a subsequent <a>PutKeyPolicy</a> request on the CMK.
     * This reduces the risk that the CMK becomes unmanageable. For more information, refer to the scenario in the <a
     * href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
     * >Default Key Policy</a> section of the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Each statement in the key policy must contain one or more principals. The principals in the key policy must exist
     * and be visible to AWS KMS. When you create a new AWS principal (for example, an IAM user or role), you might need
     * to enforce a delay before including the new principal in a key policy because the new principal might not be
     * immediately visible to AWS KMS. For more information, see <a href=
     * "https://docs.aws.amazon.com/IAM/latest/UserGuide/troubleshoot_general.html#troubleshoot_general_eventual-consistency"
     * >Changes that I make are not always immediately visible</a> in the <i>AWS Identity and Access Management User
     * Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you do not provide a key policy, AWS KMS attaches a default key policy to the CMK. For more information, see
     * <a href="https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default">Default Key
     * Policy</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * The key policy size quota is 32 kilobytes (32768 bytes).
     * </p>
     * <p>
     * For help writing and formatting a JSON policy document, see the <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies.html">IAM JSON Policy Reference</a> in
     * the <i> <i>IAM User Guide</i> </i>.
     * </p>
     * 
     * @return The key policy to attach to the CMK.</p>
     *         <p>
     *         If you provide a key policy, it must meet the following criteria:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If you don't set <code>BypassPolicyLockoutSafetyCheck</code> to true, the key policy must allow the
     *         principal that is making the <code>CreateKey</code> request to make a subsequent <a>PutKeyPolicy</a>
     *         request on the CMK. This reduces the risk that the CMK becomes unmanageable. For more information, refer
     *         to the scenario in the <a href=
     *         "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
     *         >Default Key Policy</a> section of the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Each statement in the key policy must contain one or more principals. The principals in the key policy
     *         must exist and be visible to AWS KMS. When you create a new AWS principal (for example, an IAM user or
     *         role), you might need to enforce a delay before including the new principal in a key policy because the
     *         new principal might not be immediately visible to AWS KMS. For more information, see <a href=
     *         "https://docs.aws.amazon.com/IAM/latest/UserGuide/troubleshoot_general.html#troubleshoot_general_eventual-consistency"
     *         >Changes that I make are not always immediately visible</a> in the <i>AWS Identity and Access Management
     *         User Guide</i>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you do not provide a key policy, AWS KMS attaches a default key policy to the CMK. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default">Default
     *         Key Policy</a> in the <i>AWS Key Management Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         The key policy size quota is 32 kilobytes (32768 bytes).
     *         </p>
     *         <p>
     *         For help writing and formatting a JSON policy document, see the <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies.html">IAM JSON Policy
     *         Reference</a> in the <i> <i>IAM User Guide</i> </i>.
     */
    public final String policy() {
        return policy;
    }

    /**
     * <p>
     * A description of the CMK.
     * </p>
     * <p>
     * Use a description that helps you decide whether the CMK is appropriate for a task. The default value is an empty
     * string (no description).
     * </p>
     * 
     * @return A description of the CMK.</p>
     *         <p>
     *         Use a description that helps you decide whether the CMK is appropriate for a task. The default value is
     *         an empty string (no description).
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * Determines the <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations">cryptographic
     * operations</a> for which you can use the CMK. The default value is <code>ENCRYPT_DECRYPT</code>. This parameter
     * is required only for asymmetric CMKs. You can't change the <code>KeyUsage</code> value after the CMK is created.
     * </p>
     * <p>
     * Select only one valid value.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For symmetric CMKs, omit the parameter or specify <code>ENCRYPT_DECRYPT</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For asymmetric CMKs with RSA key material, specify <code>ENCRYPT_DECRYPT</code> or <code>SIGN_VERIFY</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For asymmetric CMKs with ECC key material, specify <code>SIGN_VERIFY</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyUsage} will
     * return {@link KeyUsageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyUsageAsString}.
     * </p>
     * 
     * @return Determines the <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations"
     *         >cryptographic operations</a> for which you can use the CMK. The default value is
     *         <code>ENCRYPT_DECRYPT</code>. This parameter is required only for asymmetric CMKs. You can't change the
     *         <code>KeyUsage</code> value after the CMK is created.</p>
     *         <p>
     *         Select only one valid value.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For symmetric CMKs, omit the parameter or specify <code>ENCRYPT_DECRYPT</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For asymmetric CMKs with RSA key material, specify <code>ENCRYPT_DECRYPT</code> or
     *         <code>SIGN_VERIFY</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For asymmetric CMKs with ECC key material, specify <code>SIGN_VERIFY</code>.
     *         </p>
     *         </li>
     * @see KeyUsageType
     */
    public final KeyUsageType keyUsage() {
        return KeyUsageType.fromValue(keyUsage);
    }

    /**
     * <p>
     * Determines the <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations">cryptographic
     * operations</a> for which you can use the CMK. The default value is <code>ENCRYPT_DECRYPT</code>. This parameter
     * is required only for asymmetric CMKs. You can't change the <code>KeyUsage</code> value after the CMK is created.
     * </p>
     * <p>
     * Select only one valid value.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For symmetric CMKs, omit the parameter or specify <code>ENCRYPT_DECRYPT</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For asymmetric CMKs with RSA key material, specify <code>ENCRYPT_DECRYPT</code> or <code>SIGN_VERIFY</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For asymmetric CMKs with ECC key material, specify <code>SIGN_VERIFY</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyUsage} will
     * return {@link KeyUsageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyUsageAsString}.
     * </p>
     * 
     * @return Determines the <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations"
     *         >cryptographic operations</a> for which you can use the CMK. The default value is
     *         <code>ENCRYPT_DECRYPT</code>. This parameter is required only for asymmetric CMKs. You can't change the
     *         <code>KeyUsage</code> value after the CMK is created.</p>
     *         <p>
     *         Select only one valid value.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For symmetric CMKs, omit the parameter or specify <code>ENCRYPT_DECRYPT</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For asymmetric CMKs with RSA key material, specify <code>ENCRYPT_DECRYPT</code> or
     *         <code>SIGN_VERIFY</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For asymmetric CMKs with ECC key material, specify <code>SIGN_VERIFY</code>.
     *         </p>
     *         </li>
     * @see KeyUsageType
     */
    public final String keyUsageAsString() {
        return keyUsage;
    }

    /**
     * <p>
     * Specifies the type of CMK to create. The default value, <code>SYMMETRIC_DEFAULT</code>, creates a CMK with a
     * 256-bit symmetric key for encryption and decryption. For help choosing a key spec for your CMK, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/symm-asymm-choose.html">How to Choose Your CMK
     * Configuration</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * The <code>CustomerMasterKeySpec</code> determines whether the CMK contains a symmetric key or an asymmetric key
     * pair. It also determines the encryption algorithms or signing algorithms that the CMK supports. You can't change
     * the <code>CustomerMasterKeySpec</code> after the CMK is created. To further restrict the algorithms that can be
     * used with the CMK, use a condition key in its key policy or IAM policy. For more information, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-algorithm"
     * >kms:EncryptionAlgorithm</a> or <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-signing-algorithm"
     * >kms:Signing Algorithm</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <important>
     * <p>
     * <a href="http://aws.amazon.com/kms/features/#AWS_Service_Integration">AWS services that are integrated with AWS
     * KMS</a> use symmetric CMKs to protect your data. These services do not support asymmetric CMKs. For help
     * determining whether a CMK is symmetric or asymmetric, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/find-symm-asymm.html">Identifying Symmetric and
     * Asymmetric CMKs</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * </important>
     * <p>
     * AWS KMS supports the following key specs for CMKs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Symmetric key (default)
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SYMMETRIC_DEFAULT</code> (AES-256-GCM)
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Asymmetric RSA key pairs
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>RSA_2048</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RSA_3072</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RSA_4096</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Asymmetric NIST-recommended elliptic curve key pairs
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ECC_NIST_P256</code> (secp256r1)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ECC_NIST_P384</code> (secp384r1)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ECC_NIST_P521</code> (secp521r1)
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Other asymmetric elliptic curve key pairs
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ECC_SECG_P256K1</code> (secp256k1), commonly used for cryptocurrencies.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #customerMasterKeySpec} will return {@link CustomerMasterKeySpec#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #customerMasterKeySpecAsString}.
     * </p>
     * 
     * @return Specifies the type of CMK to create. The default value, <code>SYMMETRIC_DEFAULT</code>, creates a CMK
     *         with a 256-bit symmetric key for encryption and decryption. For help choosing a key spec for your CMK,
     *         see <a href="https://docs.aws.amazon.com/kms/latest/developerguide/symm-asymm-choose.html">How to Choose
     *         Your CMK Configuration</a> in the <i>AWS Key Management Service Developer Guide</i>.</p>
     *         <p>
     *         The <code>CustomerMasterKeySpec</code> determines whether the CMK contains a symmetric key or an
     *         asymmetric key pair. It also determines the encryption algorithms or signing algorithms that the CMK
     *         supports. You can't change the <code>CustomerMasterKeySpec</code> after the CMK is created. To further
     *         restrict the algorithms that can be used with the CMK, use a condition key in its key policy or IAM
     *         policy. For more information, see <a href=
     *         "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-algorithm"
     *         >kms:EncryptionAlgorithm</a> or <a href=
     *         "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-signing-algorithm"
     *         >kms:Signing Algorithm</a> in the <i>AWS Key Management Service Developer Guide</i>.
     *         </p>
     *         <important>
     *         <p>
     *         <a href="http://aws.amazon.com/kms/features/#AWS_Service_Integration">AWS services that are integrated
     *         with AWS KMS</a> use symmetric CMKs to protect your data. These services do not support asymmetric CMKs.
     *         For help determining whether a CMK is symmetric or asymmetric, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/find-symm-asymm.html">Identifying Symmetric
     *         and Asymmetric CMKs</a> in the <i>AWS Key Management Service Developer Guide</i>.
     *         </p>
     *         </important>
     *         <p>
     *         AWS KMS supports the following key specs for CMKs:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Symmetric key (default)
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SYMMETRIC_DEFAULT</code> (AES-256-GCM)
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Asymmetric RSA key pairs
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>RSA_2048</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RSA_3072</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RSA_4096</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Asymmetric NIST-recommended elliptic curve key pairs
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ECC_NIST_P256</code> (secp256r1)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ECC_NIST_P384</code> (secp384r1)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ECC_NIST_P521</code> (secp521r1)
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Other asymmetric elliptic curve key pairs
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ECC_SECG_P256K1</code> (secp256k1), commonly used for cryptocurrencies.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see CustomerMasterKeySpec
     */
    public final CustomerMasterKeySpec customerMasterKeySpec() {
        return CustomerMasterKeySpec.fromValue(customerMasterKeySpec);
    }

    /**
     * <p>
     * Specifies the type of CMK to create. The default value, <code>SYMMETRIC_DEFAULT</code>, creates a CMK with a
     * 256-bit symmetric key for encryption and decryption. For help choosing a key spec for your CMK, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/symm-asymm-choose.html">How to Choose Your CMK
     * Configuration</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * The <code>CustomerMasterKeySpec</code> determines whether the CMK contains a symmetric key or an asymmetric key
     * pair. It also determines the encryption algorithms or signing algorithms that the CMK supports. You can't change
     * the <code>CustomerMasterKeySpec</code> after the CMK is created. To further restrict the algorithms that can be
     * used with the CMK, use a condition key in its key policy or IAM policy. For more information, see <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-algorithm"
     * >kms:EncryptionAlgorithm</a> or <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-signing-algorithm"
     * >kms:Signing Algorithm</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <important>
     * <p>
     * <a href="http://aws.amazon.com/kms/features/#AWS_Service_Integration">AWS services that are integrated with AWS
     * KMS</a> use symmetric CMKs to protect your data. These services do not support asymmetric CMKs. For help
     * determining whether a CMK is symmetric or asymmetric, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/find-symm-asymm.html">Identifying Symmetric and
     * Asymmetric CMKs</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * </important>
     * <p>
     * AWS KMS supports the following key specs for CMKs:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Symmetric key (default)
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>SYMMETRIC_DEFAULT</code> (AES-256-GCM)
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Asymmetric RSA key pairs
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>RSA_2048</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RSA_3072</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RSA_4096</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Asymmetric NIST-recommended elliptic curve key pairs
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ECC_NIST_P256</code> (secp256r1)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ECC_NIST_P384</code> (secp384r1)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ECC_NIST_P521</code> (secp521r1)
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Other asymmetric elliptic curve key pairs
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ECC_SECG_P256K1</code> (secp256k1), commonly used for cryptocurrencies.
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #customerMasterKeySpec} will return {@link CustomerMasterKeySpec#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #customerMasterKeySpecAsString}.
     * </p>
     * 
     * @return Specifies the type of CMK to create. The default value, <code>SYMMETRIC_DEFAULT</code>, creates a CMK
     *         with a 256-bit symmetric key for encryption and decryption. For help choosing a key spec for your CMK,
     *         see <a href="https://docs.aws.amazon.com/kms/latest/developerguide/symm-asymm-choose.html">How to Choose
     *         Your CMK Configuration</a> in the <i>AWS Key Management Service Developer Guide</i>.</p>
     *         <p>
     *         The <code>CustomerMasterKeySpec</code> determines whether the CMK contains a symmetric key or an
     *         asymmetric key pair. It also determines the encryption algorithms or signing algorithms that the CMK
     *         supports. You can't change the <code>CustomerMasterKeySpec</code> after the CMK is created. To further
     *         restrict the algorithms that can be used with the CMK, use a condition key in its key policy or IAM
     *         policy. For more information, see <a href=
     *         "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-algorithm"
     *         >kms:EncryptionAlgorithm</a> or <a href=
     *         "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-signing-algorithm"
     *         >kms:Signing Algorithm</a> in the <i>AWS Key Management Service Developer Guide</i>.
     *         </p>
     *         <important>
     *         <p>
     *         <a href="http://aws.amazon.com/kms/features/#AWS_Service_Integration">AWS services that are integrated
     *         with AWS KMS</a> use symmetric CMKs to protect your data. These services do not support asymmetric CMKs.
     *         For help determining whether a CMK is symmetric or asymmetric, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/find-symm-asymm.html">Identifying Symmetric
     *         and Asymmetric CMKs</a> in the <i>AWS Key Management Service Developer Guide</i>.
     *         </p>
     *         </important>
     *         <p>
     *         AWS KMS supports the following key specs for CMKs:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Symmetric key (default)
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>SYMMETRIC_DEFAULT</code> (AES-256-GCM)
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Asymmetric RSA key pairs
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>RSA_2048</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RSA_3072</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RSA_4096</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Asymmetric NIST-recommended elliptic curve key pairs
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ECC_NIST_P256</code> (secp256r1)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ECC_NIST_P384</code> (secp384r1)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ECC_NIST_P521</code> (secp521r1)
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Other asymmetric elliptic curve key pairs
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ECC_SECG_P256K1</code> (secp256k1), commonly used for cryptocurrencies.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     * @see CustomerMasterKeySpec
     */
    public final String customerMasterKeySpecAsString() {
        return customerMasterKeySpec;
    }

    /**
     * <p>
     * The source of the key material for the CMK. You cannot change the origin after you create the CMK. The default is
     * <code>AWS_KMS</code>, which means that AWS KMS creates the key material.
     * </p>
     * <p>
     * To create a CMK with no key material (for imported key material), set the value to <code>EXTERNAL</code>. For
     * more information about importing key material into AWS KMS, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key Material</a> in
     * the <i>AWS Key Management Service Developer Guide</i>. This value is valid only for symmetric CMKs.
     * </p>
     * <p>
     * To create a CMK in an AWS KMS <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key store</a>
     * and create its key material in the associated AWS CloudHSM cluster, set this value to <code>AWS_CLOUDHSM</code>.
     * You must also use the <code>CustomKeyStoreId</code> parameter to identify the custom key store. This value is
     * valid only for symmetric CMKs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #origin} will
     * return {@link OriginType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #originAsString}.
     * </p>
     * 
     * @return The source of the key material for the CMK. You cannot change the origin after you create the CMK. The
     *         default is <code>AWS_KMS</code>, which means that AWS KMS creates the key material.</p>
     *         <p>
     *         To create a CMK with no key material (for imported key material), set the value to <code>EXTERNAL</code>.
     *         For more information about importing key material into AWS KMS, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key
     *         Material</a> in the <i>AWS Key Management Service Developer Guide</i>. This value is valid only for
     *         symmetric CMKs.
     *         </p>
     *         <p>
     *         To create a CMK in an AWS KMS <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
     *         store</a> and create its key material in the associated AWS CloudHSM cluster, set this value to
     *         <code>AWS_CLOUDHSM</code>. You must also use the <code>CustomKeyStoreId</code> parameter to identify the
     *         custom key store. This value is valid only for symmetric CMKs.
     * @see OriginType
     */
    public final OriginType origin() {
        return OriginType.fromValue(origin);
    }

    /**
     * <p>
     * The source of the key material for the CMK. You cannot change the origin after you create the CMK. The default is
     * <code>AWS_KMS</code>, which means that AWS KMS creates the key material.
     * </p>
     * <p>
     * To create a CMK with no key material (for imported key material), set the value to <code>EXTERNAL</code>. For
     * more information about importing key material into AWS KMS, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key Material</a> in
     * the <i>AWS Key Management Service Developer Guide</i>. This value is valid only for symmetric CMKs.
     * </p>
     * <p>
     * To create a CMK in an AWS KMS <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key store</a>
     * and create its key material in the associated AWS CloudHSM cluster, set this value to <code>AWS_CLOUDHSM</code>.
     * You must also use the <code>CustomKeyStoreId</code> parameter to identify the custom key store. This value is
     * valid only for symmetric CMKs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #origin} will
     * return {@link OriginType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #originAsString}.
     * </p>
     * 
     * @return The source of the key material for the CMK. You cannot change the origin after you create the CMK. The
     *         default is <code>AWS_KMS</code>, which means that AWS KMS creates the key material.</p>
     *         <p>
     *         To create a CMK with no key material (for imported key material), set the value to <code>EXTERNAL</code>.
     *         For more information about importing key material into AWS KMS, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key
     *         Material</a> in the <i>AWS Key Management Service Developer Guide</i>. This value is valid only for
     *         symmetric CMKs.
     *         </p>
     *         <p>
     *         To create a CMK in an AWS KMS <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
     *         store</a> and create its key material in the associated AWS CloudHSM cluster, set this value to
     *         <code>AWS_CLOUDHSM</code>. You must also use the <code>CustomKeyStoreId</code> parameter to identify the
     *         custom key store. This value is valid only for symmetric CMKs.
     * @see OriginType
     */
    public final String originAsString() {
        return origin;
    }

    /**
     * <p>
     * Creates the CMK in the specified <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key store</a>
     * and the key material in its associated AWS CloudHSM cluster. To create a CMK in a custom key store, you must also
     * specify the <code>Origin</code> parameter with a value of <code>AWS_CLOUDHSM</code>. The AWS CloudHSM cluster
     * that is associated with the custom key store must have at least two active HSMs, each in a different Availability
     * Zone in the Region.
     * </p>
     * <p>
     * This parameter is valid only for symmetric CMKs and regional CMKs. You cannot create an asymmetric CMK or a
     * multi-Region CMK in a custom key store.
     * </p>
     * <p>
     * To find the ID of a custom key store, use the <a>DescribeCustomKeyStores</a> operation.
     * </p>
     * <p>
     * The response includes the custom key store ID and the ID of the AWS CloudHSM cluster.
     * </p>
     * <p>
     * This operation is part of the <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">Custom Key Store
     * feature</a> feature in AWS KMS, which combines the convenience and extensive integration of AWS KMS with the
     * isolation and control of a single-tenant key store.
     * </p>
     * 
     * @return Creates the CMK in the specified <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
     *         store</a> and the key material in its associated AWS CloudHSM cluster. To create a CMK in a custom key
     *         store, you must also specify the <code>Origin</code> parameter with a value of <code>AWS_CLOUDHSM</code>.
     *         The AWS CloudHSM cluster that is associated with the custom key store must have at least two active HSMs,
     *         each in a different Availability Zone in the Region.</p>
     *         <p>
     *         This parameter is valid only for symmetric CMKs and regional CMKs. You cannot create an asymmetric CMK or
     *         a multi-Region CMK in a custom key store.
     *         </p>
     *         <p>
     *         To find the ID of a custom key store, use the <a>DescribeCustomKeyStores</a> operation.
     *         </p>
     *         <p>
     *         The response includes the custom key store ID and the ID of the AWS CloudHSM cluster.
     *         </p>
     *         <p>
     *         This operation is part of the <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">Custom Key
     *         Store feature</a> feature in AWS KMS, which combines the convenience and extensive integration of AWS KMS
     *         with the isolation and control of a single-tenant key store.
     */
    public final String customKeyStoreId() {
        return customKeyStoreId;
    }

    /**
     * <p>
     * A flag to indicate whether to bypass the key policy lockout safety check.
     * </p>
     * <important>
     * <p>
     * Setting this value to true increases the risk that the CMK becomes unmanageable. Do not set this value to true
     * indiscriminately.
     * </p>
     * <p>
     * For more information, refer to the scenario in the <a href=
     * "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
     * >Default Key Policy</a> section in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
     * </p>
     * </important>
     * <p>
     * Use this parameter only when you include a policy in the request and you intend to prevent the principal that is
     * making the request from making a subsequent <a>PutKeyPolicy</a> request on the CMK.
     * </p>
     * <p>
     * The default value is false.
     * </p>
     * 
     * @return A flag to indicate whether to bypass the key policy lockout safety check.</p> <important>
     *         <p>
     *         Setting this value to true increases the risk that the CMK becomes unmanageable. Do not set this value to
     *         true indiscriminately.
     *         </p>
     *         <p>
     *         For more information, refer to the scenario in the <a href=
     *         "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
     *         >Default Key Policy</a> section in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
     *         </p>
     *         </important>
     *         <p>
     *         Use this parameter only when you include a policy in the request and you intend to prevent the principal
     *         that is making the request from making a subsequent <a>PutKeyPolicy</a> request on the CMK.
     *         </p>
     *         <p>
     *         The default value is false.
     */
    public final Boolean bypassPolicyLockoutSafetyCheck() {
        return bypassPolicyLockoutSafetyCheck;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Assigns one or more tags to the CMK. Use this parameter to tag the CMK when it is created. To tag an existing
     * CMK, use the <a>TagResource</a> operation.
     * </p>
     * <note>
     * <p>
     * Tagging or untagging a CMK can allow or deny permission to the CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/abac.html">Using ABAC in AWS KMS</a> in the <i>AWS
     * Key Management Service Developer Guide</i>.
     * </p>
     * </note>
     * <p>
     * To use this parameter, you must have <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/kms-api-permissions-reference.html"
     * >kms:TagResource</a> permission in an IAM policy.
     * </p>
     * <p>
     * Each tag consists of a tag key and a tag value. Both the tag key and the tag value are required, but the tag
     * value can be an empty (null) string. You cannot have more than one tag on a CMK with the same tag key. If you
     * specify an existing tag key with a different tag value, AWS KMS replaces the current tag value with the specified
     * one.
     * </p>
     * <p>
     * When you assign tags to an AWS resource, AWS generates a cost allocation report with usage and costs aggregated
     * by tags. Tags can also be used to control access to a CMK. For details, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/tagging-keys.html">Tagging Keys</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Assigns one or more tags to the CMK. Use this parameter to tag the CMK when it is created. To tag an
     *         existing CMK, use the <a>TagResource</a> operation.</p> <note>
     *         <p>
     *         Tagging or untagging a CMK can allow or deny permission to the CMK. For details, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/abac.html">Using ABAC in AWS KMS</a> in the
     *         <i>AWS Key Management Service Developer Guide</i>.
     *         </p>
     *         </note>
     *         <p>
     *         To use this parameter, you must have <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/kms-api-permissions-reference.html"
     *         >kms:TagResource</a> permission in an IAM policy.
     *         </p>
     *         <p>
     *         Each tag consists of a tag key and a tag value. Both the tag key and the tag value are required, but the
     *         tag value can be an empty (null) string. You cannot have more than one tag on a CMK with the same tag
     *         key. If you specify an existing tag key with a different tag value, AWS KMS replaces the current tag
     *         value with the specified one.
     *         </p>
     *         <p>
     *         When you assign tags to an AWS resource, AWS generates a cost allocation report with usage and costs
     *         aggregated by tags. Tags can also be used to control access to a CMK. For details, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/tagging-keys.html">Tagging Keys</a>.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Creates a multi-Region primary key that you can replicate into other AWS Regions. You cannot change this value
     * after you create the CMK.
     * </p>
     * <p>
     * For a multi-Region key, set this parameter to <code>True</code>. For a single-Region CMK, omit this parameter or
     * set it to <code>False</code>. The default value is <code>False</code>.
     * </p>
     * <p>
     * This operation supports <i>multi-Region keys</i>, an AWS KMS feature that lets you create multiple interoperable
     * CMKs in different AWS Regions. Because these CMKs have the same key ID, key material, and other metadata, you can
     * use them to encrypt data in one AWS Region and decrypt it in a different AWS Region without making a cross-Region
     * call or exposing the plaintext data. For more information about multi-Region keys, see <a
     * href="https://docs.aws.amazon.com/kms/latest/developerguide/multi-region-keys-overview.html">Using multi-Region
     * keys</a> in the <i>AWS Key Management Service Developer Guide</i>.
     * </p>
     * <p>
     * This value creates a <i>primary key</i>, not a replica. To create a <i>replica key</i>, use the
     * <a>ReplicateKey</a> operation.
     * </p>
     * <p>
     * You can create a symmetric or asymmetric multi-Region CMK, and you can create a multi-Region CMK with imported
     * key material. However, you cannot create a multi-Region CMK in a custom key store.
     * </p>
     * 
     * @return Creates a multi-Region primary key that you can replicate into other AWS Regions. You cannot change this
     *         value after you create the CMK. </p>
     *         <p>
     *         For a multi-Region key, set this parameter to <code>True</code>. For a single-Region CMK, omit this
     *         parameter or set it to <code>False</code>. The default value is <code>False</code>.
     *         </p>
     *         <p>
     *         This operation supports <i>multi-Region keys</i>, an AWS KMS feature that lets you create multiple
     *         interoperable CMKs in different AWS Regions. Because these CMKs have the same key ID, key material, and
     *         other metadata, you can use them to encrypt data in one AWS Region and decrypt it in a different AWS
     *         Region without making a cross-Region call or exposing the plaintext data. For more information about
     *         multi-Region keys, see <a
     *         href="https://docs.aws.amazon.com/kms/latest/developerguide/multi-region-keys-overview.html">Using
     *         multi-Region keys</a> in the <i>AWS Key Management Service Developer Guide</i>.
     *         </p>
     *         <p>
     *         This value creates a <i>primary key</i>, not a replica. To create a <i>replica key</i>, use the
     *         <a>ReplicateKey</a> operation.
     *         </p>
     *         <p>
     *         You can create a symmetric or asymmetric multi-Region CMK, and you can create a multi-Region CMK with
     *         imported key material. However, you cannot create a multi-Region CMK in a custom key store.
     */
    public final Boolean multiRegion() {
        return multiRegion;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(policy());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(keyUsageAsString());
        hashCode = 31 * hashCode + Objects.hashCode(customerMasterKeySpecAsString());
        hashCode = 31 * hashCode + Objects.hashCode(originAsString());
        hashCode = 31 * hashCode + Objects.hashCode(customKeyStoreId());
        hashCode = 31 * hashCode + Objects.hashCode(bypassPolicyLockoutSafetyCheck());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(multiRegion());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateKeyRequest)) {
            return false;
        }
        CreateKeyRequest other = (CreateKeyRequest) obj;
        return Objects.equals(policy(), other.policy()) && Objects.equals(description(), other.description())
                && Objects.equals(keyUsageAsString(), other.keyUsageAsString())
                && Objects.equals(customerMasterKeySpecAsString(), other.customerMasterKeySpecAsString())
                && Objects.equals(originAsString(), other.originAsString())
                && Objects.equals(customKeyStoreId(), other.customKeyStoreId())
                && Objects.equals(bypassPolicyLockoutSafetyCheck(), other.bypassPolicyLockoutSafetyCheck())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(multiRegion(), other.multiRegion());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateKeyRequest").add("Policy", policy()).add("Description", description())
                .add("KeyUsage", keyUsageAsString()).add("CustomerMasterKeySpec", customerMasterKeySpecAsString())
                .add("Origin", originAsString()).add("CustomKeyStoreId", customKeyStoreId())
                .add("BypassPolicyLockoutSafetyCheck", bypassPolicyLockoutSafetyCheck()).add("Tags", hasTags() ? tags() : null)
                .add("MultiRegion", multiRegion()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Policy":
            return Optional.ofNullable(clazz.cast(policy()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "KeyUsage":
            return Optional.ofNullable(clazz.cast(keyUsageAsString()));
        case "CustomerMasterKeySpec":
            return Optional.ofNullable(clazz.cast(customerMasterKeySpecAsString()));
        case "Origin":
            return Optional.ofNullable(clazz.cast(originAsString()));
        case "CustomKeyStoreId":
            return Optional.ofNullable(clazz.cast(customKeyStoreId()));
        case "BypassPolicyLockoutSafetyCheck":
            return Optional.ofNullable(clazz.cast(bypassPolicyLockoutSafetyCheck()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "MultiRegion":
            return Optional.ofNullable(clazz.cast(multiRegion()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateKeyRequest, T> g) {
        return obj -> g.apply((CreateKeyRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends KmsRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateKeyRequest> {
        /**
         * <p>
         * The key policy to attach to the CMK.
         * </p>
         * <p>
         * If you provide a key policy, it must meet the following criteria:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If you don't set <code>BypassPolicyLockoutSafetyCheck</code> to true, the key policy must allow the principal
         * that is making the <code>CreateKey</code> request to make a subsequent <a>PutKeyPolicy</a> request on the
         * CMK. This reduces the risk that the CMK becomes unmanageable. For more information, refer to the scenario in
         * the <a href=
         * "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
         * >Default Key Policy</a> section of the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Each statement in the key policy must contain one or more principals. The principals in the key policy must
         * exist and be visible to AWS KMS. When you create a new AWS principal (for example, an IAM user or role), you
         * might need to enforce a delay before including the new principal in a key policy because the new principal
         * might not be immediately visible to AWS KMS. For more information, see <a href=
         * "https://docs.aws.amazon.com/IAM/latest/UserGuide/troubleshoot_general.html#troubleshoot_general_eventual-consistency"
         * >Changes that I make are not always immediately visible</a> in the <i>AWS Identity and Access Management User
         * Guide</i>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you do not provide a key policy, AWS KMS attaches a default key policy to the CMK. For more information,
         * see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default">Default Key
         * Policy</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * <p>
         * The key policy size quota is 32 kilobytes (32768 bytes).
         * </p>
         * <p>
         * For help writing and formatting a JSON policy document, see the <a
         * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies.html">IAM JSON Policy Reference</a>
         * in the <i> <i>IAM User Guide</i> </i>.
         * </p>
         * 
         * @param policy
         *        The key policy to attach to the CMK.</p>
         *        <p>
         *        If you provide a key policy, it must meet the following criteria:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If you don't set <code>BypassPolicyLockoutSafetyCheck</code> to true, the key policy must allow the
         *        principal that is making the <code>CreateKey</code> request to make a subsequent <a>PutKeyPolicy</a>
         *        request on the CMK. This reduces the risk that the CMK becomes unmanageable. For more information,
         *        refer to the scenario in the <a href=
         *        "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
         *        >Default Key Policy</a> section of the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Each statement in the key policy must contain one or more principals. The principals in the key policy
         *        must exist and be visible to AWS KMS. When you create a new AWS principal (for example, an IAM user or
         *        role), you might need to enforce a delay before including the new principal in a key policy because
         *        the new principal might not be immediately visible to AWS KMS. For more information, see <a href=
         *        "https://docs.aws.amazon.com/IAM/latest/UserGuide/troubleshoot_general.html#troubleshoot_general_eventual-consistency"
         *        >Changes that I make are not always immediately visible</a> in the <i>AWS Identity and Access
         *        Management User Guide</i>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you do not provide a key policy, AWS KMS attaches a default key policy to the CMK. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default"
         *        >Default Key Policy</a> in the <i>AWS Key Management Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        The key policy size quota is 32 kilobytes (32768 bytes).
         *        </p>
         *        <p>
         *        For help writing and formatting a JSON policy document, see the <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies.html">IAM JSON Policy
         *        Reference</a> in the <i> <i>IAM User Guide</i> </i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder policy(String policy);

        /**
         * <p>
         * A description of the CMK.
         * </p>
         * <p>
         * Use a description that helps you decide whether the CMK is appropriate for a task. The default value is an
         * empty string (no description).
         * </p>
         * 
         * @param description
         *        A description of the CMK.</p>
         *        <p>
         *        Use a description that helps you decide whether the CMK is appropriate for a task. The default value
         *        is an empty string (no description).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Determines the <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations"
         * >cryptographic operations</a> for which you can use the CMK. The default value is
         * <code>ENCRYPT_DECRYPT</code>. This parameter is required only for asymmetric CMKs. You can't change the
         * <code>KeyUsage</code> value after the CMK is created.
         * </p>
         * <p>
         * Select only one valid value.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For symmetric CMKs, omit the parameter or specify <code>ENCRYPT_DECRYPT</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For asymmetric CMKs with RSA key material, specify <code>ENCRYPT_DECRYPT</code> or <code>SIGN_VERIFY</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For asymmetric CMKs with ECC key material, specify <code>SIGN_VERIFY</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param keyUsage
         *        Determines the <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations"
         *        >cryptographic operations</a> for which you can use the CMK. The default value is
         *        <code>ENCRYPT_DECRYPT</code>. This parameter is required only for asymmetric CMKs. You can't change
         *        the <code>KeyUsage</code> value after the CMK is created.</p>
         *        <p>
         *        Select only one valid value.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For symmetric CMKs, omit the parameter or specify <code>ENCRYPT_DECRYPT</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For asymmetric CMKs with RSA key material, specify <code>ENCRYPT_DECRYPT</code> or
         *        <code>SIGN_VERIFY</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For asymmetric CMKs with ECC key material, specify <code>SIGN_VERIFY</code>.
         *        </p>
         *        </li>
         * @see KeyUsageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyUsageType
         */
        Builder keyUsage(String keyUsage);

        /**
         * <p>
         * Determines the <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations"
         * >cryptographic operations</a> for which you can use the CMK. The default value is
         * <code>ENCRYPT_DECRYPT</code>. This parameter is required only for asymmetric CMKs. You can't change the
         * <code>KeyUsage</code> value after the CMK is created.
         * </p>
         * <p>
         * Select only one valid value.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For symmetric CMKs, omit the parameter or specify <code>ENCRYPT_DECRYPT</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For asymmetric CMKs with RSA key material, specify <code>ENCRYPT_DECRYPT</code> or <code>SIGN_VERIFY</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For asymmetric CMKs with ECC key material, specify <code>SIGN_VERIFY</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param keyUsage
         *        Determines the <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/concepts.html#cryptographic-operations"
         *        >cryptographic operations</a> for which you can use the CMK. The default value is
         *        <code>ENCRYPT_DECRYPT</code>. This parameter is required only for asymmetric CMKs. You can't change
         *        the <code>KeyUsage</code> value after the CMK is created.</p>
         *        <p>
         *        Select only one valid value.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For symmetric CMKs, omit the parameter or specify <code>ENCRYPT_DECRYPT</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For asymmetric CMKs with RSA key material, specify <code>ENCRYPT_DECRYPT</code> or
         *        <code>SIGN_VERIFY</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For asymmetric CMKs with ECC key material, specify <code>SIGN_VERIFY</code>.
         *        </p>
         *        </li>
         * @see KeyUsageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyUsageType
         */
        Builder keyUsage(KeyUsageType keyUsage);

        /**
         * <p>
         * Specifies the type of CMK to create. The default value, <code>SYMMETRIC_DEFAULT</code>, creates a CMK with a
         * 256-bit symmetric key for encryption and decryption. For help choosing a key spec for your CMK, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/symm-asymm-choose.html">How to Choose Your CMK
         * Configuration</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * <p>
         * The <code>CustomerMasterKeySpec</code> determines whether the CMK contains a symmetric key or an asymmetric
         * key pair. It also determines the encryption algorithms or signing algorithms that the CMK supports. You can't
         * change the <code>CustomerMasterKeySpec</code> after the CMK is created. To further restrict the algorithms
         * that can be used with the CMK, use a condition key in its key policy or IAM policy. For more information, see
         * <a href=
         * "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-algorithm"
         * >kms:EncryptionAlgorithm</a> or <a href=
         * "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-signing-algorithm"
         * >kms:Signing Algorithm</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * <important>
         * <p>
         * <a href="http://aws.amazon.com/kms/features/#AWS_Service_Integration">AWS services that are integrated with
         * AWS KMS</a> use symmetric CMKs to protect your data. These services do not support asymmetric CMKs. For help
         * determining whether a CMK is symmetric or asymmetric, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/find-symm-asymm.html">Identifying Symmetric and
         * Asymmetric CMKs</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * </important>
         * <p>
         * AWS KMS supports the following key specs for CMKs:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Symmetric key (default)
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SYMMETRIC_DEFAULT</code> (AES-256-GCM)
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Asymmetric RSA key pairs
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>RSA_2048</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RSA_3072</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RSA_4096</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Asymmetric NIST-recommended elliptic curve key pairs
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ECC_NIST_P256</code> (secp256r1)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ECC_NIST_P384</code> (secp384r1)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ECC_NIST_P521</code> (secp521r1)
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Other asymmetric elliptic curve key pairs
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ECC_SECG_P256K1</code> (secp256k1), commonly used for cryptocurrencies.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param customerMasterKeySpec
         *        Specifies the type of CMK to create. The default value, <code>SYMMETRIC_DEFAULT</code>, creates a CMK
         *        with a 256-bit symmetric key for encryption and decryption. For help choosing a key spec for your CMK,
         *        see <a href="https://docs.aws.amazon.com/kms/latest/developerguide/symm-asymm-choose.html">How to
         *        Choose Your CMK Configuration</a> in the <i>AWS Key Management Service Developer Guide</i>.</p>
         *        <p>
         *        The <code>CustomerMasterKeySpec</code> determines whether the CMK contains a symmetric key or an
         *        asymmetric key pair. It also determines the encryption algorithms or signing algorithms that the CMK
         *        supports. You can't change the <code>CustomerMasterKeySpec</code> after the CMK is created. To further
         *        restrict the algorithms that can be used with the CMK, use a condition key in its key policy or IAM
         *        policy. For more information, see <a href=
         *        "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-algorithm"
         *        >kms:EncryptionAlgorithm</a> or <a href=
         *        "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-signing-algorithm"
         *        >kms:Signing Algorithm</a> in the <i>AWS Key Management Service Developer Guide</i>.
         *        </p>
         *        <important>
         *        <p>
         *        <a href="http://aws.amazon.com/kms/features/#AWS_Service_Integration">AWS services that are integrated
         *        with AWS KMS</a> use symmetric CMKs to protect your data. These services do not support asymmetric
         *        CMKs. For help determining whether a CMK is symmetric or asymmetric, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/find-symm-asymm.html">Identifying
         *        Symmetric and Asymmetric CMKs</a> in the <i>AWS Key Management Service Developer Guide</i>.
         *        </p>
         *        </important>
         *        <p>
         *        AWS KMS supports the following key specs for CMKs:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Symmetric key (default)
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SYMMETRIC_DEFAULT</code> (AES-256-GCM)
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Asymmetric RSA key pairs
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>RSA_2048</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RSA_3072</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RSA_4096</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Asymmetric NIST-recommended elliptic curve key pairs
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ECC_NIST_P256</code> (secp256r1)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ECC_NIST_P384</code> (secp384r1)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ECC_NIST_P521</code> (secp521r1)
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Other asymmetric elliptic curve key pairs
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ECC_SECG_P256K1</code> (secp256k1), commonly used for cryptocurrencies.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see CustomerMasterKeySpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CustomerMasterKeySpec
         */
        Builder customerMasterKeySpec(String customerMasterKeySpec);

        /**
         * <p>
         * Specifies the type of CMK to create. The default value, <code>SYMMETRIC_DEFAULT</code>, creates a CMK with a
         * 256-bit symmetric key for encryption and decryption. For help choosing a key spec for your CMK, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/symm-asymm-choose.html">How to Choose Your CMK
         * Configuration</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * <p>
         * The <code>CustomerMasterKeySpec</code> determines whether the CMK contains a symmetric key or an asymmetric
         * key pair. It also determines the encryption algorithms or signing algorithms that the CMK supports. You can't
         * change the <code>CustomerMasterKeySpec</code> after the CMK is created. To further restrict the algorithms
         * that can be used with the CMK, use a condition key in its key policy or IAM policy. For more information, see
         * <a href=
         * "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-algorithm"
         * >kms:EncryptionAlgorithm</a> or <a href=
         * "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-signing-algorithm"
         * >kms:Signing Algorithm</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * <important>
         * <p>
         * <a href="http://aws.amazon.com/kms/features/#AWS_Service_Integration">AWS services that are integrated with
         * AWS KMS</a> use symmetric CMKs to protect your data. These services do not support asymmetric CMKs. For help
         * determining whether a CMK is symmetric or asymmetric, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/find-symm-asymm.html">Identifying Symmetric and
         * Asymmetric CMKs</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * </important>
         * <p>
         * AWS KMS supports the following key specs for CMKs:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Symmetric key (default)
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>SYMMETRIC_DEFAULT</code> (AES-256-GCM)
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Asymmetric RSA key pairs
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>RSA_2048</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RSA_3072</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RSA_4096</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Asymmetric NIST-recommended elliptic curve key pairs
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ECC_NIST_P256</code> (secp256r1)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ECC_NIST_P384</code> (secp384r1)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ECC_NIST_P521</code> (secp521r1)
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Other asymmetric elliptic curve key pairs
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ECC_SECG_P256K1</code> (secp256k1), commonly used for cryptocurrencies.
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * 
         * @param customerMasterKeySpec
         *        Specifies the type of CMK to create. The default value, <code>SYMMETRIC_DEFAULT</code>, creates a CMK
         *        with a 256-bit symmetric key for encryption and decryption. For help choosing a key spec for your CMK,
         *        see <a href="https://docs.aws.amazon.com/kms/latest/developerguide/symm-asymm-choose.html">How to
         *        Choose Your CMK Configuration</a> in the <i>AWS Key Management Service Developer Guide</i>.</p>
         *        <p>
         *        The <code>CustomerMasterKeySpec</code> determines whether the CMK contains a symmetric key or an
         *        asymmetric key pair. It also determines the encryption algorithms or signing algorithms that the CMK
         *        supports. You can't change the <code>CustomerMasterKeySpec</code> after the CMK is created. To further
         *        restrict the algorithms that can be used with the CMK, use a condition key in its key policy or IAM
         *        policy. For more information, see <a href=
         *        "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-encryption-algorithm"
         *        >kms:EncryptionAlgorithm</a> or <a href=
         *        "https://docs.aws.amazon.com/kms/latest/developerguide/policy-conditions.html#conditions-kms-signing-algorithm"
         *        >kms:Signing Algorithm</a> in the <i>AWS Key Management Service Developer Guide</i>.
         *        </p>
         *        <important>
         *        <p>
         *        <a href="http://aws.amazon.com/kms/features/#AWS_Service_Integration">AWS services that are integrated
         *        with AWS KMS</a> use symmetric CMKs to protect your data. These services do not support asymmetric
         *        CMKs. For help determining whether a CMK is symmetric or asymmetric, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/find-symm-asymm.html">Identifying
         *        Symmetric and Asymmetric CMKs</a> in the <i>AWS Key Management Service Developer Guide</i>.
         *        </p>
         *        </important>
         *        <p>
         *        AWS KMS supports the following key specs for CMKs:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Symmetric key (default)
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>SYMMETRIC_DEFAULT</code> (AES-256-GCM)
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Asymmetric RSA key pairs
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>RSA_2048</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RSA_3072</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RSA_4096</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Asymmetric NIST-recommended elliptic curve key pairs
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ECC_NIST_P256</code> (secp256r1)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ECC_NIST_P384</code> (secp384r1)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ECC_NIST_P521</code> (secp521r1)
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Other asymmetric elliptic curve key pairs
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ECC_SECG_P256K1</code> (secp256k1), commonly used for cryptocurrencies.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         * @see CustomerMasterKeySpec
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CustomerMasterKeySpec
         */
        Builder customerMasterKeySpec(CustomerMasterKeySpec customerMasterKeySpec);

        /**
         * <p>
         * The source of the key material for the CMK. You cannot change the origin after you create the CMK. The
         * default is <code>AWS_KMS</code>, which means that AWS KMS creates the key material.
         * </p>
         * <p>
         * To create a CMK with no key material (for imported key material), set the value to <code>EXTERNAL</code>. For
         * more information about importing key material into AWS KMS, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key Material</a>
         * in the <i>AWS Key Management Service Developer Guide</i>. This value is valid only for symmetric CMKs.
         * </p>
         * <p>
         * To create a CMK in an AWS KMS <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
         * store</a> and create its key material in the associated AWS CloudHSM cluster, set this value to
         * <code>AWS_CLOUDHSM</code>. You must also use the <code>CustomKeyStoreId</code> parameter to identify the
         * custom key store. This value is valid only for symmetric CMKs.
         * </p>
         * 
         * @param origin
         *        The source of the key material for the CMK. You cannot change the origin after you create the CMK. The
         *        default is <code>AWS_KMS</code>, which means that AWS KMS creates the key material.</p>
         *        <p>
         *        To create a CMK with no key material (for imported key material), set the value to
         *        <code>EXTERNAL</code>. For more information about importing key material into AWS KMS, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key
         *        Material</a> in the <i>AWS Key Management Service Developer Guide</i>. This value is valid only for
         *        symmetric CMKs.
         *        </p>
         *        <p>
         *        To create a CMK in an AWS KMS <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
         *        store</a> and create its key material in the associated AWS CloudHSM cluster, set this value to
         *        <code>AWS_CLOUDHSM</code>. You must also use the <code>CustomKeyStoreId</code> parameter to identify
         *        the custom key store. This value is valid only for symmetric CMKs.
         * @see OriginType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OriginType
         */
        Builder origin(String origin);

        /**
         * <p>
         * The source of the key material for the CMK. You cannot change the origin after you create the CMK. The
         * default is <code>AWS_KMS</code>, which means that AWS KMS creates the key material.
         * </p>
         * <p>
         * To create a CMK with no key material (for imported key material), set the value to <code>EXTERNAL</code>. For
         * more information about importing key material into AWS KMS, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key Material</a>
         * in the <i>AWS Key Management Service Developer Guide</i>. This value is valid only for symmetric CMKs.
         * </p>
         * <p>
         * To create a CMK in an AWS KMS <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
         * store</a> and create its key material in the associated AWS CloudHSM cluster, set this value to
         * <code>AWS_CLOUDHSM</code>. You must also use the <code>CustomKeyStoreId</code> parameter to identify the
         * custom key store. This value is valid only for symmetric CMKs.
         * </p>
         * 
         * @param origin
         *        The source of the key material for the CMK. You cannot change the origin after you create the CMK. The
         *        default is <code>AWS_KMS</code>, which means that AWS KMS creates the key material.</p>
         *        <p>
         *        To create a CMK with no key material (for imported key material), set the value to
         *        <code>EXTERNAL</code>. For more information about importing key material into AWS KMS, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/importing-keys.html">Importing Key
         *        Material</a> in the <i>AWS Key Management Service Developer Guide</i>. This value is valid only for
         *        symmetric CMKs.
         *        </p>
         *        <p>
         *        To create a CMK in an AWS KMS <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
         *        store</a> and create its key material in the associated AWS CloudHSM cluster, set this value to
         *        <code>AWS_CLOUDHSM</code>. You must also use the <code>CustomKeyStoreId</code> parameter to identify
         *        the custom key store. This value is valid only for symmetric CMKs.
         * @see OriginType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OriginType
         */
        Builder origin(OriginType origin);

        /**
         * <p>
         * Creates the CMK in the specified <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
         * store</a> and the key material in its associated AWS CloudHSM cluster. To create a CMK in a custom key store,
         * you must also specify the <code>Origin</code> parameter with a value of <code>AWS_CLOUDHSM</code>. The AWS
         * CloudHSM cluster that is associated with the custom key store must have at least two active HSMs, each in a
         * different Availability Zone in the Region.
         * </p>
         * <p>
         * This parameter is valid only for symmetric CMKs and regional CMKs. You cannot create an asymmetric CMK or a
         * multi-Region CMK in a custom key store.
         * </p>
         * <p>
         * To find the ID of a custom key store, use the <a>DescribeCustomKeyStores</a> operation.
         * </p>
         * <p>
         * The response includes the custom key store ID and the ID of the AWS CloudHSM cluster.
         * </p>
         * <p>
         * This operation is part of the <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">Custom Key Store
         * feature</a> feature in AWS KMS, which combines the convenience and extensive integration of AWS KMS with the
         * isolation and control of a single-tenant key store.
         * </p>
         * 
         * @param customKeyStoreId
         *        Creates the CMK in the specified <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">custom key
         *        store</a> and the key material in its associated AWS CloudHSM cluster. To create a CMK in a custom key
         *        store, you must also specify the <code>Origin</code> parameter with a value of
         *        <code>AWS_CLOUDHSM</code>. The AWS CloudHSM cluster that is associated with the custom key store must
         *        have at least two active HSMs, each in a different Availability Zone in the Region.</p>
         *        <p>
         *        This parameter is valid only for symmetric CMKs and regional CMKs. You cannot create an asymmetric CMK
         *        or a multi-Region CMK in a custom key store.
         *        </p>
         *        <p>
         *        To find the ID of a custom key store, use the <a>DescribeCustomKeyStores</a> operation.
         *        </p>
         *        <p>
         *        The response includes the custom key store ID and the ID of the AWS CloudHSM cluster.
         *        </p>
         *        <p>
         *        This operation is part of the <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/custom-key-store-overview.html">Custom Key
         *        Store feature</a> feature in AWS KMS, which combines the convenience and extensive integration of AWS
         *        KMS with the isolation and control of a single-tenant key store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customKeyStoreId(String customKeyStoreId);

        /**
         * <p>
         * A flag to indicate whether to bypass the key policy lockout safety check.
         * </p>
         * <important>
         * <p>
         * Setting this value to true increases the risk that the CMK becomes unmanageable. Do not set this value to
         * true indiscriminately.
         * </p>
         * <p>
         * For more information, refer to the scenario in the <a href=
         * "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
         * >Default Key Policy</a> section in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
         * </p>
         * </important>
         * <p>
         * Use this parameter only when you include a policy in the request and you intend to prevent the principal that
         * is making the request from making a subsequent <a>PutKeyPolicy</a> request on the CMK.
         * </p>
         * <p>
         * The default value is false.
         * </p>
         * 
         * @param bypassPolicyLockoutSafetyCheck
         *        A flag to indicate whether to bypass the key policy lockout safety check.</p> <important>
         *        <p>
         *        Setting this value to true increases the risk that the CMK becomes unmanageable. Do not set this value
         *        to true indiscriminately.
         *        </p>
         *        <p>
         *        For more information, refer to the scenario in the <a href=
         *        "https://docs.aws.amazon.com/kms/latest/developerguide/key-policies.html#key-policy-default-allow-root-enable-iam"
         *        >Default Key Policy</a> section in the <i> <i>AWS Key Management Service Developer Guide</i> </i>.
         *        </p>
         *        </important>
         *        <p>
         *        Use this parameter only when you include a policy in the request and you intend to prevent the
         *        principal that is making the request from making a subsequent <a>PutKeyPolicy</a> request on the CMK.
         *        </p>
         *        <p>
         *        The default value is false.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bypassPolicyLockoutSafetyCheck(Boolean bypassPolicyLockoutSafetyCheck);

        /**
         * <p>
         * Assigns one or more tags to the CMK. Use this parameter to tag the CMK when it is created. To tag an existing
         * CMK, use the <a>TagResource</a> operation.
         * </p>
         * <note>
         * <p>
         * Tagging or untagging a CMK can allow or deny permission to the CMK. For details, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/abac.html">Using ABAC in AWS KMS</a> in the
         * <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * </note>
         * <p>
         * To use this parameter, you must have <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/kms-api-permissions-reference.html"
         * >kms:TagResource</a> permission in an IAM policy.
         * </p>
         * <p>
         * Each tag consists of a tag key and a tag value. Both the tag key and the tag value are required, but the tag
         * value can be an empty (null) string. You cannot have more than one tag on a CMK with the same tag key. If you
         * specify an existing tag key with a different tag value, AWS KMS replaces the current tag value with the
         * specified one.
         * </p>
         * <p>
         * When you assign tags to an AWS resource, AWS generates a cost allocation report with usage and costs
         * aggregated by tags. Tags can also be used to control access to a CMK. For details, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/tagging-keys.html">Tagging Keys</a>.
         * </p>
         * 
         * @param tags
         *        Assigns one or more tags to the CMK. Use this parameter to tag the CMK when it is created. To tag an
         *        existing CMK, use the <a>TagResource</a> operation.</p> <note>
         *        <p>
         *        Tagging or untagging a CMK can allow or deny permission to the CMK. For details, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/abac.html">Using ABAC in AWS KMS</a> in
         *        the <i>AWS Key Management Service Developer Guide</i>.
         *        </p>
         *        </note>
         *        <p>
         *        To use this parameter, you must have <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/kms-api-permissions-reference.html"
         *        >kms:TagResource</a> permission in an IAM policy.
         *        </p>
         *        <p>
         *        Each tag consists of a tag key and a tag value. Both the tag key and the tag value are required, but
         *        the tag value can be an empty (null) string. You cannot have more than one tag on a CMK with the same
         *        tag key. If you specify an existing tag key with a different tag value, AWS KMS replaces the current
         *        tag value with the specified one.
         *        </p>
         *        <p>
         *        When you assign tags to an AWS resource, AWS generates a cost allocation report with usage and costs
         *        aggregated by tags. Tags can also be used to control access to a CMK. For details, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/tagging-keys.html">Tagging Keys</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Assigns one or more tags to the CMK. Use this parameter to tag the CMK when it is created. To tag an existing
         * CMK, use the <a>TagResource</a> operation.
         * </p>
         * <note>
         * <p>
         * Tagging or untagging a CMK can allow or deny permission to the CMK. For details, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/abac.html">Using ABAC in AWS KMS</a> in the
         * <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * </note>
         * <p>
         * To use this parameter, you must have <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/kms-api-permissions-reference.html"
         * >kms:TagResource</a> permission in an IAM policy.
         * </p>
         * <p>
         * Each tag consists of a tag key and a tag value. Both the tag key and the tag value are required, but the tag
         * value can be an empty (null) string. You cannot have more than one tag on a CMK with the same tag key. If you
         * specify an existing tag key with a different tag value, AWS KMS replaces the current tag value with the
         * specified one.
         * </p>
         * <p>
         * When you assign tags to an AWS resource, AWS generates a cost allocation report with usage and costs
         * aggregated by tags. Tags can also be used to control access to a CMK. For details, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/tagging-keys.html">Tagging Keys</a>.
         * </p>
         * 
         * @param tags
         *        Assigns one or more tags to the CMK. Use this parameter to tag the CMK when it is created. To tag an
         *        existing CMK, use the <a>TagResource</a> operation.</p> <note>
         *        <p>
         *        Tagging or untagging a CMK can allow or deny permission to the CMK. For details, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/abac.html">Using ABAC in AWS KMS</a> in
         *        the <i>AWS Key Management Service Developer Guide</i>.
         *        </p>
         *        </note>
         *        <p>
         *        To use this parameter, you must have <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/kms-api-permissions-reference.html"
         *        >kms:TagResource</a> permission in an IAM policy.
         *        </p>
         *        <p>
         *        Each tag consists of a tag key and a tag value. Both the tag key and the tag value are required, but
         *        the tag value can be an empty (null) string. You cannot have more than one tag on a CMK with the same
         *        tag key. If you specify an existing tag key with a different tag value, AWS KMS replaces the current
         *        tag value with the specified one.
         *        </p>
         *        <p>
         *        When you assign tags to an AWS resource, AWS generates a cost allocation report with usage and costs
         *        aggregated by tags. Tags can also be used to control access to a CMK. For details, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/tagging-keys.html">Tagging Keys</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Assigns one or more tags to the CMK. Use this parameter to tag the CMK when it is created. To tag an existing
         * CMK, use the <a>TagResource</a> operation.
         * </p>
         * <note>
         * <p>
         * Tagging or untagging a CMK can allow or deny permission to the CMK. For details, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/abac.html">Using ABAC in AWS KMS</a> in the
         * <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * </note>
         * <p>
         * To use this parameter, you must have <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/kms-api-permissions-reference.html"
         * >kms:TagResource</a> permission in an IAM policy.
         * </p>
         * <p>
         * Each tag consists of a tag key and a tag value. Both the tag key and the tag value are required, but the tag
         * value can be an empty (null) string. You cannot have more than one tag on a CMK with the same tag key. If you
         * specify an existing tag key with a different tag value, AWS KMS replaces the current tag value with the
         * specified one.
         * </p>
         * <p>
         * When you assign tags to an AWS resource, AWS generates a cost allocation report with usage and costs
         * aggregated by tags. Tags can also be used to control access to a CMK. For details, see <a
         * href="https://docs.aws.amazon.com/kms/latest/developerguide/tagging-keys.html">Tagging Keys</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Creates a multi-Region primary key that you can replicate into other AWS Regions. You cannot change this
         * value after you create the CMK.
         * </p>
         * <p>
         * For a multi-Region key, set this parameter to <code>True</code>. For a single-Region CMK, omit this parameter
         * or set it to <code>False</code>. The default value is <code>False</code>.
         * </p>
         * <p>
         * This operation supports <i>multi-Region keys</i>, an AWS KMS feature that lets you create multiple
         * interoperable CMKs in different AWS Regions. Because these CMKs have the same key ID, key material, and other
         * metadata, you can use them to encrypt data in one AWS Region and decrypt it in a different AWS Region without
         * making a cross-Region call or exposing the plaintext data. For more information about multi-Region keys, see
         * <a href="https://docs.aws.amazon.com/kms/latest/developerguide/multi-region-keys-overview.html">Using
         * multi-Region keys</a> in the <i>AWS Key Management Service Developer Guide</i>.
         * </p>
         * <p>
         * This value creates a <i>primary key</i>, not a replica. To create a <i>replica key</i>, use the
         * <a>ReplicateKey</a> operation.
         * </p>
         * <p>
         * You can create a symmetric or asymmetric multi-Region CMK, and you can create a multi-Region CMK with
         * imported key material. However, you cannot create a multi-Region CMK in a custom key store.
         * </p>
         * 
         * @param multiRegion
         *        Creates a multi-Region primary key that you can replicate into other AWS Regions. You cannot change
         *        this value after you create the CMK. </p>
         *        <p>
         *        For a multi-Region key, set this parameter to <code>True</code>. For a single-Region CMK, omit this
         *        parameter or set it to <code>False</code>. The default value is <code>False</code>.
         *        </p>
         *        <p>
         *        This operation supports <i>multi-Region keys</i>, an AWS KMS feature that lets you create multiple
         *        interoperable CMKs in different AWS Regions. Because these CMKs have the same key ID, key material,
         *        and other metadata, you can use them to encrypt data in one AWS Region and decrypt it in a different
         *        AWS Region without making a cross-Region call or exposing the plaintext data. For more information
         *        about multi-Region keys, see <a
         *        href="https://docs.aws.amazon.com/kms/latest/developerguide/multi-region-keys-overview.html">Using
         *        multi-Region keys</a> in the <i>AWS Key Management Service Developer Guide</i>.
         *        </p>
         *        <p>
         *        This value creates a <i>primary key</i>, not a replica. To create a <i>replica key</i>, use the
         *        <a>ReplicateKey</a> operation.
         *        </p>
         *        <p>
         *        You can create a symmetric or asymmetric multi-Region CMK, and you can create a multi-Region CMK with
         *        imported key material. However, you cannot create a multi-Region CMK in a custom key store.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder multiRegion(Boolean multiRegion);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends KmsRequest.BuilderImpl implements Builder {
        private String policy;

        private String description;

        private String keyUsage;

        private String customerMasterKeySpec;

        private String origin;

        private String customKeyStoreId;

        private Boolean bypassPolicyLockoutSafetyCheck;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private Boolean multiRegion;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateKeyRequest model) {
            super(model);
            policy(model.policy);
            description(model.description);
            keyUsage(model.keyUsage);
            customerMasterKeySpec(model.customerMasterKeySpec);
            origin(model.origin);
            customKeyStoreId(model.customKeyStoreId);
            bypassPolicyLockoutSafetyCheck(model.bypassPolicyLockoutSafetyCheck);
            tags(model.tags);
            multiRegion(model.multiRegion);
        }

        public final String getPolicy() {
            return policy;
        }

        @Override
        public final Builder policy(String policy) {
            this.policy = policy;
            return this;
        }

        public final void setPolicy(String policy) {
            this.policy = policy;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final String getKeyUsage() {
            return keyUsage;
        }

        @Override
        public final Builder keyUsage(String keyUsage) {
            this.keyUsage = keyUsage;
            return this;
        }

        @Override
        public final Builder keyUsage(KeyUsageType keyUsage) {
            this.keyUsage(keyUsage == null ? null : keyUsage.toString());
            return this;
        }

        public final void setKeyUsage(String keyUsage) {
            this.keyUsage = keyUsage;
        }

        public final String getCustomerMasterKeySpec() {
            return customerMasterKeySpec;
        }

        @Override
        public final Builder customerMasterKeySpec(String customerMasterKeySpec) {
            this.customerMasterKeySpec = customerMasterKeySpec;
            return this;
        }

        @Override
        public final Builder customerMasterKeySpec(CustomerMasterKeySpec customerMasterKeySpec) {
            this.customerMasterKeySpec(customerMasterKeySpec == null ? null : customerMasterKeySpec.toString());
            return this;
        }

        public final void setCustomerMasterKeySpec(String customerMasterKeySpec) {
            this.customerMasterKeySpec = customerMasterKeySpec;
        }

        public final String getOrigin() {
            return origin;
        }

        @Override
        public final Builder origin(String origin) {
            this.origin = origin;
            return this;
        }

        @Override
        public final Builder origin(OriginType origin) {
            this.origin(origin == null ? null : origin.toString());
            return this;
        }

        public final void setOrigin(String origin) {
            this.origin = origin;
        }

        public final String getCustomKeyStoreId() {
            return customKeyStoreId;
        }

        @Override
        public final Builder customKeyStoreId(String customKeyStoreId) {
            this.customKeyStoreId = customKeyStoreId;
            return this;
        }

        public final void setCustomKeyStoreId(String customKeyStoreId) {
            this.customKeyStoreId = customKeyStoreId;
        }

        public final Boolean getBypassPolicyLockoutSafetyCheck() {
            return bypassPolicyLockoutSafetyCheck;
        }

        @Override
        public final Builder bypassPolicyLockoutSafetyCheck(Boolean bypassPolicyLockoutSafetyCheck) {
            this.bypassPolicyLockoutSafetyCheck = bypassPolicyLockoutSafetyCheck;
            return this;
        }

        public final void setBypassPolicyLockoutSafetyCheck(Boolean bypassPolicyLockoutSafetyCheck) {
            this.bypassPolicyLockoutSafetyCheck = bypassPolicyLockoutSafetyCheck;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final Boolean getMultiRegion() {
            return multiRegion;
        }

        @Override
        public final Builder multiRegion(Boolean multiRegion) {
            this.multiRegion = multiRegion;
            return this;
        }

        public final void setMultiRegion(Boolean multiRegion) {
            this.multiRegion = multiRegion;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateKeyRequest build() {
            return new CreateKeyRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
