/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntimev2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the contexts that a user is using in a session. You can configure Amazon Lex V2 to set a
 * context when an intent is fulfilled, or you can set a context using the , , or operations.
 * </p>
 * <p>
 * Use a context to indicate to Amazon Lex V2 intents that should be used as follow-up intents. For example, if the
 * active context is <code>order-fulfilled</code>, only intents that have <code>order-fulfilled</code> configured as a
 * trigger are considered for follow up.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ActiveContext implements SdkPojo, Serializable, ToCopyableBuilder<ActiveContext.Builder, ActiveContext> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(ActiveContext::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<ActiveContextTimeToLive> TIME_TO_LIVE_FIELD = SdkField
            .<ActiveContextTimeToLive> builder(MarshallingType.SDK_POJO).memberName("timeToLive")
            .getter(getter(ActiveContext::timeToLive)).setter(setter(Builder::timeToLive))
            .constructor(ActiveContextTimeToLive::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timeToLive").build()).build();

    private static final SdkField<Map<String, String>> CONTEXT_ATTRIBUTES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("contextAttributes")
            .getter(getter(ActiveContext::contextAttributes))
            .setter(setter(Builder::contextAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("contextAttributes").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            TIME_TO_LIVE_FIELD, CONTEXT_ATTRIBUTES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final ActiveContextTimeToLive timeToLive;

    private final Map<String, String> contextAttributes;

    private ActiveContext(BuilderImpl builder) {
        this.name = builder.name;
        this.timeToLive = builder.timeToLive;
        this.contextAttributes = builder.contextAttributes;
    }

    /**
     * <p>
     * The name of the context.
     * </p>
     * 
     * @return The name of the context.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Indicates the number of turns or seconds that the context is active. Once the time to live expires, the context
     * is no longer returned in a response.
     * </p>
     * 
     * @return Indicates the number of turns or seconds that the context is active. Once the time to live expires, the
     *         context is no longer returned in a response.
     */
    public final ActiveContextTimeToLive timeToLive() {
        return timeToLive;
    }

    /**
     * For responses, this returns true if the service returned a value for the ContextAttributes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasContextAttributes() {
        return contextAttributes != null && !(contextAttributes instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A list of contexts active for the request. A context can be activated when a previous intent is fulfilled, or by
     * including the context in the request.
     * </p>
     * <p>
     * If you don't specify a list of contexts, Amazon Lex V2 will use the current list of contexts for the session. If
     * you specify an empty list, all contexts for the session are cleared.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasContextAttributes} method.
     * </p>
     * 
     * @return A list of contexts active for the request. A context can be activated when a previous intent is
     *         fulfilled, or by including the context in the request.</p>
     *         <p>
     *         If you don't specify a list of contexts, Amazon Lex V2 will use the current list of contexts for the
     *         session. If you specify an empty list, all contexts for the session are cleared.
     */
    public final Map<String, String> contextAttributes() {
        return contextAttributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(timeToLive());
        hashCode = 31 * hashCode + Objects.hashCode(hasContextAttributes() ? contextAttributes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ActiveContext)) {
            return false;
        }
        ActiveContext other = (ActiveContext) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(timeToLive(), other.timeToLive())
                && hasContextAttributes() == other.hasContextAttributes()
                && Objects.equals(contextAttributes(), other.contextAttributes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ActiveContext").add("Name", name()).add("TimeToLive", timeToLive())
                .add("ContextAttributes", contextAttributes() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "timeToLive":
            return Optional.ofNullable(clazz.cast(timeToLive()));
        case "contextAttributes":
            return Optional.ofNullable(clazz.cast(contextAttributes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ActiveContext, T> g) {
        return obj -> g.apply((ActiveContext) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ActiveContext> {
        /**
         * <p>
         * The name of the context.
         * </p>
         * 
         * @param name
         *        The name of the context.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Indicates the number of turns or seconds that the context is active. Once the time to live expires, the
         * context is no longer returned in a response.
         * </p>
         * 
         * @param timeToLive
         *        Indicates the number of turns or seconds that the context is active. Once the time to live expires,
         *        the context is no longer returned in a response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeToLive(ActiveContextTimeToLive timeToLive);

        /**
         * <p>
         * Indicates the number of turns or seconds that the context is active. Once the time to live expires, the
         * context is no longer returned in a response.
         * </p>
         * This is a convenience that creates an instance of the {@link ActiveContextTimeToLive.Builder} avoiding the
         * need to create one manually via {@link ActiveContextTimeToLive#builder()}.
         *
         * When the {@link Consumer} completes, {@link ActiveContextTimeToLive.Builder#build()} is called immediately
         * and its result is passed to {@link #timeToLive(ActiveContextTimeToLive)}.
         * 
         * @param timeToLive
         *        a consumer that will call methods on {@link ActiveContextTimeToLive.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #timeToLive(ActiveContextTimeToLive)
         */
        default Builder timeToLive(Consumer<ActiveContextTimeToLive.Builder> timeToLive) {
            return timeToLive(ActiveContextTimeToLive.builder().applyMutation(timeToLive).build());
        }

        /**
         * <p>
         * A list of contexts active for the request. A context can be activated when a previous intent is fulfilled, or
         * by including the context in the request.
         * </p>
         * <p>
         * If you don't specify a list of contexts, Amazon Lex V2 will use the current list of contexts for the session.
         * If you specify an empty list, all contexts for the session are cleared.
         * </p>
         * 
         * @param contextAttributes
         *        A list of contexts active for the request. A context can be activated when a previous intent is
         *        fulfilled, or by including the context in the request.</p>
         *        <p>
         *        If you don't specify a list of contexts, Amazon Lex V2 will use the current list of contexts for the
         *        session. If you specify an empty list, all contexts for the session are cleared.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder contextAttributes(Map<String, String> contextAttributes);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private ActiveContextTimeToLive timeToLive;

        private Map<String, String> contextAttributes = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ActiveContext model) {
            name(model.name);
            timeToLive(model.timeToLive);
            contextAttributes(model.contextAttributes);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final ActiveContextTimeToLive.Builder getTimeToLive() {
            return timeToLive != null ? timeToLive.toBuilder() : null;
        }

        public final void setTimeToLive(ActiveContextTimeToLive.BuilderImpl timeToLive) {
            this.timeToLive = timeToLive != null ? timeToLive.build() : null;
        }

        @Override
        @Transient
        public final Builder timeToLive(ActiveContextTimeToLive timeToLive) {
            this.timeToLive = timeToLive;
            return this;
        }

        public final Map<String, String> getContextAttributes() {
            if (contextAttributes instanceof SdkAutoConstructMap) {
                return null;
            }
            return contextAttributes;
        }

        public final void setContextAttributes(Map<String, String> contextAttributes) {
            this.contextAttributes = ActiveContextParametersMapCopier.copy(contextAttributes);
        }

        @Override
        @Transient
        public final Builder contextAttributes(Map<String, String> contextAttributes) {
            this.contextAttributes = ActiveContextParametersMapCopier.copy(contextAttributes);
            return this;
        }

        @Override
        public ActiveContext build() {
            return new ActiveContext(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
