/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.lexruntimev2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A card that is shown to the user by a messaging platform. You define the contents of the card, the card is displayed
 * by the platform.
 * </p>
 * <p>
 * When you use a response card, the response from the user is constrained to the text associated with a button on the
 * card.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImageResponseCard implements SdkPojo, Serializable,
        ToCopyableBuilder<ImageResponseCard.Builder, ImageResponseCard> {
    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("title")
            .getter(getter(ImageResponseCard::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("title").build()).build();

    private static final SdkField<String> SUBTITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("subtitle").getter(getter(ImageResponseCard::subtitle)).setter(setter(Builder::subtitle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subtitle").build()).build();

    private static final SdkField<String> IMAGE_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("imageUrl").getter(getter(ImageResponseCard::imageUrl)).setter(setter(Builder::imageUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imageUrl").build()).build();

    private static final SdkField<List<Button>> BUTTONS_FIELD = SdkField
            .<List<Button>> builder(MarshallingType.LIST)
            .memberName("buttons")
            .getter(getter(ImageResponseCard::buttons))
            .setter(setter(Builder::buttons))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("buttons").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Button> builder(MarshallingType.SDK_POJO)
                                            .constructor(Button::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TITLE_FIELD, SUBTITLE_FIELD,
            IMAGE_URL_FIELD, BUTTONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String title;

    private final String subtitle;

    private final String imageUrl;

    private final List<Button> buttons;

    private ImageResponseCard(BuilderImpl builder) {
        this.title = builder.title;
        this.subtitle = builder.subtitle;
        this.imageUrl = builder.imageUrl;
        this.buttons = builder.buttons;
    }

    /**
     * <p>
     * The title to display on the response card. The format of the title is determined by the platform displaying the
     * response card.
     * </p>
     * 
     * @return The title to display on the response card. The format of the title is determined by the platform
     *         displaying the response card.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * The subtitle to display on the response card. The format of the subtitle is determined by the platform displaying
     * the response card.
     * </p>
     * 
     * @return The subtitle to display on the response card. The format of the subtitle is determined by the platform
     *         displaying the response card.
     */
    public final String subtitle() {
        return subtitle;
    }

    /**
     * <p>
     * The URL of an image to display on the response card. The image URL must be publicly available so that the
     * platform displaying the response card has access to the image.
     * </p>
     * 
     * @return The URL of an image to display on the response card. The image URL must be publicly available so that the
     *         platform displaying the response card has access to the image.
     */
    public final String imageUrl() {
        return imageUrl;
    }

    /**
     * For responses, this returns true if the service returned a value for the Buttons property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasButtons() {
        return buttons != null && !(buttons instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of buttons that should be displayed on the response card. The arrangement of the buttons is determined by
     * the platform that displays the button.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasButtons} method.
     * </p>
     * 
     * @return A list of buttons that should be displayed on the response card. The arrangement of the buttons is
     *         determined by the platform that displays the button.
     */
    public final List<Button> buttons() {
        return buttons;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(subtitle());
        hashCode = 31 * hashCode + Objects.hashCode(imageUrl());
        hashCode = 31 * hashCode + Objects.hashCode(hasButtons() ? buttons() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImageResponseCard)) {
            return false;
        }
        ImageResponseCard other = (ImageResponseCard) obj;
        return Objects.equals(title(), other.title()) && Objects.equals(subtitle(), other.subtitle())
                && Objects.equals(imageUrl(), other.imageUrl()) && hasButtons() == other.hasButtons()
                && Objects.equals(buttons(), other.buttons());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ImageResponseCard").add("Title", title()).add("Subtitle", subtitle())
                .add("ImageUrl", imageUrl()).add("Buttons", hasButtons() ? buttons() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "title":
            return Optional.ofNullable(clazz.cast(title()));
        case "subtitle":
            return Optional.ofNullable(clazz.cast(subtitle()));
        case "imageUrl":
            return Optional.ofNullable(clazz.cast(imageUrl()));
        case "buttons":
            return Optional.ofNullable(clazz.cast(buttons()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ImageResponseCard, T> g) {
        return obj -> g.apply((ImageResponseCard) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ImageResponseCard> {
        /**
         * <p>
         * The title to display on the response card. The format of the title is determined by the platform displaying
         * the response card.
         * </p>
         * 
         * @param title
         *        The title to display on the response card. The format of the title is determined by the platform
         *        displaying the response card.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * The subtitle to display on the response card. The format of the subtitle is determined by the platform
         * displaying the response card.
         * </p>
         * 
         * @param subtitle
         *        The subtitle to display on the response card. The format of the subtitle is determined by the platform
         *        displaying the response card.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subtitle(String subtitle);

        /**
         * <p>
         * The URL of an image to display on the response card. The image URL must be publicly available so that the
         * platform displaying the response card has access to the image.
         * </p>
         * 
         * @param imageUrl
         *        The URL of an image to display on the response card. The image URL must be publicly available so that
         *        the platform displaying the response card has access to the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageUrl(String imageUrl);

        /**
         * <p>
         * A list of buttons that should be displayed on the response card. The arrangement of the buttons is determined
         * by the platform that displays the button.
         * </p>
         * 
         * @param buttons
         *        A list of buttons that should be displayed on the response card. The arrangement of the buttons is
         *        determined by the platform that displays the button.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buttons(Collection<Button> buttons);

        /**
         * <p>
         * A list of buttons that should be displayed on the response card. The arrangement of the buttons is determined
         * by the platform that displays the button.
         * </p>
         * 
         * @param buttons
         *        A list of buttons that should be displayed on the response card. The arrangement of the buttons is
         *        determined by the platform that displays the button.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder buttons(Button... buttons);

        /**
         * <p>
         * A list of buttons that should be displayed on the response card. The arrangement of the buttons is determined
         * by the platform that displays the button.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Button>.Builder} avoiding the need to
         * create one manually via {@link List<Button>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Button>.Builder#build()} is called immediately and its
         * result is passed to {@link #buttons(List<Button>)}.
         * 
         * @param buttons
         *        a consumer that will call methods on {@link List<Button>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #buttons(List<Button>)
         */
        Builder buttons(Consumer<Button.Builder>... buttons);
    }

    static final class BuilderImpl implements Builder {
        private String title;

        private String subtitle;

        private String imageUrl;

        private List<Button> buttons = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ImageResponseCard model) {
            title(model.title);
            subtitle(model.subtitle);
            imageUrl(model.imageUrl);
            buttons(model.buttons);
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        @Transient
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final String getSubtitle() {
            return subtitle;
        }

        public final void setSubtitle(String subtitle) {
            this.subtitle = subtitle;
        }

        @Override
        @Transient
        public final Builder subtitle(String subtitle) {
            this.subtitle = subtitle;
            return this;
        }

        public final String getImageUrl() {
            return imageUrl;
        }

        public final void setImageUrl(String imageUrl) {
            this.imageUrl = imageUrl;
        }

        @Override
        @Transient
        public final Builder imageUrl(String imageUrl) {
            this.imageUrl = imageUrl;
            return this;
        }

        public final List<Button.Builder> getButtons() {
            List<Button.Builder> result = ButtonsListCopier.copyToBuilder(this.buttons);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setButtons(Collection<Button.BuilderImpl> buttons) {
            this.buttons = ButtonsListCopier.copyFromBuilder(buttons);
        }

        @Override
        @Transient
        public final Builder buttons(Collection<Button> buttons) {
            this.buttons = ButtonsListCopier.copy(buttons);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder buttons(Button... buttons) {
            buttons(Arrays.asList(buttons));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder buttons(Consumer<Button.Builder>... buttons) {
            buttons(Stream.of(buttons).map(c -> Button.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public ImageResponseCard build() {
            return new ImageResponseCard(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
