/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.memorydb.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * You create users and assign them specific permissions by using an access string. You assign the users to Access
 * Control Lists aligned with a specific role (administrators, human resources) that are then deployed to one or more
 * MemoryDB clusters.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class User implements SdkPojo, Serializable, ToCopyableBuilder<User.Builder, User> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(User::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(User::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> ACCESS_STRING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccessString").getter(getter(User::accessString)).setter(setter(Builder::accessString))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessString").build()).build();

    private static final SdkField<List<String>> ACL_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ACLNames")
            .getter(getter(User::aclNames))
            .setter(setter(Builder::aclNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ACLNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> MINIMUM_ENGINE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MinimumEngineVersion").getter(getter(User::minimumEngineVersion))
            .setter(setter(Builder::minimumEngineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinimumEngineVersion").build())
            .build();

    private static final SdkField<Authentication> AUTHENTICATION_FIELD = SdkField
            .<Authentication> builder(MarshallingType.SDK_POJO).memberName("Authentication").getter(getter(User::authentication))
            .setter(setter(Builder::authentication)).constructor(Authentication::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Authentication").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ARN")
            .getter(getter(User::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ARN").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, STATUS_FIELD,
            ACCESS_STRING_FIELD, ACL_NAMES_FIELD, MINIMUM_ENGINE_VERSION_FIELD, AUTHENTICATION_FIELD, ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String status;

    private final String accessString;

    private final List<String> aclNames;

    private final String minimumEngineVersion;

    private final Authentication authentication;

    private final String arn;

    private User(BuilderImpl builder) {
        this.name = builder.name;
        this.status = builder.status;
        this.accessString = builder.accessString;
        this.aclNames = builder.aclNames;
        this.minimumEngineVersion = builder.minimumEngineVersion;
        this.authentication = builder.authentication;
        this.arn = builder.arn;
    }

    /**
     * <p>
     * The name of the user
     * </p>
     * 
     * @return The name of the user
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Indicates the user status. Can be "active", "modifying" or "deleting".
     * </p>
     * 
     * @return Indicates the user status. Can be "active", "modifying" or "deleting".
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * Access permissions string used for this user.
     * </p>
     * 
     * @return Access permissions string used for this user.
     */
    public final String accessString() {
        return accessString;
    }

    /**
     * For responses, this returns true if the service returned a value for the ACLNames property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAclNames() {
        return aclNames != null && !(aclNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The names of the Access Control Lists to which the user belongs
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAclNames} method.
     * </p>
     * 
     * @return The names of the Access Control Lists to which the user belongs
     */
    public final List<String> aclNames() {
        return aclNames;
    }

    /**
     * <p>
     * The minimum engine version supported for the user
     * </p>
     * 
     * @return The minimum engine version supported for the user
     */
    public final String minimumEngineVersion() {
        return minimumEngineVersion;
    }

    /**
     * <p>
     * Denotes whether the user requires a password to authenticate.
     * </p>
     * 
     * @return Denotes whether the user requires a password to authenticate.
     */
    public final Authentication authentication() {
        return authentication;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the user.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the user.
     */
    public final String arn() {
        return arn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(accessString());
        hashCode = 31 * hashCode + Objects.hashCode(hasAclNames() ? aclNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(minimumEngineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(authentication());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof User)) {
            return false;
        }
        User other = (User) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(status(), other.status())
                && Objects.equals(accessString(), other.accessString()) && hasAclNames() == other.hasAclNames()
                && Objects.equals(aclNames(), other.aclNames())
                && Objects.equals(minimumEngineVersion(), other.minimumEngineVersion())
                && Objects.equals(authentication(), other.authentication()) && Objects.equals(arn(), other.arn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("User").add("Name", name()).add("Status", status()).add("AccessString", accessString())
                .add("ACLNames", hasAclNames() ? aclNames() : null).add("MinimumEngineVersion", minimumEngineVersion())
                .add("Authentication", authentication()).add("ARN", arn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "AccessString":
            return Optional.ofNullable(clazz.cast(accessString()));
        case "ACLNames":
            return Optional.ofNullable(clazz.cast(aclNames()));
        case "MinimumEngineVersion":
            return Optional.ofNullable(clazz.cast(minimumEngineVersion()));
        case "Authentication":
            return Optional.ofNullable(clazz.cast(authentication()));
        case "ARN":
            return Optional.ofNullable(clazz.cast(arn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<User, T> g) {
        return obj -> g.apply((User) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, User> {
        /**
         * <p>
         * The name of the user
         * </p>
         * 
         * @param name
         *        The name of the user
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Indicates the user status. Can be "active", "modifying" or "deleting".
         * </p>
         * 
         * @param status
         *        Indicates the user status. Can be "active", "modifying" or "deleting".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * Access permissions string used for this user.
         * </p>
         * 
         * @param accessString
         *        Access permissions string used for this user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessString(String accessString);

        /**
         * <p>
         * The names of the Access Control Lists to which the user belongs
         * </p>
         * 
         * @param aclNames
         *        The names of the Access Control Lists to which the user belongs
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aclNames(Collection<String> aclNames);

        /**
         * <p>
         * The names of the Access Control Lists to which the user belongs
         * </p>
         * 
         * @param aclNames
         *        The names of the Access Control Lists to which the user belongs
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder aclNames(String... aclNames);

        /**
         * <p>
         * The minimum engine version supported for the user
         * </p>
         * 
         * @param minimumEngineVersion
         *        The minimum engine version supported for the user
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minimumEngineVersion(String minimumEngineVersion);

        /**
         * <p>
         * Denotes whether the user requires a password to authenticate.
         * </p>
         * 
         * @param authentication
         *        Denotes whether the user requires a password to authenticate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authentication(Authentication authentication);

        /**
         * <p>
         * Denotes whether the user requires a password to authenticate.
         * </p>
         * This is a convenience that creates an instance of the {@link Authentication.Builder} avoiding the need to
         * create one manually via {@link Authentication#builder()}.
         *
         * When the {@link Consumer} completes, {@link Authentication.Builder#build()} is called immediately and its
         * result is passed to {@link #authentication(Authentication)}.
         * 
         * @param authentication
         *        a consumer that will call methods on {@link Authentication.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #authentication(Authentication)
         */
        default Builder authentication(Consumer<Authentication.Builder> authentication) {
            return authentication(Authentication.builder().applyMutation(authentication).build());
        }

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the user.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the user.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String status;

        private String accessString;

        private List<String> aclNames = DefaultSdkAutoConstructList.getInstance();

        private String minimumEngineVersion;

        private Authentication authentication;

        private String arn;

        private BuilderImpl() {
        }

        private BuilderImpl(User model) {
            name(model.name);
            status(model.status);
            accessString(model.accessString);
            aclNames(model.aclNames);
            minimumEngineVersion(model.minimumEngineVersion);
            authentication(model.authentication);
            arn(model.arn);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        @Transient
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        @Transient
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final String getAccessString() {
            return accessString;
        }

        public final void setAccessString(String accessString) {
            this.accessString = accessString;
        }

        @Override
        @Transient
        public final Builder accessString(String accessString) {
            this.accessString = accessString;
            return this;
        }

        public final Collection<String> getAclNames() {
            if (aclNames instanceof SdkAutoConstructList) {
                return null;
            }
            return aclNames;
        }

        public final void setAclNames(Collection<String> aclNames) {
            this.aclNames = ACLNameListCopier.copy(aclNames);
        }

        @Override
        @Transient
        public final Builder aclNames(Collection<String> aclNames) {
            this.aclNames = ACLNameListCopier.copy(aclNames);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder aclNames(String... aclNames) {
            aclNames(Arrays.asList(aclNames));
            return this;
        }

        public final String getMinimumEngineVersion() {
            return minimumEngineVersion;
        }

        public final void setMinimumEngineVersion(String minimumEngineVersion) {
            this.minimumEngineVersion = minimumEngineVersion;
        }

        @Override
        @Transient
        public final Builder minimumEngineVersion(String minimumEngineVersion) {
            this.minimumEngineVersion = minimumEngineVersion;
            return this;
        }

        public final Authentication.Builder getAuthentication() {
            return authentication != null ? authentication.toBuilder() : null;
        }

        public final void setAuthentication(Authentication.BuilderImpl authentication) {
            this.authentication = authentication != null ? authentication.build() : null;
        }

        @Override
        @Transient
        public final Builder authentication(Authentication authentication) {
            this.authentication = authentication;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        @Transient
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        @Override
        public User build() {
            return new User(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
