/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshift.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The request parameters to get cluster credentials.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetClusterCredentialsRequest extends RedshiftRequest implements
        ToCopyableBuilder<GetClusterCredentialsRequest.Builder, GetClusterCredentialsRequest> {
    private static final SdkField<String> DB_USER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("DbUser")
            .getter(getter(GetClusterCredentialsRequest::dbUser)).setter(setter(Builder::dbUser))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DbUser").build()).build();

    private static final SdkField<String> DB_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("DbName")
            .getter(getter(GetClusterCredentialsRequest::dbName)).setter(setter(Builder::dbName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DbName").build()).build();

    private static final SdkField<String> CLUSTER_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ClusterIdentifier").getter(getter(GetClusterCredentialsRequest::clusterIdentifier))
            .setter(setter(Builder::clusterIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClusterIdentifier").build()).build();

    private static final SdkField<Integer> DURATION_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("DurationSeconds").getter(getter(GetClusterCredentialsRequest::durationSeconds))
            .setter(setter(Builder::durationSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DurationSeconds").build()).build();

    private static final SdkField<Boolean> AUTO_CREATE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AutoCreate").getter(getter(GetClusterCredentialsRequest::autoCreate))
            .setter(setter(Builder::autoCreate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoCreate").build()).build();

    private static final SdkField<List<String>> DB_GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("DbGroups")
            .getter(getter(GetClusterCredentialsRequest::dbGroups))
            .setter(setter(Builder::dbGroups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DbGroups").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("DbGroup")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("DbGroup").build()).build()).build()).build();

    private static final SdkField<String> CUSTOM_DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CustomDomainName").getter(getter(GetClusterCredentialsRequest::customDomainName))
            .setter(setter(Builder::customDomainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomDomainName").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DB_USER_FIELD, DB_NAME_FIELD,
            CLUSTER_IDENTIFIER_FIELD, DURATION_SECONDS_FIELD, AUTO_CREATE_FIELD, DB_GROUPS_FIELD, CUSTOM_DOMAIN_NAME_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String dbUser;

    private final String dbName;

    private final String clusterIdentifier;

    private final Integer durationSeconds;

    private final Boolean autoCreate;

    private final List<String> dbGroups;

    private final String customDomainName;

    private GetClusterCredentialsRequest(BuilderImpl builder) {
        super(builder);
        this.dbUser = builder.dbUser;
        this.dbName = builder.dbName;
        this.clusterIdentifier = builder.clusterIdentifier;
        this.durationSeconds = builder.durationSeconds;
        this.autoCreate = builder.autoCreate;
        this.dbGroups = builder.dbGroups;
        this.customDomainName = builder.customDomainName;
    }

    /**
     * <p>
     * The name of a database user. If a user name matching <code>DbUser</code> exists in the database, the temporary
     * user credentials have the same permissions as the existing user. If <code>DbUser</code> doesn't exist in the
     * database and <code>Autocreate</code> is <code>True</code>, a new user is created using the value for
     * <code>DbUser</code> with PUBLIC permissions. If a database user matching the value for <code>DbUser</code>
     * doesn't exist and <code>Autocreate</code> is <code>False</code>, then the command succeeds but the connection
     * attempt will fail because the user doesn't exist in the database.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/redshift/latest/dg/r_CREATE_USER.html">CREATE
     * USER</a> in the Amazon Redshift Database Developer Guide.
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must be 1 to 64 alphanumeric characters or hyphens. The user name can't be <code>PUBLIC</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Must contain uppercase or lowercase letters, numbers, underscore, plus sign, period (dot), at symbol (@), or
     * hyphen.
     * </p>
     * </li>
     * <li>
     * <p>
     * First character must be a letter.
     * </p>
     * </li>
     * <li>
     * <p>
     * Must not contain a colon ( : ) or slash ( / ).
     * </p>
     * </li>
     * <li>
     * <p>
     * Cannot be a reserved word. A list of reserved words can be found in <a
     * href="http://docs.aws.amazon.com/redshift/latest/dg/r_pg_keywords.html">Reserved Words</a> in the Amazon Redshift
     * Database Developer Guide.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name of a database user. If a user name matching <code>DbUser</code> exists in the database, the
     *         temporary user credentials have the same permissions as the existing user. If <code>DbUser</code> doesn't
     *         exist in the database and <code>Autocreate</code> is <code>True</code>, a new user is created using the
     *         value for <code>DbUser</code> with PUBLIC permissions. If a database user matching the value for
     *         <code>DbUser</code> doesn't exist and <code>Autocreate</code> is <code>False</code>, then the command
     *         succeeds but the connection attempt will fail because the user doesn't exist in the database.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/redshift/latest/dg/r_CREATE_USER.html">CREATE USER</a> in the Amazon
     *         Redshift Database Developer Guide.
     *         </p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Must be 1 to 64 alphanumeric characters or hyphens. The user name can't be <code>PUBLIC</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must contain uppercase or lowercase letters, numbers, underscore, plus sign, period (dot), at symbol (@),
     *         or hyphen.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         First character must be a letter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must not contain a colon ( : ) or slash ( / ).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cannot be a reserved word. A list of reserved words can be found in <a
     *         href="http://docs.aws.amazon.com/redshift/latest/dg/r_pg_keywords.html">Reserved Words</a> in the Amazon
     *         Redshift Database Developer Guide.
     *         </p>
     *         </li>
     */
    public final String dbUser() {
        return dbUser;
    }

    /**
     * <p>
     * The name of a database that <code>DbUser</code> is authorized to log on to. If <code>DbName</code> is not
     * specified, <code>DbUser</code> can log on to any existing database.
     * </p>
     * <p>
     * Constraints:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must be 1 to 64 alphanumeric characters or hyphens
     * </p>
     * </li>
     * <li>
     * <p>
     * Must contain uppercase or lowercase letters, numbers, underscore, plus sign, period (dot), at symbol (@), or
     * hyphen.
     * </p>
     * </li>
     * <li>
     * <p>
     * First character must be a letter.
     * </p>
     * </li>
     * <li>
     * <p>
     * Must not contain a colon ( : ) or slash ( / ).
     * </p>
     * </li>
     * <li>
     * <p>
     * Cannot be a reserved word. A list of reserved words can be found in <a
     * href="http://docs.aws.amazon.com/redshift/latest/dg/r_pg_keywords.html">Reserved Words</a> in the Amazon Redshift
     * Database Developer Guide.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The name of a database that <code>DbUser</code> is authorized to log on to. If <code>DbName</code> is not
     *         specified, <code>DbUser</code> can log on to any existing database.</p>
     *         <p>
     *         Constraints:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Must be 1 to 64 alphanumeric characters or hyphens
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must contain uppercase or lowercase letters, numbers, underscore, plus sign, period (dot), at symbol (@),
     *         or hyphen.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         First character must be a letter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must not contain a colon ( : ) or slash ( / ).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cannot be a reserved word. A list of reserved words can be found in <a
     *         href="http://docs.aws.amazon.com/redshift/latest/dg/r_pg_keywords.html">Reserved Words</a> in the Amazon
     *         Redshift Database Developer Guide.
     *         </p>
     *         </li>
     */
    public final String dbName() {
        return dbName;
    }

    /**
     * <p>
     * The unique identifier of the cluster that contains the database for which you are requesting credentials. This
     * parameter is case sensitive.
     * </p>
     * 
     * @return The unique identifier of the cluster that contains the database for which you are requesting credentials.
     *         This parameter is case sensitive.
     */
    public final String clusterIdentifier() {
        return clusterIdentifier;
    }

    /**
     * <p>
     * The number of seconds until the returned temporary password expires.
     * </p>
     * <p>
     * Constraint: minimum 900, maximum 3600.
     * </p>
     * <p>
     * Default: 900
     * </p>
     * 
     * @return The number of seconds until the returned temporary password expires.</p>
     *         <p>
     *         Constraint: minimum 900, maximum 3600.
     *         </p>
     *         <p>
     *         Default: 900
     */
    public final Integer durationSeconds() {
        return durationSeconds;
    }

    /**
     * <p>
     * Create a database user with the name specified for the user named in <code>DbUser</code> if one does not exist.
     * </p>
     * 
     * @return Create a database user with the name specified for the user named in <code>DbUser</code> if one does not
     *         exist.
     */
    public final Boolean autoCreate() {
        return autoCreate;
    }

    /**
     * For responses, this returns true if the service returned a value for the DbGroups property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDbGroups() {
        return dbGroups != null && !(dbGroups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the names of existing database groups that the user named in <code>DbUser</code> will join for the
     * current session, in addition to any group memberships for an existing user. If not specified, a new user is added
     * only to PUBLIC.
     * </p>
     * <p>
     * Database group name constraints
     * </p>
     * <ul>
     * <li>
     * <p>
     * Must be 1 to 64 alphanumeric characters or hyphens
     * </p>
     * </li>
     * <li>
     * <p>
     * Must contain only lowercase letters, numbers, underscore, plus sign, period (dot), at symbol (@), or hyphen.
     * </p>
     * </li>
     * <li>
     * <p>
     * First character must be a letter.
     * </p>
     * </li>
     * <li>
     * <p>
     * Must not contain a colon ( : ) or slash ( / ).
     * </p>
     * </li>
     * <li>
     * <p>
     * Cannot be a reserved word. A list of reserved words can be found in <a
     * href="http://docs.aws.amazon.com/redshift/latest/dg/r_pg_keywords.html">Reserved Words</a> in the Amazon Redshift
     * Database Developer Guide.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDbGroups} method.
     * </p>
     * 
     * @return A list of the names of existing database groups that the user named in <code>DbUser</code> will join for
     *         the current session, in addition to any group memberships for an existing user. If not specified, a new
     *         user is added only to PUBLIC.</p>
     *         <p>
     *         Database group name constraints
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Must be 1 to 64 alphanumeric characters or hyphens
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must contain only lowercase letters, numbers, underscore, plus sign, period (dot), at symbol (@), or
     *         hyphen.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         First character must be a letter.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Must not contain a colon ( : ) or slash ( / ).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Cannot be a reserved word. A list of reserved words can be found in <a
     *         href="http://docs.aws.amazon.com/redshift/latest/dg/r_pg_keywords.html">Reserved Words</a> in the Amazon
     *         Redshift Database Developer Guide.
     *         </p>
     *         </li>
     */
    public final List<String> dbGroups() {
        return dbGroups;
    }

    /**
     * <p>
     * The custom domain name for the cluster credentials.
     * </p>
     * 
     * @return The custom domain name for the cluster credentials.
     */
    public final String customDomainName() {
        return customDomainName;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(dbUser());
        hashCode = 31 * hashCode + Objects.hashCode(dbName());
        hashCode = 31 * hashCode + Objects.hashCode(clusterIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(durationSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(autoCreate());
        hashCode = 31 * hashCode + Objects.hashCode(hasDbGroups() ? dbGroups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(customDomainName());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetClusterCredentialsRequest)) {
            return false;
        }
        GetClusterCredentialsRequest other = (GetClusterCredentialsRequest) obj;
        return Objects.equals(dbUser(), other.dbUser()) && Objects.equals(dbName(), other.dbName())
                && Objects.equals(clusterIdentifier(), other.clusterIdentifier())
                && Objects.equals(durationSeconds(), other.durationSeconds()) && Objects.equals(autoCreate(), other.autoCreate())
                && hasDbGroups() == other.hasDbGroups() && Objects.equals(dbGroups(), other.dbGroups())
                && Objects.equals(customDomainName(), other.customDomainName());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GetClusterCredentialsRequest").add("DbUser", dbUser()).add("DbName", dbName())
                .add("ClusterIdentifier", clusterIdentifier()).add("DurationSeconds", durationSeconds())
                .add("AutoCreate", autoCreate()).add("DbGroups", hasDbGroups() ? dbGroups() : null)
                .add("CustomDomainName", customDomainName()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DbUser":
            return Optional.ofNullable(clazz.cast(dbUser()));
        case "DbName":
            return Optional.ofNullable(clazz.cast(dbName()));
        case "ClusterIdentifier":
            return Optional.ofNullable(clazz.cast(clusterIdentifier()));
        case "DurationSeconds":
            return Optional.ofNullable(clazz.cast(durationSeconds()));
        case "AutoCreate":
            return Optional.ofNullable(clazz.cast(autoCreate()));
        case "DbGroups":
            return Optional.ofNullable(clazz.cast(dbGroups()));
        case "CustomDomainName":
            return Optional.ofNullable(clazz.cast(customDomainName()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("DbUser", DB_USER_FIELD);
        map.put("DbName", DB_NAME_FIELD);
        map.put("ClusterIdentifier", CLUSTER_IDENTIFIER_FIELD);
        map.put("DurationSeconds", DURATION_SECONDS_FIELD);
        map.put("AutoCreate", AUTO_CREATE_FIELD);
        map.put("DbGroups", DB_GROUPS_FIELD);
        map.put("CustomDomainName", CUSTOM_DOMAIN_NAME_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GetClusterCredentialsRequest, T> g) {
        return obj -> g.apply((GetClusterCredentialsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends RedshiftRequest.Builder, SdkPojo, CopyableBuilder<Builder, GetClusterCredentialsRequest> {
        /**
         * <p>
         * The name of a database user. If a user name matching <code>DbUser</code> exists in the database, the
         * temporary user credentials have the same permissions as the existing user. If <code>DbUser</code> doesn't
         * exist in the database and <code>Autocreate</code> is <code>True</code>, a new user is created using the value
         * for <code>DbUser</code> with PUBLIC permissions. If a database user matching the value for
         * <code>DbUser</code> doesn't exist and <code>Autocreate</code> is <code>False</code>, then the command
         * succeeds but the connection attempt will fail because the user doesn't exist in the database.
         * </p>
         * <p>
         * For more information, see <a href="https://docs.aws.amazon.com/redshift/latest/dg/r_CREATE_USER.html">CREATE
         * USER</a> in the Amazon Redshift Database Developer Guide.
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must be 1 to 64 alphanumeric characters or hyphens. The user name can't be <code>PUBLIC</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Must contain uppercase or lowercase letters, numbers, underscore, plus sign, period (dot), at symbol (@), or
         * hyphen.
         * </p>
         * </li>
         * <li>
         * <p>
         * First character must be a letter.
         * </p>
         * </li>
         * <li>
         * <p>
         * Must not contain a colon ( : ) or slash ( / ).
         * </p>
         * </li>
         * <li>
         * <p>
         * Cannot be a reserved word. A list of reserved words can be found in <a
         * href="http://docs.aws.amazon.com/redshift/latest/dg/r_pg_keywords.html">Reserved Words</a> in the Amazon
         * Redshift Database Developer Guide.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dbUser
         *        The name of a database user. If a user name matching <code>DbUser</code> exists in the database, the
         *        temporary user credentials have the same permissions as the existing user. If <code>DbUser</code>
         *        doesn't exist in the database and <code>Autocreate</code> is <code>True</code>, a new user is created
         *        using the value for <code>DbUser</code> with PUBLIC permissions. If a database user matching the value
         *        for <code>DbUser</code> doesn't exist and <code>Autocreate</code> is <code>False</code>, then the
         *        command succeeds but the connection attempt will fail because the user doesn't exist in the
         *        database.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/redshift/latest/dg/r_CREATE_USER.html">CREATE USER</a> in the Amazon
         *        Redshift Database Developer Guide.
         *        </p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must be 1 to 64 alphanumeric characters or hyphens. The user name can't be <code>PUBLIC</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must contain uppercase or lowercase letters, numbers, underscore, plus sign, period (dot), at symbol
         *        (@), or hyphen.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        First character must be a letter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must not contain a colon ( : ) or slash ( / ).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cannot be a reserved word. A list of reserved words can be found in <a
         *        href="http://docs.aws.amazon.com/redshift/latest/dg/r_pg_keywords.html">Reserved Words</a> in the
         *        Amazon Redshift Database Developer Guide.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbUser(String dbUser);

        /**
         * <p>
         * The name of a database that <code>DbUser</code> is authorized to log on to. If <code>DbName</code> is not
         * specified, <code>DbUser</code> can log on to any existing database.
         * </p>
         * <p>
         * Constraints:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must be 1 to 64 alphanumeric characters or hyphens
         * </p>
         * </li>
         * <li>
         * <p>
         * Must contain uppercase or lowercase letters, numbers, underscore, plus sign, period (dot), at symbol (@), or
         * hyphen.
         * </p>
         * </li>
         * <li>
         * <p>
         * First character must be a letter.
         * </p>
         * </li>
         * <li>
         * <p>
         * Must not contain a colon ( : ) or slash ( / ).
         * </p>
         * </li>
         * <li>
         * <p>
         * Cannot be a reserved word. A list of reserved words can be found in <a
         * href="http://docs.aws.amazon.com/redshift/latest/dg/r_pg_keywords.html">Reserved Words</a> in the Amazon
         * Redshift Database Developer Guide.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dbName
         *        The name of a database that <code>DbUser</code> is authorized to log on to. If <code>DbName</code> is
         *        not specified, <code>DbUser</code> can log on to any existing database.</p>
         *        <p>
         *        Constraints:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must be 1 to 64 alphanumeric characters or hyphens
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must contain uppercase or lowercase letters, numbers, underscore, plus sign, period (dot), at symbol
         *        (@), or hyphen.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        First character must be a letter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must not contain a colon ( : ) or slash ( / ).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cannot be a reserved word. A list of reserved words can be found in <a
         *        href="http://docs.aws.amazon.com/redshift/latest/dg/r_pg_keywords.html">Reserved Words</a> in the
         *        Amazon Redshift Database Developer Guide.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbName(String dbName);

        /**
         * <p>
         * The unique identifier of the cluster that contains the database for which you are requesting credentials.
         * This parameter is case sensitive.
         * </p>
         * 
         * @param clusterIdentifier
         *        The unique identifier of the cluster that contains the database for which you are requesting
         *        credentials. This parameter is case sensitive.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterIdentifier(String clusterIdentifier);

        /**
         * <p>
         * The number of seconds until the returned temporary password expires.
         * </p>
         * <p>
         * Constraint: minimum 900, maximum 3600.
         * </p>
         * <p>
         * Default: 900
         * </p>
         * 
         * @param durationSeconds
         *        The number of seconds until the returned temporary password expires.</p>
         *        <p>
         *        Constraint: minimum 900, maximum 3600.
         *        </p>
         *        <p>
         *        Default: 900
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder durationSeconds(Integer durationSeconds);

        /**
         * <p>
         * Create a database user with the name specified for the user named in <code>DbUser</code> if one does not
         * exist.
         * </p>
         * 
         * @param autoCreate
         *        Create a database user with the name specified for the user named in <code>DbUser</code> if one does
         *        not exist.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoCreate(Boolean autoCreate);

        /**
         * <p>
         * A list of the names of existing database groups that the user named in <code>DbUser</code> will join for the
         * current session, in addition to any group memberships for an existing user. If not specified, a new user is
         * added only to PUBLIC.
         * </p>
         * <p>
         * Database group name constraints
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must be 1 to 64 alphanumeric characters or hyphens
         * </p>
         * </li>
         * <li>
         * <p>
         * Must contain only lowercase letters, numbers, underscore, plus sign, period (dot), at symbol (@), or hyphen.
         * </p>
         * </li>
         * <li>
         * <p>
         * First character must be a letter.
         * </p>
         * </li>
         * <li>
         * <p>
         * Must not contain a colon ( : ) or slash ( / ).
         * </p>
         * </li>
         * <li>
         * <p>
         * Cannot be a reserved word. A list of reserved words can be found in <a
         * href="http://docs.aws.amazon.com/redshift/latest/dg/r_pg_keywords.html">Reserved Words</a> in the Amazon
         * Redshift Database Developer Guide.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dbGroups
         *        A list of the names of existing database groups that the user named in <code>DbUser</code> will join
         *        for the current session, in addition to any group memberships for an existing user. If not specified,
         *        a new user is added only to PUBLIC.</p>
         *        <p>
         *        Database group name constraints
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must be 1 to 64 alphanumeric characters or hyphens
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must contain only lowercase letters, numbers, underscore, plus sign, period (dot), at symbol (@), or
         *        hyphen.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        First character must be a letter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must not contain a colon ( : ) or slash ( / ).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cannot be a reserved word. A list of reserved words can be found in <a
         *        href="http://docs.aws.amazon.com/redshift/latest/dg/r_pg_keywords.html">Reserved Words</a> in the
         *        Amazon Redshift Database Developer Guide.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbGroups(Collection<String> dbGroups);

        /**
         * <p>
         * A list of the names of existing database groups that the user named in <code>DbUser</code> will join for the
         * current session, in addition to any group memberships for an existing user. If not specified, a new user is
         * added only to PUBLIC.
         * </p>
         * <p>
         * Database group name constraints
         * </p>
         * <ul>
         * <li>
         * <p>
         * Must be 1 to 64 alphanumeric characters or hyphens
         * </p>
         * </li>
         * <li>
         * <p>
         * Must contain only lowercase letters, numbers, underscore, plus sign, period (dot), at symbol (@), or hyphen.
         * </p>
         * </li>
         * <li>
         * <p>
         * First character must be a letter.
         * </p>
         * </li>
         * <li>
         * <p>
         * Must not contain a colon ( : ) or slash ( / ).
         * </p>
         * </li>
         * <li>
         * <p>
         * Cannot be a reserved word. A list of reserved words can be found in <a
         * href="http://docs.aws.amazon.com/redshift/latest/dg/r_pg_keywords.html">Reserved Words</a> in the Amazon
         * Redshift Database Developer Guide.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dbGroups
         *        A list of the names of existing database groups that the user named in <code>DbUser</code> will join
         *        for the current session, in addition to any group memberships for an existing user. If not specified,
         *        a new user is added only to PUBLIC.</p>
         *        <p>
         *        Database group name constraints
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Must be 1 to 64 alphanumeric characters or hyphens
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must contain only lowercase letters, numbers, underscore, plus sign, period (dot), at symbol (@), or
         *        hyphen.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        First character must be a letter.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Must not contain a colon ( : ) or slash ( / ).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Cannot be a reserved word. A list of reserved words can be found in <a
         *        href="http://docs.aws.amazon.com/redshift/latest/dg/r_pg_keywords.html">Reserved Words</a> in the
         *        Amazon Redshift Database Developer Guide.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dbGroups(String... dbGroups);

        /**
         * <p>
         * The custom domain name for the cluster credentials.
         * </p>
         * 
         * @param customDomainName
         *        The custom domain name for the cluster credentials.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customDomainName(String customDomainName);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends RedshiftRequest.BuilderImpl implements Builder {
        private String dbUser;

        private String dbName;

        private String clusterIdentifier;

        private Integer durationSeconds;

        private Boolean autoCreate;

        private List<String> dbGroups = DefaultSdkAutoConstructList.getInstance();

        private String customDomainName;

        private BuilderImpl() {
        }

        private BuilderImpl(GetClusterCredentialsRequest model) {
            super(model);
            dbUser(model.dbUser);
            dbName(model.dbName);
            clusterIdentifier(model.clusterIdentifier);
            durationSeconds(model.durationSeconds);
            autoCreate(model.autoCreate);
            dbGroups(model.dbGroups);
            customDomainName(model.customDomainName);
        }

        public final String getDbUser() {
            return dbUser;
        }

        public final void setDbUser(String dbUser) {
            this.dbUser = dbUser;
        }

        @Override
        public final Builder dbUser(String dbUser) {
            this.dbUser = dbUser;
            return this;
        }

        public final String getDbName() {
            return dbName;
        }

        public final void setDbName(String dbName) {
            this.dbName = dbName;
        }

        @Override
        public final Builder dbName(String dbName) {
            this.dbName = dbName;
            return this;
        }

        public final String getClusterIdentifier() {
            return clusterIdentifier;
        }

        public final void setClusterIdentifier(String clusterIdentifier) {
            this.clusterIdentifier = clusterIdentifier;
        }

        @Override
        public final Builder clusterIdentifier(String clusterIdentifier) {
            this.clusterIdentifier = clusterIdentifier;
            return this;
        }

        public final Integer getDurationSeconds() {
            return durationSeconds;
        }

        public final void setDurationSeconds(Integer durationSeconds) {
            this.durationSeconds = durationSeconds;
        }

        @Override
        public final Builder durationSeconds(Integer durationSeconds) {
            this.durationSeconds = durationSeconds;
            return this;
        }

        public final Boolean getAutoCreate() {
            return autoCreate;
        }

        public final void setAutoCreate(Boolean autoCreate) {
            this.autoCreate = autoCreate;
        }

        @Override
        public final Builder autoCreate(Boolean autoCreate) {
            this.autoCreate = autoCreate;
            return this;
        }

        public final Collection<String> getDbGroups() {
            if (dbGroups instanceof SdkAutoConstructList) {
                return null;
            }
            return dbGroups;
        }

        public final void setDbGroups(Collection<String> dbGroups) {
            this.dbGroups = DbGroupListCopier.copy(dbGroups);
        }

        @Override
        public final Builder dbGroups(Collection<String> dbGroups) {
            this.dbGroups = DbGroupListCopier.copy(dbGroups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dbGroups(String... dbGroups) {
            dbGroups(Arrays.asList(dbGroups));
            return this;
        }

        public final String getCustomDomainName() {
            return customDomainName;
        }

        public final void setCustomDomainName(String customDomainName) {
            this.customDomainName = customDomainName;
        }

        @Override
        public final Builder customDomainName(String customDomainName) {
            this.customDomainName = customDomainName;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetClusterCredentialsRequest build() {
            return new GetClusterCredentialsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
