/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.route53recoveryreadiness;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.route53recoveryreadiness.model.AccessDeniedException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ConflictException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateCellRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateCellResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateCrossAccountAuthorizationRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateCrossAccountAuthorizationResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateReadinessCheckRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateReadinessCheckResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateRecoveryGroupRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateRecoveryGroupResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateResourceSetRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.CreateResourceSetResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteCellRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteCellResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteCrossAccountAuthorizationRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteCrossAccountAuthorizationResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteReadinessCheckRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteReadinessCheckResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteRecoveryGroupRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteRecoveryGroupResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteResourceSetRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.DeleteResourceSetResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetArchitectureRecommendationsRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetArchitectureRecommendationsResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetCellReadinessSummaryRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetCellReadinessSummaryResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetCellRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetCellResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckResourceStatusRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckResourceStatusResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckStatusRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckStatusResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetRecoveryGroupReadinessSummaryRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetRecoveryGroupReadinessSummaryResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetRecoveryGroupRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetRecoveryGroupResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetResourceSetRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.GetResourceSetResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.InternalServerException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListCellsRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListCellsResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListCrossAccountAuthorizationsRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListCrossAccountAuthorizationsResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListReadinessChecksRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListReadinessChecksResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListRecoveryGroupsRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListRecoveryGroupsResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListResourceSetsRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListResourceSetsResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListRulesRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListRulesResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListTagsForResourcesRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ListTagsForResourcesResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ResourceNotFoundException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.Route53RecoveryReadinessException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.Route53RecoveryReadinessRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.TagResourceRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.TagResourceResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ThrottlingException;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UntagResourceRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UntagResourceResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateCellRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateCellResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateReadinessCheckRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateReadinessCheckResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateRecoveryGroupRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateRecoveryGroupResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateResourceSetRequest;
import software.amazon.awssdk.services.route53recoveryreadiness.model.UpdateResourceSetResponse;
import software.amazon.awssdk.services.route53recoveryreadiness.model.ValidationException;
import software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetCellReadinessSummaryIterable;
import software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetReadinessCheckResourceStatusIterable;
import software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetReadinessCheckStatusIterable;
import software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetRecoveryGroupReadinessSummaryIterable;
import software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListCellsIterable;
import software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListCrossAccountAuthorizationsIterable;
import software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListReadinessChecksIterable;
import software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListRecoveryGroupsIterable;
import software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListResourceSetsIterable;
import software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListRulesIterable;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.CreateCellRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.CreateCrossAccountAuthorizationRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.CreateReadinessCheckRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.CreateRecoveryGroupRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.CreateResourceSetRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.DeleteCellRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.DeleteCrossAccountAuthorizationRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.DeleteReadinessCheckRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.DeleteRecoveryGroupRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.DeleteResourceSetRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetArchitectureRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetCellReadinessSummaryRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetCellRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetReadinessCheckRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetReadinessCheckResourceStatusRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetReadinessCheckStatusRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetRecoveryGroupReadinessSummaryRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetRecoveryGroupRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.GetResourceSetRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListCellsRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListCrossAccountAuthorizationsRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListReadinessChecksRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListRecoveryGroupsRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListResourceSetsRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListRulesRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.ListTagsForResourcesRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.UpdateCellRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.UpdateReadinessCheckRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.UpdateRecoveryGroupRequestMarshaller;
import software.amazon.awssdk.services.route53recoveryreadiness.transform.UpdateResourceSetRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link Route53RecoveryReadinessClient}.
 *
 * @see Route53RecoveryReadinessClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultRoute53RecoveryReadinessClient implements Route53RecoveryReadinessClient {
    private static final Logger log = Logger.loggerFor(DefaultRoute53RecoveryReadinessClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultRoute53RecoveryReadinessClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * Creates a new Cell.
     *
     * @param createCellRequest
     *        The Cell to create
     * @return Result of the CreateCell operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.CreateCell
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/CreateCell"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateCellResponse createCell(CreateCellRequest createCellRequest) throws ThrottlingException, ValidationException,
            InternalServerException, ConflictException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCellResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateCellResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCellRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCell");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateCellRequest, CreateCellResponse>().withOperationName("CreateCell")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createCellRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateCellRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create a new cross account readiness authorization.
     *
     * @param createCrossAccountAuthorizationRequest
     *        The cross account authorization
     * @return Result of the CreateCrossAccountAuthorization operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.CreateCrossAccountAuthorization
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/CreateCrossAccountAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateCrossAccountAuthorizationResponse createCrossAccountAuthorization(
            CreateCrossAccountAuthorizationRequest createCrossAccountAuthorizationRequest) throws ThrottlingException,
            ValidationException, InternalServerException, ConflictException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCrossAccountAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateCrossAccountAuthorizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createCrossAccountAuthorizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCrossAccountAuthorization");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateCrossAccountAuthorizationRequest, CreateCrossAccountAuthorizationResponse>()
                            .withOperationName("CreateCrossAccountAuthorization").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createCrossAccountAuthorizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateCrossAccountAuthorizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Creates a new Readiness Check.
     *
     * @param createReadinessCheckRequest
     *        The ReadinessCheck to create
     * @return Result of the CreateReadinessCheck operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.CreateReadinessCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/CreateReadinessCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateReadinessCheckResponse createReadinessCheck(CreateReadinessCheckRequest createReadinessCheckRequest)
            throws ThrottlingException, ValidationException, InternalServerException, ConflictException, AccessDeniedException,
            AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateReadinessCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateReadinessCheckResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createReadinessCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateReadinessCheck");

            return clientHandler.execute(new ClientExecutionParams<CreateReadinessCheckRequest, CreateReadinessCheckResponse>()
                    .withOperationName("CreateReadinessCheck").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createReadinessCheckRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateReadinessCheckRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Creates a new Recovery Group.
     *
     * @param createRecoveryGroupRequest
     *        The RecoveryGroup to create
     * @return Result of the CreateRecoveryGroup operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.CreateRecoveryGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/CreateRecoveryGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateRecoveryGroupResponse createRecoveryGroup(CreateRecoveryGroupRequest createRecoveryGroupRequest)
            throws ThrottlingException, ValidationException, InternalServerException, ConflictException, AccessDeniedException,
            AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRecoveryGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateRecoveryGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRecoveryGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRecoveryGroup");

            return clientHandler.execute(new ClientExecutionParams<CreateRecoveryGroupRequest, CreateRecoveryGroupResponse>()
                    .withOperationName("CreateRecoveryGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createRecoveryGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateRecoveryGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Creates a new Resource Set.
     *
     * @param createResourceSetRequest
     *        The ResourceSet to create
     * @return Result of the CreateResourceSet operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws ConflictException
     *         Updating or deleting a resource can cause an inconsistent state.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.CreateResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/CreateResourceSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateResourceSetResponse createResourceSet(CreateResourceSetRequest createResourceSetRequest)
            throws ThrottlingException, ValidationException, InternalServerException, ConflictException, AccessDeniedException,
            AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateResourceSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResourceSet");

            return clientHandler.execute(new ClientExecutionParams<CreateResourceSetRequest, CreateResourceSetResponse>()
                    .withOperationName("CreateResourceSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createResourceSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateResourceSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Deletes an existing Cell.
     *
     * @param deleteCellRequest
     * @return Result of the DeleteCell operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.DeleteCell
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/DeleteCell"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteCellResponse deleteCell(DeleteCellRequest deleteCellRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCellResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteCellResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCellRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCell");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteCellRequest, DeleteCellResponse>().withOperationName("DeleteCell")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteCellRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteCellRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Delete cross account readiness authorization
     *
     * @param deleteCrossAccountAuthorizationRequest
     * @return Result of the DeleteCrossAccountAuthorization operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.DeleteCrossAccountAuthorization
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/DeleteCrossAccountAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteCrossAccountAuthorizationResponse deleteCrossAccountAuthorization(
            DeleteCrossAccountAuthorizationRequest deleteCrossAccountAuthorizationRequest) throws ThrottlingException,
            ValidationException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCrossAccountAuthorizationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteCrossAccountAuthorizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteCrossAccountAuthorizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCrossAccountAuthorization");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteCrossAccountAuthorizationRequest, DeleteCrossAccountAuthorizationResponse>()
                            .withOperationName("DeleteCrossAccountAuthorization").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteCrossAccountAuthorizationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteCrossAccountAuthorizationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Deletes an existing Readiness Check.
     *
     * @param deleteReadinessCheckRequest
     * @return Result of the DeleteReadinessCheck operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.DeleteReadinessCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/DeleteReadinessCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteReadinessCheckResponse deleteReadinessCheck(DeleteReadinessCheckRequest deleteReadinessCheckRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteReadinessCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteReadinessCheckResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteReadinessCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteReadinessCheck");

            return clientHandler.execute(new ClientExecutionParams<DeleteReadinessCheckRequest, DeleteReadinessCheckResponse>()
                    .withOperationName("DeleteReadinessCheck").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteReadinessCheckRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteReadinessCheckRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Deletes an existing Recovery Group.
     *
     * @param deleteRecoveryGroupRequest
     * @return Result of the DeleteRecoveryGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.DeleteRecoveryGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/DeleteRecoveryGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteRecoveryGroupResponse deleteRecoveryGroup(DeleteRecoveryGroupRequest deleteRecoveryGroupRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRecoveryGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRecoveryGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRecoveryGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRecoveryGroup");

            return clientHandler.execute(new ClientExecutionParams<DeleteRecoveryGroupRequest, DeleteRecoveryGroupResponse>()
                    .withOperationName("DeleteRecoveryGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteRecoveryGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteRecoveryGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Deletes an existing Resource Set.
     *
     * @param deleteResourceSetRequest
     * @return Result of the DeleteResourceSet operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.DeleteResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/DeleteResourceSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteResourceSetResponse deleteResourceSet(DeleteResourceSetRequest deleteResourceSetRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteResourceSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourceSet");

            return clientHandler.execute(new ClientExecutionParams<DeleteResourceSetRequest, DeleteResourceSetResponse>()
                    .withOperationName("DeleteResourceSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteResourceSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteResourceSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a collection of recommendations to improve resilliance and readiness check quality for a Recovery Group.
     *
     * @param getArchitectureRecommendationsRequest
     * @return Result of the GetArchitectureRecommendations operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetArchitectureRecommendations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetArchitectureRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetArchitectureRecommendationsResponse getArchitectureRecommendations(
            GetArchitectureRecommendationsRequest getArchitectureRecommendationsRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetArchitectureRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetArchitectureRecommendationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getArchitectureRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetArchitectureRecommendations");

            return clientHandler
                    .execute(new ClientExecutionParams<GetArchitectureRecommendationsRequest, GetArchitectureRecommendationsResponse>()
                            .withOperationName("GetArchitectureRecommendations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getArchitectureRecommendationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetArchitectureRecommendationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns information about a Cell.
     *
     * @param getCellRequest
     * @return Result of the GetCell operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetCell
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetCell"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetCellResponse getCell(GetCellRequest getCellRequest) throws ResourceNotFoundException, ThrottlingException,
            ValidationException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCellResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetCellResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCellRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCell");

            return clientHandler.execute(new ClientExecutionParams<GetCellRequest, GetCellResponse>()
                    .withOperationName("GetCell").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getCellRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetCellRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns information about readiness of a Cell.
     *
     * @param getCellReadinessSummaryRequest
     * @return Result of the GetCellReadinessSummary operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetCellReadinessSummary
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetCellReadinessSummary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetCellReadinessSummaryResponse getCellReadinessSummary(GetCellReadinessSummaryRequest getCellReadinessSummaryRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCellReadinessSummaryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetCellReadinessSummaryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCellReadinessSummaryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCellReadinessSummary");

            return clientHandler
                    .execute(new ClientExecutionParams<GetCellReadinessSummaryRequest, GetCellReadinessSummaryResponse>()
                            .withOperationName("GetCellReadinessSummary").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getCellReadinessSummaryRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetCellReadinessSummaryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns information about readiness of a Cell.<br/>
     * <p>
     * This is a variant of
     * {@link #getCellReadinessSummary(software.amazon.awssdk.services.route53recoveryreadiness.model.GetCellReadinessSummaryRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetCellReadinessSummaryIterable responses = client.getCellReadinessSummaryPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetCellReadinessSummaryIterable responses = client
     *             .getCellReadinessSummaryPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoveryreadiness.model.GetCellReadinessSummaryResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetCellReadinessSummaryIterable responses = client.getCellReadinessSummaryPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getCellReadinessSummary(software.amazon.awssdk.services.route53recoveryreadiness.model.GetCellReadinessSummaryRequest)}
     * operation.</b>
     * </p>
     *
     * @param getCellReadinessSummaryRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetCellReadinessSummary
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetCellReadinessSummary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetCellReadinessSummaryIterable getCellReadinessSummaryPaginator(
            GetCellReadinessSummaryRequest getCellReadinessSummaryRequest) throws ResourceNotFoundException, ThrottlingException,
            ValidationException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        return new GetCellReadinessSummaryIterable(this, applyPaginatorUserAgent(getCellReadinessSummaryRequest));
    }

    /**
     * Returns information about a ReadinessCheck.
     *
     * @param getReadinessCheckRequest
     * @return Result of the GetReadinessCheck operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetReadinessCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetReadinessCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetReadinessCheckResponse getReadinessCheck(GetReadinessCheckRequest getReadinessCheckRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetReadinessCheckResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetReadinessCheckResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getReadinessCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetReadinessCheck");

            return clientHandler.execute(new ClientExecutionParams<GetReadinessCheckRequest, GetReadinessCheckResponse>()
                    .withOperationName("GetReadinessCheck").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getReadinessCheckRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetReadinessCheckRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns detailed information about the status of an individual resource within a Readiness Check's Resource Set.
     *
     * @param getReadinessCheckResourceStatusRequest
     * @return Result of the GetReadinessCheckResourceStatus operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetReadinessCheckResourceStatus
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetReadinessCheckResourceStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetReadinessCheckResourceStatusResponse getReadinessCheckResourceStatus(
            GetReadinessCheckResourceStatusRequest getReadinessCheckResourceStatusRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetReadinessCheckResourceStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetReadinessCheckResourceStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getReadinessCheckResourceStatusRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetReadinessCheckResourceStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<GetReadinessCheckResourceStatusRequest, GetReadinessCheckResourceStatusResponse>()
                            .withOperationName("GetReadinessCheckResourceStatus").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getReadinessCheckResourceStatusRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetReadinessCheckResourceStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns detailed information about the status of an individual resource within a Readiness Check's Resource Set.<br/>
     * <p>
     * This is a variant of
     * {@link #getReadinessCheckResourceStatus(software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckResourceStatusRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetReadinessCheckResourceStatusIterable responses = client.getReadinessCheckResourceStatusPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetReadinessCheckResourceStatusIterable responses = client
     *             .getReadinessCheckResourceStatusPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckResourceStatusResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetReadinessCheckResourceStatusIterable responses = client.getReadinessCheckResourceStatusPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getReadinessCheckResourceStatus(software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckResourceStatusRequest)}
     * operation.</b>
     * </p>
     *
     * @param getReadinessCheckResourceStatusRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetReadinessCheckResourceStatus
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetReadinessCheckResourceStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetReadinessCheckResourceStatusIterable getReadinessCheckResourceStatusPaginator(
            GetReadinessCheckResourceStatusRequest getReadinessCheckResourceStatusRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        return new GetReadinessCheckResourceStatusIterable(this, applyPaginatorUserAgent(getReadinessCheckResourceStatusRequest));
    }

    /**
     * Returns information about the status of a Readiness Check.
     *
     * @param getReadinessCheckStatusRequest
     * @return Result of the GetReadinessCheckStatus operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetReadinessCheckStatus
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetReadinessCheckStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetReadinessCheckStatusResponse getReadinessCheckStatus(GetReadinessCheckStatusRequest getReadinessCheckStatusRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetReadinessCheckStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetReadinessCheckStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getReadinessCheckStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetReadinessCheckStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<GetReadinessCheckStatusRequest, GetReadinessCheckStatusResponse>()
                            .withOperationName("GetReadinessCheckStatus").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getReadinessCheckStatusRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetReadinessCheckStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns information about the status of a Readiness Check.<br/>
     * <p>
     * This is a variant of
     * {@link #getReadinessCheckStatus(software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckStatusRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetReadinessCheckStatusIterable responses = client.getReadinessCheckStatusPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetReadinessCheckStatusIterable responses = client
     *             .getReadinessCheckStatusPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckStatusResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetReadinessCheckStatusIterable responses = client.getReadinessCheckStatusPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getReadinessCheckStatus(software.amazon.awssdk.services.route53recoveryreadiness.model.GetReadinessCheckStatusRequest)}
     * operation.</b>
     * </p>
     *
     * @param getReadinessCheckStatusRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetReadinessCheckStatus
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetReadinessCheckStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetReadinessCheckStatusIterable getReadinessCheckStatusPaginator(
            GetReadinessCheckStatusRequest getReadinessCheckStatusRequest) throws ResourceNotFoundException, ThrottlingException,
            ValidationException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        return new GetReadinessCheckStatusIterable(this, applyPaginatorUserAgent(getReadinessCheckStatusRequest));
    }

    /**
     * Returns information about a Recovery Group.
     *
     * @param getRecoveryGroupRequest
     * @return Result of the GetRecoveryGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetRecoveryGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetRecoveryGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRecoveryGroupResponse getRecoveryGroup(GetRecoveryGroupRequest getRecoveryGroupRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRecoveryGroupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetRecoveryGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecoveryGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecoveryGroup");

            return clientHandler.execute(new ClientExecutionParams<GetRecoveryGroupRequest, GetRecoveryGroupResponse>()
                    .withOperationName("GetRecoveryGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getRecoveryGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRecoveryGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns information about a Recovery Group.
     *
     * @param getRecoveryGroupReadinessSummaryRequest
     * @return Result of the GetRecoveryGroupReadinessSummary operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetRecoveryGroupReadinessSummary
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetRecoveryGroupReadinessSummary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRecoveryGroupReadinessSummaryResponse getRecoveryGroupReadinessSummary(
            GetRecoveryGroupReadinessSummaryRequest getRecoveryGroupReadinessSummaryRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRecoveryGroupReadinessSummaryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRecoveryGroupReadinessSummaryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getRecoveryGroupReadinessSummaryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecoveryGroupReadinessSummary");

            return clientHandler
                    .execute(new ClientExecutionParams<GetRecoveryGroupReadinessSummaryRequest, GetRecoveryGroupReadinessSummaryResponse>()
                            .withOperationName("GetRecoveryGroupReadinessSummary").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getRecoveryGroupReadinessSummaryRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetRecoveryGroupReadinessSummaryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns information about a Recovery Group.<br/>
     * <p>
     * This is a variant of
     * {@link #getRecoveryGroupReadinessSummary(software.amazon.awssdk.services.route53recoveryreadiness.model.GetRecoveryGroupReadinessSummaryRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetRecoveryGroupReadinessSummaryIterable responses = client.getRecoveryGroupReadinessSummaryPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetRecoveryGroupReadinessSummaryIterable responses = client
     *             .getRecoveryGroupReadinessSummaryPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoveryreadiness.model.GetRecoveryGroupReadinessSummaryResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.GetRecoveryGroupReadinessSummaryIterable responses = client.getRecoveryGroupReadinessSummaryPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getRecoveryGroupReadinessSummary(software.amazon.awssdk.services.route53recoveryreadiness.model.GetRecoveryGroupReadinessSummaryRequest)}
     * operation.</b>
     * </p>
     *
     * @param getRecoveryGroupReadinessSummaryRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetRecoveryGroupReadinessSummary
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetRecoveryGroupReadinessSummary"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRecoveryGroupReadinessSummaryIterable getRecoveryGroupReadinessSummaryPaginator(
            GetRecoveryGroupReadinessSummaryRequest getRecoveryGroupReadinessSummaryRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        return new GetRecoveryGroupReadinessSummaryIterable(this,
                applyPaginatorUserAgent(getRecoveryGroupReadinessSummaryRequest));
    }

    /**
     * Returns information about a Resource Set.
     *
     * @param getResourceSetRequest
     * @return Result of the GetResourceSet operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.GetResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/GetResourceSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourceSetResponse getResourceSet(GetResourceSetRequest getResourceSetRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourceSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceSet");

            return clientHandler.execute(new ClientExecutionParams<GetResourceSetRequest, GetResourceSetResponse>()
                    .withOperationName("GetResourceSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getResourceSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourceSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a collection of Cells.
     *
     * @param listCellsRequest
     * @return Result of the ListCells operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListCells
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListCells"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCellsResponse listCells(ListCellsRequest listCellsRequest) throws ThrottlingException, ValidationException,
            InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCellsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListCellsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCellsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCells");

            return clientHandler.execute(new ClientExecutionParams<ListCellsRequest, ListCellsResponse>()
                    .withOperationName("ListCells").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listCellsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListCellsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a collection of Cells.<br/>
     * <p>
     * This is a variant of
     * {@link #listCells(software.amazon.awssdk.services.route53recoveryreadiness.model.ListCellsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListCellsIterable responses = client.listCellsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListCellsIterable responses = client
     *             .listCellsPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoveryreadiness.model.ListCellsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListCellsIterable responses = client.listCellsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCells(software.amazon.awssdk.services.route53recoveryreadiness.model.ListCellsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listCellsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListCells
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListCells"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCellsIterable listCellsPaginator(ListCellsRequest listCellsRequest) throws ThrottlingException,
            ValidationException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        return new ListCellsIterable(this, applyPaginatorUserAgent(listCellsRequest));
    }

    /**
     * Returns a collection of cross account readiness authorizations.
     *
     * @param listCrossAccountAuthorizationsRequest
     * @return Result of the ListCrossAccountAuthorizations operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListCrossAccountAuthorizations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListCrossAccountAuthorizations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCrossAccountAuthorizationsResponse listCrossAccountAuthorizations(
            ListCrossAccountAuthorizationsRequest listCrossAccountAuthorizationsRequest) throws ThrottlingException,
            ValidationException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCrossAccountAuthorizationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListCrossAccountAuthorizationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listCrossAccountAuthorizationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCrossAccountAuthorizations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListCrossAccountAuthorizationsRequest, ListCrossAccountAuthorizationsResponse>()
                            .withOperationName("ListCrossAccountAuthorizations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listCrossAccountAuthorizationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListCrossAccountAuthorizationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a collection of cross account readiness authorizations.<br/>
     * <p>
     * This is a variant of
     * {@link #listCrossAccountAuthorizations(software.amazon.awssdk.services.route53recoveryreadiness.model.ListCrossAccountAuthorizationsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListCrossAccountAuthorizationsIterable responses = client.listCrossAccountAuthorizationsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListCrossAccountAuthorizationsIterable responses = client
     *             .listCrossAccountAuthorizationsPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoveryreadiness.model.ListCrossAccountAuthorizationsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListCrossAccountAuthorizationsIterable responses = client.listCrossAccountAuthorizationsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCrossAccountAuthorizations(software.amazon.awssdk.services.route53recoveryreadiness.model.ListCrossAccountAuthorizationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listCrossAccountAuthorizationsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListCrossAccountAuthorizations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListCrossAccountAuthorizations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListCrossAccountAuthorizationsIterable listCrossAccountAuthorizationsPaginator(
            ListCrossAccountAuthorizationsRequest listCrossAccountAuthorizationsRequest) throws ThrottlingException,
            ValidationException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        return new ListCrossAccountAuthorizationsIterable(this, applyPaginatorUserAgent(listCrossAccountAuthorizationsRequest));
    }

    /**
     * Returns a collection of Readiness Checks.
     *
     * @param listReadinessChecksRequest
     * @return Result of the ListReadinessChecks operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListReadinessChecks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListReadinessChecks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListReadinessChecksResponse listReadinessChecks(ListReadinessChecksRequest listReadinessChecksRequest)
            throws ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListReadinessChecksResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListReadinessChecksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReadinessChecksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReadinessChecks");

            return clientHandler.execute(new ClientExecutionParams<ListReadinessChecksRequest, ListReadinessChecksResponse>()
                    .withOperationName("ListReadinessChecks").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listReadinessChecksRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListReadinessChecksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a collection of Readiness Checks.<br/>
     * <p>
     * This is a variant of
     * {@link #listReadinessChecks(software.amazon.awssdk.services.route53recoveryreadiness.model.ListReadinessChecksRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListReadinessChecksIterable responses = client.listReadinessChecksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListReadinessChecksIterable responses = client
     *             .listReadinessChecksPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoveryreadiness.model.ListReadinessChecksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListReadinessChecksIterable responses = client.listReadinessChecksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listReadinessChecks(software.amazon.awssdk.services.route53recoveryreadiness.model.ListReadinessChecksRequest)}
     * operation.</b>
     * </p>
     *
     * @param listReadinessChecksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListReadinessChecks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListReadinessChecks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListReadinessChecksIterable listReadinessChecksPaginator(ListReadinessChecksRequest listReadinessChecksRequest)
            throws ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        return new ListReadinessChecksIterable(this, applyPaginatorUserAgent(listReadinessChecksRequest));
    }

    /**
     * Returns a collection of Recovery Groups.
     *
     * @param listRecoveryGroupsRequest
     * @return Result of the ListRecoveryGroups operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListRecoveryGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListRecoveryGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRecoveryGroupsResponse listRecoveryGroups(ListRecoveryGroupsRequest listRecoveryGroupsRequest)
            throws ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRecoveryGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRecoveryGroupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecoveryGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecoveryGroups");

            return clientHandler.execute(new ClientExecutionParams<ListRecoveryGroupsRequest, ListRecoveryGroupsResponse>()
                    .withOperationName("ListRecoveryGroups").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listRecoveryGroupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRecoveryGroupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a collection of Recovery Groups.<br/>
     * <p>
     * This is a variant of
     * {@link #listRecoveryGroups(software.amazon.awssdk.services.route53recoveryreadiness.model.ListRecoveryGroupsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListRecoveryGroupsIterable responses = client.listRecoveryGroupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListRecoveryGroupsIterable responses = client
     *             .listRecoveryGroupsPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoveryreadiness.model.ListRecoveryGroupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListRecoveryGroupsIterable responses = client.listRecoveryGroupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRecoveryGroups(software.amazon.awssdk.services.route53recoveryreadiness.model.ListRecoveryGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRecoveryGroupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListRecoveryGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListRecoveryGroups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRecoveryGroupsIterable listRecoveryGroupsPaginator(ListRecoveryGroupsRequest listRecoveryGroupsRequest)
            throws ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        return new ListRecoveryGroupsIterable(this, applyPaginatorUserAgent(listRecoveryGroupsRequest));
    }

    /**
     * Returns a collection of Resource Sets.
     *
     * @param listResourceSetsRequest
     * @return Result of the ListResourceSets operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListResourceSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListResourceSets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResourceSetsResponse listResourceSets(ListResourceSetsRequest listResourceSetsRequest) throws ThrottlingException,
            ValidationException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourceSetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListResourceSetsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceSetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceSets");

            return clientHandler.execute(new ClientExecutionParams<ListResourceSetsRequest, ListResourceSetsResponse>()
                    .withOperationName("ListResourceSets").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listResourceSetsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResourceSetsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a collection of Resource Sets.<br/>
     * <p>
     * This is a variant of
     * {@link #listResourceSets(software.amazon.awssdk.services.route53recoveryreadiness.model.ListResourceSetsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListResourceSetsIterable responses = client.listResourceSetsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListResourceSetsIterable responses = client
     *             .listResourceSetsPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoveryreadiness.model.ListResourceSetsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListResourceSetsIterable responses = client.listResourceSetsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResourceSets(software.amazon.awssdk.services.route53recoveryreadiness.model.ListResourceSetsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResourceSetsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListResourceSets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListResourceSets"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResourceSetsIterable listResourceSetsPaginator(ListResourceSetsRequest listResourceSetsRequest)
            throws ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        return new ListResourceSetsIterable(this, applyPaginatorUserAgent(listResourceSetsRequest));
    }

    /**
     * Returns a collection of rules that are applied as part of Readiness Checks.
     *
     * @param listRulesRequest
     * @return Result of the ListRules operation returned by the service.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRulesResponse listRules(ListRulesRequest listRulesRequest) throws ThrottlingException, ValidationException,
            InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRulesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRules");

            return clientHandler.execute(new ClientExecutionParams<ListRulesRequest, ListRulesResponse>()
                    .withOperationName("ListRules").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listRulesRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Returns a collection of rules that are applied as part of Readiness Checks.<br/>
     * <p>
     * This is a variant of
     * {@link #listRules(software.amazon.awssdk.services.route53recoveryreadiness.model.ListRulesRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListRulesIterable responses = client.listRulesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListRulesIterable responses = client
     *             .listRulesPaginator(request);
     *     for (software.amazon.awssdk.services.route53recoveryreadiness.model.ListRulesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.route53recoveryreadiness.paginators.ListRulesIterable responses = client.listRulesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRules(software.amazon.awssdk.services.route53recoveryreadiness.model.ListRulesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRulesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListRulesIterable listRulesPaginator(ListRulesRequest listRulesRequest) throws ThrottlingException,
            ValidationException, InternalServerException, AccessDeniedException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        return new ListRulesIterable(this, applyPaginatorUserAgent(listRulesRequest));
    }

    /**
     * Returns a list of the tags assigned to the specified resource.
     *
     * @param listTagsForResourcesRequest
     * @return Result of the ListTagsForResources operation returned by the service.
     * @throws ResourceNotFoundException
     *         requested resource was not found
     * @throws ValidationException
     *         an invalid request
     * @throws InternalServerException
     *         Internal service error
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.ListTagsForResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/ListTagsForResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourcesResponse listTagsForResources(ListTagsForResourcesRequest listTagsForResourcesRequest)
            throws ResourceNotFoundException, ValidationException, InternalServerException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResources");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourcesRequest, ListTagsForResourcesResponse>()
                    .withOperationName("ListTagsForResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Adds tags to the specified resource. You can specify one or more tags to add.
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         requested resource was not found
     * @throws ValidationException
     *         an invalid request
     * @throws InternalServerException
     *         Internal service error
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/TagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Removes tags from the specified resource. You can specify one or more tags to remove.
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         requested resource was not found
     * @throws ValidationException
     *         an invalid request
     * @throws InternalServerException
     *         Internal service error
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/UntagResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            ValidationException, InternalServerException, AwsServiceException, SdkClientException,
            Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Updates an existing Cell.
     *
     * @param updateCellRequest
     *        Parameters to update for the Cell
     * @return Result of the UpdateCell operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.UpdateCell
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/UpdateCell"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateCellResponse updateCell(UpdateCellRequest updateCellRequest) throws ResourceNotFoundException,
            ThrottlingException, ValidationException, InternalServerException, AccessDeniedException, AwsServiceException,
            SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateCellResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateCellResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCellRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCell");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateCellRequest, UpdateCellResponse>().withOperationName("UpdateCell")
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateCellRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateCellRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Updates an exisiting Readiness Check.
     *
     * @param updateReadinessCheckRequest
     *        The new Readiness Check values
     * @return Result of the UpdateReadinessCheck operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.UpdateReadinessCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/UpdateReadinessCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateReadinessCheckResponse updateReadinessCheck(UpdateReadinessCheckRequest updateReadinessCheckRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateReadinessCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateReadinessCheckResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateReadinessCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateReadinessCheck");

            return clientHandler.execute(new ClientExecutionParams<UpdateReadinessCheckRequest, UpdateReadinessCheckResponse>()
                    .withOperationName("UpdateReadinessCheck").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateReadinessCheckRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateReadinessCheckRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Updates an existing Recovery Group.
     *
     * @param updateRecoveryGroupRequest
     *        Parameters to update for the RecoveryGroup
     * @return Result of the UpdateRecoveryGroup operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.UpdateRecoveryGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/UpdateRecoveryGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateRecoveryGroupResponse updateRecoveryGroup(UpdateRecoveryGroupRequest updateRecoveryGroupRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRecoveryGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateRecoveryGroupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRecoveryGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRecoveryGroup");

            return clientHandler.execute(new ClientExecutionParams<UpdateRecoveryGroupRequest, UpdateRecoveryGroupResponse>()
                    .withOperationName("UpdateRecoveryGroup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateRecoveryGroupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateRecoveryGroupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Updates an existing Resource Set.
     *
     * @param updateResourceSetRequest
     *        configuration for the desired
     * @return Result of the UpdateResourceSet operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource does not exist.
     * @throws ThrottlingException
     *         Request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws InternalServerException
     *         An unexpected error occurred.
     * @throws AccessDeniedException
     *         User does not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Route53RecoveryReadinessException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Route53RecoveryReadinessClient.UpdateResourceSet
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/route53-recovery-readiness-2019-12-02/UpdateResourceSet"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateResourceSetResponse updateResourceSet(UpdateResourceSetRequest updateResourceSetRequest)
            throws ResourceNotFoundException, ThrottlingException, ValidationException, InternalServerException,
            AccessDeniedException, AwsServiceException, SdkClientException, Route53RecoveryReadinessException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateResourceSetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateResourceSetResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResourceSetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Route53 Recovery Readiness");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResourceSet");

            return clientHandler.execute(new ClientExecutionParams<UpdateResourceSetRequest, UpdateResourceSetResponse>()
                    .withOperationName("UpdateResourceSet").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateResourceSetRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateResourceSetRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Route53RecoveryReadinessException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends Route53RecoveryReadinessRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
