/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.s3tables.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetTableResponse extends S3TablesResponse implements
        ToCopyableBuilder<GetTableResponse.Builder, GetTableResponse> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(GetTableResponse::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(GetTableResponse::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> TABLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("tableARN").getter(getter(GetTableResponse::tableARN)).setter(setter(Builder::tableARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tableARN").build()).build();

    private static final SdkField<List<String>> NAMESPACE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("namespace")
            .getter(getter(GetTableResponse::namespace))
            .setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("namespace").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> VERSION_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("versionToken").getter(getter(GetTableResponse::versionToken)).setter(setter(Builder::versionToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("versionToken").build()).build();

    private static final SdkField<String> METADATA_LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("metadataLocation").getter(getter(GetTableResponse::metadataLocation))
            .setter(setter(Builder::metadataLocation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("metadataLocation").build()).build();

    private static final SdkField<String> WAREHOUSE_LOCATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("warehouseLocation").getter(getter(GetTableResponse::warehouseLocation))
            .setter(setter(Builder::warehouseLocation))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("warehouseLocation").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(GetTableResponse::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> CREATED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("createdBy").getter(getter(GetTableResponse::createdBy)).setter(setter(Builder::createdBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdBy").build()).build();

    private static final SdkField<String> MANAGED_BY_SERVICE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("managedByService").getter(getter(GetTableResponse::managedByService))
            .setter(setter(Builder::managedByService))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("managedByService").build()).build();

    private static final SdkField<Instant> MODIFIED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("modifiedAt")
            .getter(getter(GetTableResponse::modifiedAt))
            .setter(setter(Builder::modifiedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modifiedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> MODIFIED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("modifiedBy").getter(getter(GetTableResponse::modifiedBy)).setter(setter(Builder::modifiedBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modifiedBy").build()).build();

    private static final SdkField<String> OWNER_ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ownerAccountId").getter(getter(GetTableResponse::ownerAccountId))
            .setter(setter(Builder::ownerAccountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ownerAccountId").build()).build();

    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("format")
            .getter(getter(GetTableResponse::formatAsString)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("format").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, TYPE_FIELD,
            TABLE_ARN_FIELD, NAMESPACE_FIELD, VERSION_TOKEN_FIELD, METADATA_LOCATION_FIELD, WAREHOUSE_LOCATION_FIELD,
            CREATED_AT_FIELD, CREATED_BY_FIELD, MANAGED_BY_SERVICE_FIELD, MODIFIED_AT_FIELD, MODIFIED_BY_FIELD,
            OWNER_ACCOUNT_ID_FIELD, FORMAT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("name", NAME_FIELD);
                    put("type", TYPE_FIELD);
                    put("tableARN", TABLE_ARN_FIELD);
                    put("namespace", NAMESPACE_FIELD);
                    put("versionToken", VERSION_TOKEN_FIELD);
                    put("metadataLocation", METADATA_LOCATION_FIELD);
                    put("warehouseLocation", WAREHOUSE_LOCATION_FIELD);
                    put("createdAt", CREATED_AT_FIELD);
                    put("createdBy", CREATED_BY_FIELD);
                    put("managedByService", MANAGED_BY_SERVICE_FIELD);
                    put("modifiedAt", MODIFIED_AT_FIELD);
                    put("modifiedBy", MODIFIED_BY_FIELD);
                    put("ownerAccountId", OWNER_ACCOUNT_ID_FIELD);
                    put("format", FORMAT_FIELD);
                }
            });

    private final String name;

    private final String type;

    private final String tableARN;

    private final List<String> namespace;

    private final String versionToken;

    private final String metadataLocation;

    private final String warehouseLocation;

    private final Instant createdAt;

    private final String createdBy;

    private final String managedByService;

    private final Instant modifiedAt;

    private final String modifiedBy;

    private final String ownerAccountId;

    private final String format;

    private GetTableResponse(BuilderImpl builder) {
        super(builder);
        this.name = builder.name;
        this.type = builder.type;
        this.tableARN = builder.tableARN;
        this.namespace = builder.namespace;
        this.versionToken = builder.versionToken;
        this.metadataLocation = builder.metadataLocation;
        this.warehouseLocation = builder.warehouseLocation;
        this.createdAt = builder.createdAt;
        this.createdBy = builder.createdBy;
        this.managedByService = builder.managedByService;
        this.modifiedAt = builder.modifiedAt;
        this.modifiedBy = builder.modifiedBy;
        this.ownerAccountId = builder.ownerAccountId;
        this.format = builder.format;
    }

    /**
     * <p>
     * The name of the table.
     * </p>
     * 
     * @return The name of the table.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The type of the table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TableType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the table.
     * @see TableType
     */
    public final TableType type() {
        return TableType.fromValue(type);
    }

    /**
     * <p>
     * The type of the table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link TableType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the table.
     * @see TableType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the table.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the table.
     */
    public final String tableARN() {
        return tableARN;
    }

    /**
     * For responses, this returns true if the service returned a value for the Namespace property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasNamespace() {
        return namespace != null && !(namespace instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The namespace associated with the table.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNamespace} method.
     * </p>
     * 
     * @return The namespace associated with the table.
     */
    public final List<String> namespace() {
        return namespace;
    }

    /**
     * <p>
     * The version token of the table.
     * </p>
     * 
     * @return The version token of the table.
     */
    public final String versionToken() {
        return versionToken;
    }

    /**
     * <p>
     * The metadata location of the table.
     * </p>
     * 
     * @return The metadata location of the table.
     */
    public final String metadataLocation() {
        return metadataLocation;
    }

    /**
     * <p>
     * The warehouse location of the table.
     * </p>
     * 
     * @return The warehouse location of the table.
     */
    public final String warehouseLocation() {
        return warehouseLocation;
    }

    /**
     * <p>
     * The date and time the table bucket was created at.
     * </p>
     * 
     * @return The date and time the table bucket was created at.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The ID of the account that created the table.
     * </p>
     * 
     * @return The ID of the account that created the table.
     */
    public final String createdBy() {
        return createdBy;
    }

    /**
     * <p>
     * The service that manages the table.
     * </p>
     * 
     * @return The service that manages the table.
     */
    public final String managedByService() {
        return managedByService;
    }

    /**
     * <p>
     * The date and time the table was last modified on.
     * </p>
     * 
     * @return The date and time the table was last modified on.
     */
    public final Instant modifiedAt() {
        return modifiedAt;
    }

    /**
     * <p>
     * The ID of the account that last modified the table.
     * </p>
     * 
     * @return The ID of the account that last modified the table.
     */
    public final String modifiedBy() {
        return modifiedBy;
    }

    /**
     * <p>
     * The ID of the account that owns the table.
     * </p>
     * 
     * @return The ID of the account that owns the table.
     */
    public final String ownerAccountId() {
        return ownerAccountId;
    }

    /**
     * <p>
     * The format of the table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link OpenTableFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the table.
     * @see OpenTableFormat
     */
    public final OpenTableFormat format() {
        return OpenTableFormat.fromValue(format);
    }

    /**
     * <p>
     * The format of the table.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link OpenTableFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the table.
     * @see OpenTableFormat
     */
    public final String formatAsString() {
        return format;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tableARN());
        hashCode = 31 * hashCode + Objects.hashCode(hasNamespace() ? namespace() : null);
        hashCode = 31 * hashCode + Objects.hashCode(versionToken());
        hashCode = 31 * hashCode + Objects.hashCode(metadataLocation());
        hashCode = 31 * hashCode + Objects.hashCode(warehouseLocation());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(createdBy());
        hashCode = 31 * hashCode + Objects.hashCode(managedByService());
        hashCode = 31 * hashCode + Objects.hashCode(modifiedAt());
        hashCode = 31 * hashCode + Objects.hashCode(modifiedBy());
        hashCode = 31 * hashCode + Objects.hashCode(ownerAccountId());
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetTableResponse)) {
            return false;
        }
        GetTableResponse other = (GetTableResponse) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(tableARN(), other.tableARN()) && hasNamespace() == other.hasNamespace()
                && Objects.equals(namespace(), other.namespace()) && Objects.equals(versionToken(), other.versionToken())
                && Objects.equals(metadataLocation(), other.metadataLocation())
                && Objects.equals(warehouseLocation(), other.warehouseLocation())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(createdBy(), other.createdBy())
                && Objects.equals(managedByService(), other.managedByService())
                && Objects.equals(modifiedAt(), other.modifiedAt()) && Objects.equals(modifiedBy(), other.modifiedBy())
                && Objects.equals(ownerAccountId(), other.ownerAccountId())
                && Objects.equals(formatAsString(), other.formatAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GetTableResponse").add("Name", name()).add("Type", typeAsString()).add("TableARN", tableARN())
                .add("Namespace", hasNamespace() ? namespace() : null).add("VersionToken", versionToken())
                .add("MetadataLocation", metadataLocation()).add("WarehouseLocation", warehouseLocation())
                .add("CreatedAt", createdAt()).add("CreatedBy", createdBy()).add("ManagedByService", managedByService())
                .add("ModifiedAt", modifiedAt()).add("ModifiedBy", modifiedBy()).add("OwnerAccountId", ownerAccountId())
                .add("Format", formatAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "tableARN":
            return Optional.ofNullable(clazz.cast(tableARN()));
        case "namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "versionToken":
            return Optional.ofNullable(clazz.cast(versionToken()));
        case "metadataLocation":
            return Optional.ofNullable(clazz.cast(metadataLocation()));
        case "warehouseLocation":
            return Optional.ofNullable(clazz.cast(warehouseLocation()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "createdBy":
            return Optional.ofNullable(clazz.cast(createdBy()));
        case "managedByService":
            return Optional.ofNullable(clazz.cast(managedByService()));
        case "modifiedAt":
            return Optional.ofNullable(clazz.cast(modifiedAt()));
        case "modifiedBy":
            return Optional.ofNullable(clazz.cast(modifiedBy()));
        case "ownerAccountId":
            return Optional.ofNullable(clazz.cast(ownerAccountId()));
        case "format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<GetTableResponse, T> g) {
        return obj -> g.apply((GetTableResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends S3TablesResponse.Builder, SdkPojo, CopyableBuilder<Builder, GetTableResponse> {
        /**
         * <p>
         * The name of the table.
         * </p>
         * 
         * @param name
         *        The name of the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of the table.
         * </p>
         * 
         * @param type
         *        The type of the table.
         * @see TableType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TableType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the table.
         * </p>
         * 
         * @param type
         *        The type of the table.
         * @see TableType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TableType
         */
        Builder type(TableType type);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the table.
         * </p>
         * 
         * @param tableARN
         *        The Amazon Resource Name (ARN) of the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableARN(String tableARN);

        /**
         * <p>
         * The namespace associated with the table.
         * </p>
         * 
         * @param namespace
         *        The namespace associated with the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(Collection<String> namespace);

        /**
         * <p>
         * The namespace associated with the table.
         * </p>
         * 
         * @param namespace
         *        The namespace associated with the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String... namespace);

        /**
         * <p>
         * The version token of the table.
         * </p>
         * 
         * @param versionToken
         *        The version token of the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionToken(String versionToken);

        /**
         * <p>
         * The metadata location of the table.
         * </p>
         * 
         * @param metadataLocation
         *        The metadata location of the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metadataLocation(String metadataLocation);

        /**
         * <p>
         * The warehouse location of the table.
         * </p>
         * 
         * @param warehouseLocation
         *        The warehouse location of the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder warehouseLocation(String warehouseLocation);

        /**
         * <p>
         * The date and time the table bucket was created at.
         * </p>
         * 
         * @param createdAt
         *        The date and time the table bucket was created at.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The ID of the account that created the table.
         * </p>
         * 
         * @param createdBy
         *        The ID of the account that created the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBy(String createdBy);

        /**
         * <p>
         * The service that manages the table.
         * </p>
         * 
         * @param managedByService
         *        The service that manages the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedByService(String managedByService);

        /**
         * <p>
         * The date and time the table was last modified on.
         * </p>
         * 
         * @param modifiedAt
         *        The date and time the table was last modified on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifiedAt(Instant modifiedAt);

        /**
         * <p>
         * The ID of the account that last modified the table.
         * </p>
         * 
         * @param modifiedBy
         *        The ID of the account that last modified the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifiedBy(String modifiedBy);

        /**
         * <p>
         * The ID of the account that owns the table.
         * </p>
         * 
         * @param ownerAccountId
         *        The ID of the account that owns the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerAccountId(String ownerAccountId);

        /**
         * <p>
         * The format of the table.
         * </p>
         * 
         * @param format
         *        The format of the table.
         * @see OpenTableFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenTableFormat
         */
        Builder format(String format);

        /**
         * <p>
         * The format of the table.
         * </p>
         * 
         * @param format
         *        The format of the table.
         * @see OpenTableFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenTableFormat
         */
        Builder format(OpenTableFormat format);
    }

    static final class BuilderImpl extends S3TablesResponse.BuilderImpl implements Builder {
        private String name;

        private String type;

        private String tableARN;

        private List<String> namespace = DefaultSdkAutoConstructList.getInstance();

        private String versionToken;

        private String metadataLocation;

        private String warehouseLocation;

        private Instant createdAt;

        private String createdBy;

        private String managedByService;

        private Instant modifiedAt;

        private String modifiedBy;

        private String ownerAccountId;

        private String format;

        private BuilderImpl() {
        }

        private BuilderImpl(GetTableResponse model) {
            super(model);
            name(model.name);
            type(model.type);
            tableARN(model.tableARN);
            namespace(model.namespace);
            versionToken(model.versionToken);
            metadataLocation(model.metadataLocation);
            warehouseLocation(model.warehouseLocation);
            createdAt(model.createdAt);
            createdBy(model.createdBy);
            managedByService(model.managedByService);
            modifiedAt(model.modifiedAt);
            modifiedBy(model.modifiedBy);
            ownerAccountId(model.ownerAccountId);
            format(model.format);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(TableType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getTableARN() {
            return tableARN;
        }

        public final void setTableARN(String tableARN) {
            this.tableARN = tableARN;
        }

        @Override
        public final Builder tableARN(String tableARN) {
            this.tableARN = tableARN;
            return this;
        }

        public final Collection<String> getNamespace() {
            if (namespace instanceof SdkAutoConstructList) {
                return null;
            }
            return namespace;
        }

        public final void setNamespace(Collection<String> namespace) {
            this.namespace = NamespaceListCopier.copy(namespace);
        }

        @Override
        public final Builder namespace(Collection<String> namespace) {
            this.namespace = NamespaceListCopier.copy(namespace);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder namespace(String... namespace) {
            namespace(Arrays.asList(namespace));
            return this;
        }

        public final String getVersionToken() {
            return versionToken;
        }

        public final void setVersionToken(String versionToken) {
            this.versionToken = versionToken;
        }

        @Override
        public final Builder versionToken(String versionToken) {
            this.versionToken = versionToken;
            return this;
        }

        public final String getMetadataLocation() {
            return metadataLocation;
        }

        public final void setMetadataLocation(String metadataLocation) {
            this.metadataLocation = metadataLocation;
        }

        @Override
        public final Builder metadataLocation(String metadataLocation) {
            this.metadataLocation = metadataLocation;
            return this;
        }

        public final String getWarehouseLocation() {
            return warehouseLocation;
        }

        public final void setWarehouseLocation(String warehouseLocation) {
            this.warehouseLocation = warehouseLocation;
        }

        @Override
        public final Builder warehouseLocation(String warehouseLocation) {
            this.warehouseLocation = warehouseLocation;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getCreatedBy() {
            return createdBy;
        }

        public final void setCreatedBy(String createdBy) {
            this.createdBy = createdBy;
        }

        @Override
        public final Builder createdBy(String createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public final String getManagedByService() {
            return managedByService;
        }

        public final void setManagedByService(String managedByService) {
            this.managedByService = managedByService;
        }

        @Override
        public final Builder managedByService(String managedByService) {
            this.managedByService = managedByService;
            return this;
        }

        public final Instant getModifiedAt() {
            return modifiedAt;
        }

        public final void setModifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
        }

        @Override
        public final Builder modifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
            return this;
        }

        public final String getModifiedBy() {
            return modifiedBy;
        }

        public final void setModifiedBy(String modifiedBy) {
            this.modifiedBy = modifiedBy;
        }

        @Override
        public final Builder modifiedBy(String modifiedBy) {
            this.modifiedBy = modifiedBy;
            return this;
        }

        public final String getOwnerAccountId() {
            return ownerAccountId;
        }

        public final void setOwnerAccountId(String ownerAccountId) {
            this.ownerAccountId = ownerAccountId;
        }

        @Override
        public final Builder ownerAccountId(String ownerAccountId) {
            this.ownerAccountId = ownerAccountId;
            return this;
        }

        public final String getFormat() {
            return format;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(OpenTableFormat format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        @Override
        public GetTableResponse build() {
            return new GetTableResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
