/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ses.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to return the Amazon SES verification status of a list of identities. For domain identities,
 * this request also returns the verification token. For information about verifying identities with Amazon SES, see the
 * <a href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/verify-addresses-and-domains.html">Amazon SES
 * Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetIdentityVerificationAttributesRequest extends SesRequest implements
        ToCopyableBuilder<GetIdentityVerificationAttributesRequest.Builder, GetIdentityVerificationAttributesRequest> {
    private static final SdkField<List<String>> IDENTITIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Identities")
            .getter(getter(GetIdentityVerificationAttributesRequest::identities))
            .setter(setter(Builder::identities))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Identities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IDENTITIES_FIELD));

    private final List<String> identities;

    private GetIdentityVerificationAttributesRequest(BuilderImpl builder) {
        super(builder);
        this.identities = builder.identities;
    }

    /**
     * For responses, this returns true if the service returned a value for the Identities property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasIdentities() {
        return identities != null && !(identities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of identities.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIdentities} method.
     * </p>
     * 
     * @return A list of identities.
     */
    public final List<String> identities() {
        return identities;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(hasIdentities() ? identities() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetIdentityVerificationAttributesRequest)) {
            return false;
        }
        GetIdentityVerificationAttributesRequest other = (GetIdentityVerificationAttributesRequest) obj;
        return hasIdentities() == other.hasIdentities() && Objects.equals(identities(), other.identities());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GetIdentityVerificationAttributesRequest")
                .add("Identities", hasIdentities() ? identities() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Identities":
            return Optional.ofNullable(clazz.cast(identities()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetIdentityVerificationAttributesRequest, T> g) {
        return obj -> g.apply((GetIdentityVerificationAttributesRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SesRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, GetIdentityVerificationAttributesRequest> {
        /**
         * <p>
         * A list of identities.
         * </p>
         * 
         * @param identities
         *        A list of identities.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identities(Collection<String> identities);

        /**
         * <p>
         * A list of identities.
         * </p>
         * 
         * @param identities
         *        A list of identities.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identities(String... identities);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends SesRequest.BuilderImpl implements Builder {
        private List<String> identities = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(GetIdentityVerificationAttributesRequest model) {
            super(model);
            identities(model.identities);
        }

        public final Collection<String> getIdentities() {
            if (identities instanceof SdkAutoConstructList) {
                return null;
            }
            return identities;
        }

        public final void setIdentities(Collection<String> identities) {
            this.identities = IdentityListCopier.copy(identities);
        }

        @Override
        public final Builder identities(Collection<String> identities) {
            this.identities = IdentityListCopier.copy(identities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder identities(String... identities) {
            identities(Arrays.asList(identities));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetIdentityVerificationAttributesRequest build() {
            return new GetIdentityVerificationAttributesRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
