/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.shield;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.shield.model.AccessDeniedException;
import software.amazon.awssdk.services.shield.model.AccessDeniedForDependencyException;
import software.amazon.awssdk.services.shield.model.AssociateDrtLogBucketRequest;
import software.amazon.awssdk.services.shield.model.AssociateDrtLogBucketResponse;
import software.amazon.awssdk.services.shield.model.AssociateDrtRoleRequest;
import software.amazon.awssdk.services.shield.model.AssociateDrtRoleResponse;
import software.amazon.awssdk.services.shield.model.AssociateHealthCheckRequest;
import software.amazon.awssdk.services.shield.model.AssociateHealthCheckResponse;
import software.amazon.awssdk.services.shield.model.AssociateProactiveEngagementDetailsRequest;
import software.amazon.awssdk.services.shield.model.AssociateProactiveEngagementDetailsResponse;
import software.amazon.awssdk.services.shield.model.CreateProtectionGroupRequest;
import software.amazon.awssdk.services.shield.model.CreateProtectionGroupResponse;
import software.amazon.awssdk.services.shield.model.CreateProtectionRequest;
import software.amazon.awssdk.services.shield.model.CreateProtectionResponse;
import software.amazon.awssdk.services.shield.model.CreateSubscriptionRequest;
import software.amazon.awssdk.services.shield.model.CreateSubscriptionResponse;
import software.amazon.awssdk.services.shield.model.DeleteProtectionGroupRequest;
import software.amazon.awssdk.services.shield.model.DeleteProtectionGroupResponse;
import software.amazon.awssdk.services.shield.model.DeleteProtectionRequest;
import software.amazon.awssdk.services.shield.model.DeleteProtectionResponse;
import software.amazon.awssdk.services.shield.model.DescribeAttackRequest;
import software.amazon.awssdk.services.shield.model.DescribeAttackResponse;
import software.amazon.awssdk.services.shield.model.DescribeAttackStatisticsRequest;
import software.amazon.awssdk.services.shield.model.DescribeAttackStatisticsResponse;
import software.amazon.awssdk.services.shield.model.DescribeDrtAccessRequest;
import software.amazon.awssdk.services.shield.model.DescribeDrtAccessResponse;
import software.amazon.awssdk.services.shield.model.DescribeEmergencyContactSettingsRequest;
import software.amazon.awssdk.services.shield.model.DescribeEmergencyContactSettingsResponse;
import software.amazon.awssdk.services.shield.model.DescribeProtectionGroupRequest;
import software.amazon.awssdk.services.shield.model.DescribeProtectionGroupResponse;
import software.amazon.awssdk.services.shield.model.DescribeProtectionRequest;
import software.amazon.awssdk.services.shield.model.DescribeProtectionResponse;
import software.amazon.awssdk.services.shield.model.DescribeSubscriptionRequest;
import software.amazon.awssdk.services.shield.model.DescribeSubscriptionResponse;
import software.amazon.awssdk.services.shield.model.DisableApplicationLayerAutomaticResponseRequest;
import software.amazon.awssdk.services.shield.model.DisableApplicationLayerAutomaticResponseResponse;
import software.amazon.awssdk.services.shield.model.DisableProactiveEngagementRequest;
import software.amazon.awssdk.services.shield.model.DisableProactiveEngagementResponse;
import software.amazon.awssdk.services.shield.model.DisassociateDrtLogBucketRequest;
import software.amazon.awssdk.services.shield.model.DisassociateDrtLogBucketResponse;
import software.amazon.awssdk.services.shield.model.DisassociateDrtRoleRequest;
import software.amazon.awssdk.services.shield.model.DisassociateDrtRoleResponse;
import software.amazon.awssdk.services.shield.model.DisassociateHealthCheckRequest;
import software.amazon.awssdk.services.shield.model.DisassociateHealthCheckResponse;
import software.amazon.awssdk.services.shield.model.EnableApplicationLayerAutomaticResponseRequest;
import software.amazon.awssdk.services.shield.model.EnableApplicationLayerAutomaticResponseResponse;
import software.amazon.awssdk.services.shield.model.EnableProactiveEngagementRequest;
import software.amazon.awssdk.services.shield.model.EnableProactiveEngagementResponse;
import software.amazon.awssdk.services.shield.model.GetSubscriptionStateRequest;
import software.amazon.awssdk.services.shield.model.GetSubscriptionStateResponse;
import software.amazon.awssdk.services.shield.model.InternalErrorException;
import software.amazon.awssdk.services.shield.model.InvalidOperationException;
import software.amazon.awssdk.services.shield.model.InvalidPaginationTokenException;
import software.amazon.awssdk.services.shield.model.InvalidParameterException;
import software.amazon.awssdk.services.shield.model.InvalidResourceException;
import software.amazon.awssdk.services.shield.model.LimitsExceededException;
import software.amazon.awssdk.services.shield.model.ListAttacksRequest;
import software.amazon.awssdk.services.shield.model.ListAttacksResponse;
import software.amazon.awssdk.services.shield.model.ListProtectionGroupsRequest;
import software.amazon.awssdk.services.shield.model.ListProtectionGroupsResponse;
import software.amazon.awssdk.services.shield.model.ListProtectionsRequest;
import software.amazon.awssdk.services.shield.model.ListProtectionsResponse;
import software.amazon.awssdk.services.shield.model.ListResourcesInProtectionGroupRequest;
import software.amazon.awssdk.services.shield.model.ListResourcesInProtectionGroupResponse;
import software.amazon.awssdk.services.shield.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.shield.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.shield.model.LockedSubscriptionException;
import software.amazon.awssdk.services.shield.model.NoAssociatedRoleException;
import software.amazon.awssdk.services.shield.model.OptimisticLockException;
import software.amazon.awssdk.services.shield.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.shield.model.ResourceNotFoundException;
import software.amazon.awssdk.services.shield.model.ShieldException;
import software.amazon.awssdk.services.shield.model.ShieldRequest;
import software.amazon.awssdk.services.shield.model.TagResourceRequest;
import software.amazon.awssdk.services.shield.model.TagResourceResponse;
import software.amazon.awssdk.services.shield.model.UntagResourceRequest;
import software.amazon.awssdk.services.shield.model.UntagResourceResponse;
import software.amazon.awssdk.services.shield.model.UpdateApplicationLayerAutomaticResponseRequest;
import software.amazon.awssdk.services.shield.model.UpdateApplicationLayerAutomaticResponseResponse;
import software.amazon.awssdk.services.shield.model.UpdateEmergencyContactSettingsRequest;
import software.amazon.awssdk.services.shield.model.UpdateEmergencyContactSettingsResponse;
import software.amazon.awssdk.services.shield.model.UpdateProtectionGroupRequest;
import software.amazon.awssdk.services.shield.model.UpdateProtectionGroupResponse;
import software.amazon.awssdk.services.shield.model.UpdateSubscriptionRequest;
import software.amazon.awssdk.services.shield.model.UpdateSubscriptionResponse;
import software.amazon.awssdk.services.shield.paginators.ListAttacksPublisher;
import software.amazon.awssdk.services.shield.paginators.ListProtectionGroupsPublisher;
import software.amazon.awssdk.services.shield.paginators.ListProtectionsPublisher;
import software.amazon.awssdk.services.shield.paginators.ListResourcesInProtectionGroupPublisher;
import software.amazon.awssdk.services.shield.transform.AssociateDrtLogBucketRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.AssociateDrtRoleRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.AssociateHealthCheckRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.AssociateProactiveEngagementDetailsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.CreateProtectionGroupRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.CreateProtectionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.CreateSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DeleteProtectionGroupRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DeleteProtectionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeAttackRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeAttackStatisticsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeDrtAccessRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeEmergencyContactSettingsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeProtectionGroupRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeProtectionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DescribeSubscriptionRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DisableApplicationLayerAutomaticResponseRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DisableProactiveEngagementRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DisassociateDrtLogBucketRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DisassociateDrtRoleRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.DisassociateHealthCheckRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.EnableApplicationLayerAutomaticResponseRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.EnableProactiveEngagementRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.GetSubscriptionStateRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.ListAttacksRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.ListProtectionGroupsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.ListProtectionsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.ListResourcesInProtectionGroupRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.UpdateApplicationLayerAutomaticResponseRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.UpdateEmergencyContactSettingsRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.UpdateProtectionGroupRequestMarshaller;
import software.amazon.awssdk.services.shield.transform.UpdateSubscriptionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ShieldAsyncClient}.
 *
 * @see ShieldAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultShieldAsyncClient implements ShieldAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultShieldAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultShieldAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Authorizes the Shield Response Team (SRT) to access the specified Amazon S3 bucket containing log data such as
     * Application Load Balancer access logs, CloudFront logs, or logs from third party sources. You can associate up to
     * 10 Amazon S3 buckets with your subscription.
     * </p>
     * <p>
     * To use the services of the SRT and make an <code>AssociateDRTLogBucket</code> request, you must be subscribed to
     * the <a href="https://docs.aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://docs.aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>.
     * </p>
     *
     * @param associateDrtLogBucketRequest
     * @return A Java Future containing the result of the AssociateDRTLogBucket operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>NoAssociatedRoleException The ARN of the role that you specifed does not exist.</li>
     *         <li>LimitsExceededException Exception that indicates that the operation would exceed a limit.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>AccessDeniedForDependencyException In order to grant the necessary access to the Shield Response Team
     *         (SRT) the user submitting the request must have the <code>iam:PassRole</code> permission. This error
     *         indicates the user did not have the appropriate permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a User
     *         Permissions to Pass a Role to an Amazon Web Services Service</a>.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.AssociateDRTLogBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/AssociateDRTLogBucket" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateDrtLogBucketResponse> associateDRTLogBucket(
            AssociateDrtLogBucketRequest associateDrtLogBucketRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateDrtLogBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateDRTLogBucket");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateDrtLogBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateDrtLogBucketResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateDrtLogBucketResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateDrtLogBucketRequest, AssociateDrtLogBucketResponse>()
                            .withOperationName("AssociateDRTLogBucket")
                            .withMarshaller(new AssociateDrtLogBucketRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateDrtLogBucketRequest));
            CompletableFuture<AssociateDrtLogBucketResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Authorizes the Shield Response Team (SRT) using the specified role, to access your Amazon Web Services account to
     * assist with DDoS attack mitigation during potential attacks. This enables the SRT to inspect your WAF
     * configuration and create or update WAF rules and web ACLs.
     * </p>
     * <p>
     * You can associate only one <code>RoleArn</code> with your subscription. If you submit an
     * <code>AssociateDRTRole</code> request for an account that already has an associated role, the new
     * <code>RoleArn</code> will replace the existing <code>RoleArn</code>.
     * </p>
     * <p>
     * Prior to making the <code>AssociateDRTRole</code> request, you must attach the
     * <code>AWSShieldDRTAccessPolicy</code> managed policy to the role that you'll specify in the request. You can
     * access this policy in the IAM console at <a href=
     * "https://console.aws.amazon.com/iam/home?#/policies/arn:aws:iam::aws:policy/service-role/AWSShieldDRTAccessPolicy"
     * >AWSShieldDRTAccessPolicy</a>. For more information see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_manage-attach-detach.html">Adding and
     * removing IAM identity permissions</a>. The role must also trust the service principal
     * <code>drt.shield.amazonaws.com</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html">IAM JSON
     * policy elements: Principal</a>.
     * </p>
     * <p>
     * The SRT will have access only to your WAF and Shield resources. By submitting this request, you authorize the SRT
     * to inspect your WAF and Shield configuration and create and update WAF rules and web ACLs on your behalf. The SRT
     * takes these actions only if explicitly authorized by you.
     * </p>
     * <p>
     * You must have the <code>iam:PassRole</code> permission to make an <code>AssociateDRTRole</code> request. For more
     * information, see <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a
     * user permissions to pass a role to an Amazon Web Services service</a>.
     * </p>
     * <p>
     * To use the services of the SRT and make an <code>AssociateDRTRole</code> request, you must be subscribed to the
     * <a href="https://docs.aws.amazon.com/premiumsupport/business-support/">Business Support plan</a> or the <a
     * href="https://docs.aws.amazon.com/premiumsupport/enterprise-support/">Enterprise Support plan</a>.
     * </p>
     *
     * @param associateDrtRoleRequest
     * @return A Java Future containing the result of the AssociateDRTRole operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>AccessDeniedForDependencyException In order to grant the necessary access to the Shield Response Team
     *         (SRT) the user submitting the request must have the <code>iam:PassRole</code> permission. This error
     *         indicates the user did not have the appropriate permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a User
     *         Permissions to Pass a Role to an Amazon Web Services Service</a>.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.AssociateDRTRole
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/AssociateDRTRole" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateDrtRoleResponse> associateDRTRole(AssociateDrtRoleRequest associateDrtRoleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateDrtRoleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateDRTRole");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateDrtRoleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateDrtRoleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateDrtRoleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateDrtRoleRequest, AssociateDrtRoleResponse>()
                            .withOperationName("AssociateDRTRole")
                            .withMarshaller(new AssociateDrtRoleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateDrtRoleRequest));
            CompletableFuture<AssociateDrtRoleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds health-based detection to the Shield Advanced protection for a resource. Shield Advanced health-based
     * detection uses the health of your Amazon Web Services resource to improve responsiveness and accuracy in attack
     * detection and response.
     * </p>
     * <p>
     * You define the health check in Route 53 and then associate it with your Shield Advanced protection. For more
     * information, see <a href=
     * "https://docs.aws.amazon.com/waf/latest/developerguide/ddos-overview.html#ddos-advanced-health-check-option"
     * >Shield Advanced Health-Based Detection</a> in the <i>WAF Developer Guide</i>.
     * </p>
     *
     * @param associateHealthCheckRequest
     * @return A Java Future containing the result of the AssociateHealthCheck operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>LimitsExceededException Exception that indicates that the operation would exceed a limit.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>InvalidResourceException Exception that indicates that the resource is invalid. You might not have
     *         access to the resource, or the resource might not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.AssociateHealthCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/AssociateHealthCheck" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateHealthCheckResponse> associateHealthCheck(
            AssociateHealthCheckRequest associateHealthCheckRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateHealthCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateHealthCheck");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateHealthCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateHealthCheckResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateHealthCheckResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateHealthCheckRequest, AssociateHealthCheckResponse>()
                            .withOperationName("AssociateHealthCheck")
                            .withMarshaller(new AssociateHealthCheckRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateHealthCheckRequest));
            CompletableFuture<AssociateHealthCheckResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Initializes proactive engagement and sets the list of contacts for the Shield Response Team (SRT) to use. You
     * must provide at least one phone number in the emergency contact list.
     * </p>
     * <p>
     * After you have initialized proactive engagement using this call, to disable or enable proactive engagement, use
     * the calls <code>DisableProactiveEngagement</code> and <code>EnableProactiveEngagement</code>.
     * </p>
     * <note>
     * <p>
     * This call defines the list of email addresses and phone numbers that the SRT can use to contact you for
     * escalations to the SRT and to initiate proactive customer support.
     * </p>
     * <p>
     * The contacts that you provide in the request replace any contacts that were already defined. If you already have
     * contacts defined and want to use them, retrieve the list using <code>DescribeEmergencyContactSettings</code> and
     * then provide it to this call.
     * </p>
     * </note>
     *
     * @param associateProactiveEngagementDetailsRequest
     * @return A Java Future containing the result of the AssociateProactiveEngagementDetails operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.AssociateProactiveEngagementDetails
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/AssociateProactiveEngagementDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateProactiveEngagementDetailsResponse> associateProactiveEngagementDetails(
            AssociateProactiveEngagementDetailsRequest associateProactiveEngagementDetailsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateProactiveEngagementDetailsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateProactiveEngagementDetails");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateProactiveEngagementDetailsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, AssociateProactiveEngagementDetailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateProactiveEngagementDetailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateProactiveEngagementDetailsRequest, AssociateProactiveEngagementDetailsResponse>()
                            .withOperationName("AssociateProactiveEngagementDetails")
                            .withMarshaller(new AssociateProactiveEngagementDetailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(associateProactiveEngagementDetailsRequest));
            CompletableFuture<AssociateProactiveEngagementDetailsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables Shield Advanced for a specific Amazon Web Services resource. The resource can be an Amazon CloudFront
     * distribution, Elastic Load Balancing load balancer, Global Accelerator accelerator, Elastic IP Address, or an
     * Amazon Route 53 hosted zone.
     * </p>
     * <p>
     * You can add protection to only a single resource with each <code>CreateProtection</code> request. You can add
     * protection to multiple resources at once through the Shield Advanced console at <a
     * href="https://console.aws.amazon.com/wafv2/shieldv2#/">https://console.aws.amazon.com/wafv2/shieldv2#/</a>. For
     * more information see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/getting-started-ddos.html">Getting Started with
     * Shield Advanced</a> and <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/configure-new-protection.html">Adding Shield Advanced
     * protection to Amazon Web Services resources</a>.
     * </p>
     *
     * @param createProtectionRequest
     * @return A Java Future containing the result of the CreateProtection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidResourceException Exception that indicates that the resource is invalid. You might not have
     *         access to the resource, or the resource might not exist.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>LimitsExceededException Exception that indicates that the operation would exceed a limit.</li>
     *         <li>ResourceAlreadyExistsException Exception indicating the specified resource already exists. If
     *         available, this exception includes details in additional properties.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.CreateProtection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/CreateProtection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProtectionResponse> createProtection(CreateProtectionRequest createProtectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProtectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProtection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProtectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateProtectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProtectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProtectionRequest, CreateProtectionResponse>()
                            .withOperationName("CreateProtection")
                            .withMarshaller(new CreateProtectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createProtectionRequest));
            CompletableFuture<CreateProtectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a grouping of protected resources so they can be handled as a collective. This resource grouping improves
     * the accuracy of detection and reduces false positives.
     * </p>
     *
     * @param createProtectionGroupRequest
     * @return A Java Future containing the result of the CreateProtectionGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceAlreadyExistsException Exception indicating the specified resource already exists. If
     *         available, this exception includes details in additional properties.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>LimitsExceededException Exception that indicates that the operation would exceed a limit.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.CreateProtectionGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/CreateProtectionGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateProtectionGroupResponse> createProtectionGroup(
            CreateProtectionGroupRequest createProtectionGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createProtectionGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProtectionGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateProtectionGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateProtectionGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateProtectionGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateProtectionGroupRequest, CreateProtectionGroupResponse>()
                            .withOperationName("CreateProtectionGroup")
                            .withMarshaller(new CreateProtectionGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createProtectionGroupRequest));
            CompletableFuture<CreateProtectionGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Activates Shield Advanced for an account.
     * </p>
     * <p>
     * When you initally create a subscription, your subscription is set to be automatically renewed at the end of the
     * existing subscription period. You can change this by submitting an <code>UpdateSubscription</code> request.
     * </p>
     *
     * @param createSubscriptionRequest
     * @return A Java Future containing the result of the CreateSubscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceAlreadyExistsException Exception indicating the specified resource already exists. If
     *         available, this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.CreateSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/CreateSubscription" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSubscriptionResponse> createSubscription(CreateSubscriptionRequest createSubscriptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSubscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSubscriptionRequest, CreateSubscriptionResponse>()
                            .withOperationName("CreateSubscription")
                            .withMarshaller(new CreateSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createSubscriptionRequest));
            CompletableFuture<CreateSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an Shield Advanced <a>Protection</a>.
     * </p>
     *
     * @param deleteProtectionRequest
     * @return A Java Future containing the result of the DeleteProtection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DeleteProtection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DeleteProtection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProtectionResponse> deleteProtection(DeleteProtectionRequest deleteProtectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProtectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProtection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProtectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteProtectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProtectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProtectionRequest, DeleteProtectionResponse>()
                            .withOperationName("DeleteProtection")
                            .withMarshaller(new DeleteProtectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteProtectionRequest));
            CompletableFuture<DeleteProtectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified protection group.
     * </p>
     *
     * @param deleteProtectionGroupRequest
     * @return A Java Future containing the result of the DeleteProtectionGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DeleteProtectionGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DeleteProtectionGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteProtectionGroupResponse> deleteProtectionGroup(
            DeleteProtectionGroupRequest deleteProtectionGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteProtectionGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProtectionGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteProtectionGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteProtectionGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteProtectionGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteProtectionGroupRequest, DeleteProtectionGroupResponse>()
                            .withOperationName("DeleteProtectionGroup")
                            .withMarshaller(new DeleteProtectionGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteProtectionGroupRequest));
            CompletableFuture<DeleteProtectionGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the details of a DDoS attack.
     * </p>
     *
     * @param describeAttackRequest
     * @return A Java Future containing the result of the DescribeAttack operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>AccessDeniedException Exception that indicates the specified <code>AttackId</code> does not exist, or
     *         the requester does not have the appropriate permissions to access the <code>AttackId</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DescribeAttack
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeAttack" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAttackResponse> describeAttack(DescribeAttackRequest describeAttackRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAttackRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAttack");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAttackResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAttackResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAttackResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAttackRequest, DescribeAttackResponse>()
                            .withOperationName("DescribeAttack")
                            .withMarshaller(new DescribeAttackRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAttackRequest));
            CompletableFuture<DescribeAttackResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides information about the number and type of attacks Shield has detected in the last year for all resources
     * that belong to your account, regardless of whether you've defined Shield protections for them. This operation is
     * available to Shield customers as well as to Shield Advanced customers.
     * </p>
     * <p>
     * The operation returns data for the time range of midnight UTC, one year ago, to midnight UTC, today. For example,
     * if the current time is <code>2020-10-26 15:39:32 PDT</code>, equal to <code>2020-10-26 22:39:32 UTC</code>, then
     * the time range for the attack data returned is from <code>2019-10-26 00:00:00 UTC</code> to
     * <code>2020-10-26 00:00:00 UTC</code>.
     * </p>
     * <p>
     * The time range indicates the period covered by the attack statistics data items.
     * </p>
     *
     * @param describeAttackStatisticsRequest
     * @return A Java Future containing the result of the DescribeAttackStatistics operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DescribeAttackStatistics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeAttackStatistics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAttackStatisticsResponse> describeAttackStatistics(
            DescribeAttackStatisticsRequest describeAttackStatisticsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAttackStatisticsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAttackStatistics");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAttackStatisticsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAttackStatisticsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAttackStatisticsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAttackStatisticsRequest, DescribeAttackStatisticsResponse>()
                            .withOperationName("DescribeAttackStatistics")
                            .withMarshaller(new DescribeAttackStatisticsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAttackStatisticsRequest));
            CompletableFuture<DescribeAttackStatisticsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the current role and list of Amazon S3 log buckets used by the Shield Response Team (SRT) to access your
     * Amazon Web Services account while assisting with attack mitigation.
     * </p>
     *
     * @param describeDrtAccessRequest
     * @return A Java Future containing the result of the DescribeDRTAccess operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DescribeDRTAccess
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeDRTAccess" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDrtAccessResponse> describeDRTAccess(DescribeDrtAccessRequest describeDrtAccessRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDrtAccessRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDRTAccess");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDrtAccessResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDrtAccessResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDrtAccessResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDrtAccessRequest, DescribeDrtAccessResponse>()
                            .withOperationName("DescribeDRTAccess")
                            .withMarshaller(new DescribeDrtAccessRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeDrtAccessRequest));
            CompletableFuture<DescribeDrtAccessResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A list of email addresses and phone numbers that the Shield Response Team (SRT) can use to contact you if you
     * have proactive engagement enabled, for escalations to the SRT and to initiate proactive customer support.
     * </p>
     *
     * @param describeEmergencyContactSettingsRequest
     * @return A Java Future containing the result of the DescribeEmergencyContactSettings operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DescribeEmergencyContactSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeEmergencyContactSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEmergencyContactSettingsResponse> describeEmergencyContactSettings(
            DescribeEmergencyContactSettingsRequest describeEmergencyContactSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeEmergencyContactSettingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEmergencyContactSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEmergencyContactSettingsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeEmergencyContactSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEmergencyContactSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEmergencyContactSettingsRequest, DescribeEmergencyContactSettingsResponse>()
                            .withOperationName("DescribeEmergencyContactSettings")
                            .withMarshaller(new DescribeEmergencyContactSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeEmergencyContactSettingsRequest));
            CompletableFuture<DescribeEmergencyContactSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the details of a <a>Protection</a> object.
     * </p>
     *
     * @param describeProtectionRequest
     * @return A Java Future containing the result of the DescribeProtection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DescribeProtection
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeProtection" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeProtectionResponse> describeProtection(DescribeProtectionRequest describeProtectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProtectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProtection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeProtectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeProtectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeProtectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeProtectionRequest, DescribeProtectionResponse>()
                            .withOperationName("DescribeProtection")
                            .withMarshaller(new DescribeProtectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeProtectionRequest));
            CompletableFuture<DescribeProtectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the specification for the specified protection group.
     * </p>
     *
     * @param describeProtectionGroupRequest
     * @return A Java Future containing the result of the DescribeProtectionGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DescribeProtectionGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeProtectionGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeProtectionGroupResponse> describeProtectionGroup(
            DescribeProtectionGroupRequest describeProtectionGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeProtectionGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeProtectionGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeProtectionGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeProtectionGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeProtectionGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeProtectionGroupRequest, DescribeProtectionGroupResponse>()
                            .withOperationName("DescribeProtectionGroup")
                            .withMarshaller(new DescribeProtectionGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeProtectionGroupRequest));
            CompletableFuture<DescribeProtectionGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides details about the Shield Advanced subscription for an account.
     * </p>
     *
     * @param describeSubscriptionRequest
     * @return A Java Future containing the result of the DescribeSubscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DescribeSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DescribeSubscription" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSubscriptionResponse> describeSubscription(
            DescribeSubscriptionRequest describeSubscriptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSubscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSubscriptionRequest, DescribeSubscriptionResponse>()
                            .withOperationName("DescribeSubscription")
                            .withMarshaller(new DescribeSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeSubscriptionRequest));
            CompletableFuture<DescribeSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Disable the Shield Advanced automatic application layer DDoS mitigation feature for the resource. This stops
     * Shield Advanced from creating, verifying, and applying WAF rules for attacks that it detects for the resource.
     * </p>
     *
     * @param disableApplicationLayerAutomaticResponseRequest
     * @return A Java Future containing the result of the DisableApplicationLayerAutomaticResponse operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DisableApplicationLayerAutomaticResponse
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DisableApplicationLayerAutomaticResponse"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableApplicationLayerAutomaticResponseResponse> disableApplicationLayerAutomaticResponse(
            DisableApplicationLayerAutomaticResponseRequest disableApplicationLayerAutomaticResponseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disableApplicationLayerAutomaticResponseRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableApplicationLayerAutomaticResponse");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableApplicationLayerAutomaticResponseResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DisableApplicationLayerAutomaticResponseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableApplicationLayerAutomaticResponseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableApplicationLayerAutomaticResponseRequest, DisableApplicationLayerAutomaticResponseResponse>()
                            .withOperationName("DisableApplicationLayerAutomaticResponse")
                            .withMarshaller(new DisableApplicationLayerAutomaticResponseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(disableApplicationLayerAutomaticResponseRequest));
            CompletableFuture<DisableApplicationLayerAutomaticResponseResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes authorization from the Shield Response Team (SRT) to notify contacts about escalations to the SRT and to
     * initiate proactive customer support.
     * </p>
     *
     * @param disableProactiveEngagementRequest
     * @return A Java Future containing the result of the DisableProactiveEngagement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DisableProactiveEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DisableProactiveEngagement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisableProactiveEngagementResponse> disableProactiveEngagement(
            DisableProactiveEngagementRequest disableProactiveEngagementRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableProactiveEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableProactiveEngagement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisableProactiveEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisableProactiveEngagementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisableProactiveEngagementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisableProactiveEngagementRequest, DisableProactiveEngagementResponse>()
                            .withOperationName("DisableProactiveEngagement")
                            .withMarshaller(new DisableProactiveEngagementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disableProactiveEngagementRequest));
            CompletableFuture<DisableProactiveEngagementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the Shield Response Team's (SRT) access to the specified Amazon S3 bucket containing the logs that you
     * shared previously.
     * </p>
     *
     * @param disassociateDrtLogBucketRequest
     * @return A Java Future containing the result of the DisassociateDRTLogBucket operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>NoAssociatedRoleException The ARN of the role that you specifed does not exist.</li>
     *         <li>AccessDeniedForDependencyException In order to grant the necessary access to the Shield Response Team
     *         (SRT) the user submitting the request must have the <code>iam:PassRole</code> permission. This error
     *         indicates the user did not have the appropriate permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_use_passrole.html">Granting a User
     *         Permissions to Pass a Role to an Amazon Web Services Service</a>.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DisassociateDRTLogBucket
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DisassociateDRTLogBucket"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateDrtLogBucketResponse> disassociateDRTLogBucket(
            DisassociateDrtLogBucketRequest disassociateDrtLogBucketRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateDrtLogBucketRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateDRTLogBucket");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateDrtLogBucketResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateDrtLogBucketResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateDrtLogBucketResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateDrtLogBucketRequest, DisassociateDrtLogBucketResponse>()
                            .withOperationName("DisassociateDRTLogBucket")
                            .withMarshaller(new DisassociateDrtLogBucketRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateDrtLogBucketRequest));
            CompletableFuture<DisassociateDrtLogBucketResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the Shield Response Team's (SRT) access to your Amazon Web Services account.
     * </p>
     *
     * @param disassociateDrtRoleRequest
     * @return A Java Future containing the result of the DisassociateDRTRole operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DisassociateDRTRole
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DisassociateDRTRole" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateDrtRoleResponse> disassociateDRTRole(
            DisassociateDrtRoleRequest disassociateDrtRoleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateDrtRoleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateDRTRole");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateDrtRoleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateDrtRoleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateDrtRoleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateDrtRoleRequest, DisassociateDrtRoleResponse>()
                            .withOperationName("DisassociateDRTRole")
                            .withMarshaller(new DisassociateDrtRoleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateDrtRoleRequest));
            CompletableFuture<DisassociateDrtRoleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes health-based detection from the Shield Advanced protection for a resource. Shield Advanced health-based
     * detection uses the health of your Amazon Web Services resource to improve responsiveness and accuracy in attack
     * detection and response.
     * </p>
     * <p>
     * You define the health check in Route 53 and then associate or disassociate it with your Shield Advanced
     * protection. For more information, see <a href=
     * "https://docs.aws.amazon.com/waf/latest/developerguide/ddos-overview.html#ddos-advanced-health-check-option"
     * >Shield Advanced Health-Based Detection</a> in the <i>WAF Developer Guide</i>.
     * </p>
     *
     * @param disassociateHealthCheckRequest
     * @return A Java Future containing the result of the DisassociateHealthCheck operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>InvalidResourceException Exception that indicates that the resource is invalid. You might not have
     *         access to the resource, or the resource might not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.DisassociateHealthCheck
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/DisassociateHealthCheck"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateHealthCheckResponse> disassociateHealthCheck(
            DisassociateHealthCheckRequest disassociateHealthCheckRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateHealthCheckRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateHealthCheck");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateHealthCheckResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateHealthCheckResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateHealthCheckResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateHealthCheckRequest, DisassociateHealthCheckResponse>()
                            .withOperationName("DisassociateHealthCheck")
                            .withMarshaller(new DisassociateHealthCheckRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(disassociateHealthCheckRequest));
            CompletableFuture<DisassociateHealthCheckResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enable the Shield Advanced automatic application layer DDoS mitigation for the resource.
     * </p>
     * <note>
     * <p>
     * This feature is available for Amazon CloudFront distributions only.
     * </p>
     * </note>
     * <p>
     * This causes Shield Advanced to create, verify, and apply WAF rules for DDoS attacks that it detects for the
     * resource. Shield Advanced applies the rules in a Shield rule group inside the web ACL that you've associated with
     * the resource. For information about how automatic mitigation works and the requirements for using it, see <a
     * href=
     * "https://docs.aws.amazon.com/waf/latest/developerguide/ddos-advanced-automatic-app-layer-response.html">Shield
     * Advanced automatic application layer DDoS mitigation</a>.
     * </p>
     * <p>
     * Don't use this action to make changes to automatic mitigation settings when it's already enabled for a resource.
     * Instead, use <a>UpdateApplicationLayerAutomaticResponse</a>.
     * </p>
     * <p>
     * To use this feature, you must associate a web ACL with the protected resource. The web ACL must be created using
     * the latest version of WAF (v2). You can associate the web ACL through the Shield Advanced console at <a
     * href="https://console.aws.amazon.com/wafv2/shieldv2#/">https://console.aws.amazon.com/wafv2/shieldv2#/</a>. For
     * more information, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/getting-started-ddos.html">Getting Started with
     * Shield Advanced</a>.
     * </p>
     * <p>
     * You can also do this through the WAF console or the WAF API, but you must manage Shield Advanced automatic
     * mitigation through Shield Advanced. For information about WAF, see <a
     * href="https://docs.aws.amazon.com/waf/latest/developerguide/">WAF Developer Guide</a>.
     * </p>
     *
     * @param enableApplicationLayerAutomaticResponseRequest
     * @return A Java Future containing the result of the EnableApplicationLayerAutomaticResponse operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitsExceededException Exception that indicates that the operation would exceed a limit.</li>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.EnableApplicationLayerAutomaticResponse
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/EnableApplicationLayerAutomaticResponse"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableApplicationLayerAutomaticResponseResponse> enableApplicationLayerAutomaticResponse(
            EnableApplicationLayerAutomaticResponseRequest enableApplicationLayerAutomaticResponseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                enableApplicationLayerAutomaticResponseRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableApplicationLayerAutomaticResponse");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableApplicationLayerAutomaticResponseResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, EnableApplicationLayerAutomaticResponseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableApplicationLayerAutomaticResponseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableApplicationLayerAutomaticResponseRequest, EnableApplicationLayerAutomaticResponseResponse>()
                            .withOperationName("EnableApplicationLayerAutomaticResponse")
                            .withMarshaller(new EnableApplicationLayerAutomaticResponseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(enableApplicationLayerAutomaticResponseRequest));
            CompletableFuture<EnableApplicationLayerAutomaticResponseResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Authorizes the Shield Response Team (SRT) to use email and phone to notify contacts about escalations to the SRT
     * and to initiate proactive customer support.
     * </p>
     *
     * @param enableProactiveEngagementRequest
     * @return A Java Future containing the result of the EnableProactiveEngagement operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.EnableProactiveEngagement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/EnableProactiveEngagement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<EnableProactiveEngagementResponse> enableProactiveEngagement(
            EnableProactiveEngagementRequest enableProactiveEngagementRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableProactiveEngagementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableProactiveEngagement");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<EnableProactiveEngagementResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, EnableProactiveEngagementResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<EnableProactiveEngagementResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<EnableProactiveEngagementRequest, EnableProactiveEngagementResponse>()
                            .withOperationName("EnableProactiveEngagement")
                            .withMarshaller(new EnableProactiveEngagementRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(enableProactiveEngagementRequest));
            CompletableFuture<EnableProactiveEngagementResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the <code>SubscriptionState</code>, either <code>Active</code> or <code>Inactive</code>.
     * </p>
     *
     * @param getSubscriptionStateRequest
     * @return A Java Future containing the result of the GetSubscriptionState operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.GetSubscriptionState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/GetSubscriptionState" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSubscriptionStateResponse> getSubscriptionState(
            GetSubscriptionStateRequest getSubscriptionStateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSubscriptionStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSubscriptionState");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSubscriptionStateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSubscriptionStateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSubscriptionStateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSubscriptionStateRequest, GetSubscriptionStateResponse>()
                            .withOperationName("GetSubscriptionState")
                            .withMarshaller(new GetSubscriptionStateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSubscriptionStateRequest));
            CompletableFuture<GetSubscriptionStateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns all ongoing DDoS attacks or all DDoS attacks during a specified time period.
     * </p>
     *
     * @param listAttacksRequest
     * @return A Java Future containing the result of the ListAttacks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.ListAttacks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListAttacks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAttacksResponse> listAttacks(ListAttacksRequest listAttacksRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAttacksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAttacks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAttacksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListAttacksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAttacksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAttacksRequest, ListAttacksResponse>()
                            .withOperationName("ListAttacks").withMarshaller(new ListAttacksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAttacksRequest));
            CompletableFuture<ListAttacksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns all ongoing DDoS attacks or all DDoS attacks during a specified time period.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listAttacks(software.amazon.awssdk.services.shield.model.ListAttacksRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListAttacksPublisher publisher = client.listAttacksPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListAttacksPublisher publisher = client.listAttacksPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.shield.model.ListAttacksResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.shield.model.ListAttacksResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAttacks(software.amazon.awssdk.services.shield.model.ListAttacksRequest)} operation.</b>
     * </p>
     *
     * @param listAttacksRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.ListAttacks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListAttacks" target="_top">AWS API
     *      Documentation</a>
     */
    public ListAttacksPublisher listAttacksPaginator(ListAttacksRequest listAttacksRequest) {
        return new ListAttacksPublisher(this, applyPaginatorUserAgent(listAttacksRequest));
    }

    /**
     * <p>
     * Retrieves the <a>ProtectionGroup</a> objects for the account.
     * </p>
     *
     * @param listProtectionGroupsRequest
     * @return A Java Future containing the result of the ListProtectionGroups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>InvalidPaginationTokenException Exception that indicates that the <code>NextToken</code> specified in
     *         the request is invalid. Submit the request using the <code>NextToken</code> value that was returned in
     *         the prior response.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.ListProtectionGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListProtectionGroups" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProtectionGroupsResponse> listProtectionGroups(
            ListProtectionGroupsRequest listProtectionGroupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProtectionGroupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProtectionGroups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProtectionGroupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProtectionGroupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProtectionGroupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProtectionGroupsRequest, ListProtectionGroupsResponse>()
                            .withOperationName("ListProtectionGroups")
                            .withMarshaller(new ListProtectionGroupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listProtectionGroupsRequest));
            CompletableFuture<ListProtectionGroupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the <a>ProtectionGroup</a> objects for the account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProtectionGroups(software.amazon.awssdk.services.shield.model.ListProtectionGroupsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListProtectionGroupsPublisher publisher = client.listProtectionGroupsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListProtectionGroupsPublisher publisher = client.listProtectionGroupsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.shield.model.ListProtectionGroupsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.shield.model.ListProtectionGroupsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProtectionGroups(software.amazon.awssdk.services.shield.model.ListProtectionGroupsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listProtectionGroupsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>InvalidPaginationTokenException Exception that indicates that the <code>NextToken</code> specified in
     *         the request is invalid. Submit the request using the <code>NextToken</code> value that was returned in
     *         the prior response.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.ListProtectionGroups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListProtectionGroups" target="_top">AWS
     *      API Documentation</a>
     */
    public ListProtectionGroupsPublisher listProtectionGroupsPaginator(ListProtectionGroupsRequest listProtectionGroupsRequest) {
        return new ListProtectionGroupsPublisher(this, applyPaginatorUserAgent(listProtectionGroupsRequest));
    }

    /**
     * <p>
     * Lists all <a>Protection</a> objects for the account.
     * </p>
     *
     * @param listProtectionsRequest
     * @return A Java Future containing the result of the ListProtections operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>InvalidPaginationTokenException Exception that indicates that the <code>NextToken</code> specified in
     *         the request is invalid. Submit the request using the <code>NextToken</code> value that was returned in
     *         the prior response.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.ListProtections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListProtections" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListProtectionsResponse> listProtections(ListProtectionsRequest listProtectionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProtectionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProtections");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProtectionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProtectionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProtectionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProtectionsRequest, ListProtectionsResponse>()
                            .withOperationName("ListProtections")
                            .withMarshaller(new ListProtectionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listProtectionsRequest));
            CompletableFuture<ListProtectionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all <a>Protection</a> objects for the account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listProtections(software.amazon.awssdk.services.shield.model.ListProtectionsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListProtectionsPublisher publisher = client.listProtectionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListProtectionsPublisher publisher = client.listProtectionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.shield.model.ListProtectionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.shield.model.ListProtectionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listProtections(software.amazon.awssdk.services.shield.model.ListProtectionsRequest)} operation.</b>
     * </p>
     *
     * @param listProtectionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>InvalidPaginationTokenException Exception that indicates that the <code>NextToken</code> specified in
     *         the request is invalid. Submit the request using the <code>NextToken</code> value that was returned in
     *         the prior response.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.ListProtections
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListProtections" target="_top">AWS API
     *      Documentation</a>
     */
    public ListProtectionsPublisher listProtectionsPaginator(ListProtectionsRequest listProtectionsRequest) {
        return new ListProtectionsPublisher(this, applyPaginatorUserAgent(listProtectionsRequest));
    }

    /**
     * <p>
     * Retrieves the resources that are included in the protection group.
     * </p>
     *
     * @param listResourcesInProtectionGroupRequest
     * @return A Java Future containing the result of the ListResourcesInProtectionGroup operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>InvalidPaginationTokenException Exception that indicates that the <code>NextToken</code> specified in
     *         the request is invalid. Submit the request using the <code>NextToken</code> value that was returned in
     *         the prior response.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.ListResourcesInProtectionGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListResourcesInProtectionGroup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourcesInProtectionGroupResponse> listResourcesInProtectionGroup(
            ListResourcesInProtectionGroupRequest listResourcesInProtectionGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listResourcesInProtectionGroupRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourcesInProtectionGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourcesInProtectionGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResourcesInProtectionGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourcesInProtectionGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourcesInProtectionGroupRequest, ListResourcesInProtectionGroupResponse>()
                            .withOperationName("ListResourcesInProtectionGroup")
                            .withMarshaller(new ListResourcesInProtectionGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listResourcesInProtectionGroupRequest));
            CompletableFuture<ListResourcesInProtectionGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the resources that are included in the protection group.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listResourcesInProtectionGroup(software.amazon.awssdk.services.shield.model.ListResourcesInProtectionGroupRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListResourcesInProtectionGroupPublisher publisher = client.listResourcesInProtectionGroupPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.shield.paginators.ListResourcesInProtectionGroupPublisher publisher = client.listResourcesInProtectionGroupPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.shield.model.ListResourcesInProtectionGroupResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.shield.model.ListResourcesInProtectionGroupResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listResourcesInProtectionGroup(software.amazon.awssdk.services.shield.model.ListResourcesInProtectionGroupRequest)}
     * operation.</b>
     * </p>
     *
     * @param listResourcesInProtectionGroupRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>InvalidPaginationTokenException Exception that indicates that the <code>NextToken</code> specified in
     *         the request is invalid. Submit the request using the <code>NextToken</code> value that was returned in
     *         the prior response.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.ListResourcesInProtectionGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListResourcesInProtectionGroup"
     *      target="_top">AWS API Documentation</a>
     */
    public ListResourcesInProtectionGroupPublisher listResourcesInProtectionGroupPaginator(
            ListResourcesInProtectionGroupRequest listResourcesInProtectionGroupRequest) {
        return new ListResourcesInProtectionGroupPublisher(this, applyPaginatorUserAgent(listResourcesInProtectionGroupRequest));
    }

    /**
     * <p>
     * Gets information about Amazon Web Services tags for a specified Amazon Resource Name (ARN) in Shield.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidResourceException Exception that indicates that the resource is invalid. You might not have
     *         access to the resource, or the resource might not exist.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or updates tags for a resource in Shield.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidResourceException Exception that indicates that the resource is invalid. You might not have
     *         access to the resource, or the resource might not exist.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes tags from a resource in Shield.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidResourceException Exception that indicates that the resource is invalid. You might not have
     *         access to the resource, or the resource might not exist.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing Shield Advanced automatic application layer DDoS mitigation configuration for the specified
     * resource.
     * </p>
     *
     * @param updateApplicationLayerAutomaticResponseRequest
     * @return A Java Future containing the result of the UpdateApplicationLayerAutomaticResponse operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>InvalidOperationException Exception that indicates that the operation would not cause any change to
     *         occur.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.UpdateApplicationLayerAutomaticResponse
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/UpdateApplicationLayerAutomaticResponse"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateApplicationLayerAutomaticResponseResponse> updateApplicationLayerAutomaticResponse(
            UpdateApplicationLayerAutomaticResponseRequest updateApplicationLayerAutomaticResponseRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateApplicationLayerAutomaticResponseRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApplicationLayerAutomaticResponse");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateApplicationLayerAutomaticResponseResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateApplicationLayerAutomaticResponseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateApplicationLayerAutomaticResponseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateApplicationLayerAutomaticResponseRequest, UpdateApplicationLayerAutomaticResponseResponse>()
                            .withOperationName("UpdateApplicationLayerAutomaticResponse")
                            .withMarshaller(new UpdateApplicationLayerAutomaticResponseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(updateApplicationLayerAutomaticResponseRequest));
            CompletableFuture<UpdateApplicationLayerAutomaticResponseResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the details of the list of email addresses and phone numbers that the Shield Response Team (SRT) can use
     * to contact you if you have proactive engagement enabled, for escalations to the SRT and to initiate proactive
     * customer support.
     * </p>
     *
     * @param updateEmergencyContactSettingsRequest
     * @return A Java Future containing the result of the UpdateEmergencyContactSettings operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.UpdateEmergencyContactSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/UpdateEmergencyContactSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEmergencyContactSettingsResponse> updateEmergencyContactSettings(
            UpdateEmergencyContactSettingsRequest updateEmergencyContactSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateEmergencyContactSettingsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEmergencyContactSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEmergencyContactSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEmergencyContactSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEmergencyContactSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEmergencyContactSettingsRequest, UpdateEmergencyContactSettingsResponse>()
                            .withOperationName("UpdateEmergencyContactSettings")
                            .withMarshaller(new UpdateEmergencyContactSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateEmergencyContactSettingsRequest));
            CompletableFuture<UpdateEmergencyContactSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing protection group. A protection group is a grouping of protected resources so they can be
     * handled as a collective. This resource grouping improves the accuracy of detection and reduces false positives.
     * </p>
     *
     * @param updateProtectionGroupRequest
     * @return A Java Future containing the result of the UpdateProtectionGroup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.UpdateProtectionGroup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/UpdateProtectionGroup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProtectionGroupResponse> updateProtectionGroup(
            UpdateProtectionGroupRequest updateProtectionGroupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProtectionGroupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProtectionGroup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProtectionGroupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateProtectionGroupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProtectionGroupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProtectionGroupRequest, UpdateProtectionGroupResponse>()
                            .withOperationName("UpdateProtectionGroup")
                            .withMarshaller(new UpdateProtectionGroupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateProtectionGroupRequest));
            CompletableFuture<UpdateProtectionGroupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the details of an existing subscription. Only enter values for parameters you want to change. Empty
     * parameters are not updated.
     * </p>
     *
     * @param updateSubscriptionRequest
     * @return A Java Future containing the result of the UpdateSubscription operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException Exception that indicates that a problem occurred with the service
     *         infrastructure. You can retry the request.</li>
     *         <li>LockedSubscriptionException You are trying to update a subscription that has not yet completed the
     *         1-year commitment. You can change the <code>AutoRenew</code> parameter during the last 30 days of your
     *         subscription. This exception indicates that you are attempting to change <code>AutoRenew</code> prior to
     *         that period.</li>
     *         <li>ResourceNotFoundException Exception indicating the specified resource does not exist. If available,
     *         this exception includes details in additional properties.</li>
     *         <li>InvalidParameterException Exception that indicates that the parameters passed to the API are invalid.
     *         If available, this exception includes details in additional properties.</li>
     *         <li>OptimisticLockException Exception that indicates that the resource state has been modified by another
     *         client. Retrieve the resource and then retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ShieldException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ShieldAsyncClient.UpdateSubscription
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/shield-2016-06-02/UpdateSubscription" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSubscriptionResponse> updateSubscription(UpdateSubscriptionRequest updateSubscriptionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSubscriptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Shield");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSubscription");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSubscriptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSubscriptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateSubscriptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSubscriptionRequest, UpdateSubscriptionResponse>()
                            .withOperationName("UpdateSubscription")
                            .withMarshaller(new UpdateSubscriptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateSubscriptionRequest));
            CompletableFuture<UpdateSubscriptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ShieldException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OptimisticLockException")
                                .exceptionBuilderSupplier(OptimisticLockException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterException")
                                .exceptionBuilderSupplier(InvalidParameterException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidOperationException")
                                .exceptionBuilderSupplier(InvalidOperationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalErrorException")
                                .exceptionBuilderSupplier(InternalErrorException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoAssociatedRoleException")
                                .exceptionBuilderSupplier(NoAssociatedRoleException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedForDependencyException")
                                .exceptionBuilderSupplier(AccessDeniedForDependencyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LockedSubscriptionException")
                                .exceptionBuilderSupplier(LockedSubscriptionException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceException")
                                .exceptionBuilderSupplier(InvalidResourceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPaginationTokenException")
                                .exceptionBuilderSupplier(InvalidPaginationTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitsExceededException")
                                .exceptionBuilderSupplier(LimitsExceededException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends ShieldRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
