/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.timestreamwrite.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration that specifies an S3 location.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3Configuration implements SdkPojo, Serializable, ToCopyableBuilder<S3Configuration.Builder, S3Configuration> {
    private static final SdkField<String> BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BucketName").getter(getter(S3Configuration::bucketName)).setter(setter(Builder::bucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BucketName").build()).build();

    private static final SdkField<String> OBJECT_KEY_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ObjectKeyPrefix").getter(getter(S3Configuration::objectKeyPrefix))
            .setter(setter(Builder::objectKeyPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ObjectKeyPrefix").build()).build();

    private static final SdkField<String> ENCRYPTION_OPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EncryptionOption").getter(getter(S3Configuration::encryptionOptionAsString))
            .setter(setter(Builder::encryptionOption))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionOption").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(S3Configuration::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BUCKET_NAME_FIELD,
            OBJECT_KEY_PREFIX_FIELD, ENCRYPTION_OPTION_FIELD, KMS_KEY_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String bucketName;

    private final String objectKeyPrefix;

    private final String encryptionOption;

    private final String kmsKeyId;

    private S3Configuration(BuilderImpl builder) {
        this.bucketName = builder.bucketName;
        this.objectKeyPrefix = builder.objectKeyPrefix;
        this.encryptionOption = builder.encryptionOption;
        this.kmsKeyId = builder.kmsKeyId;
    }

    /**
     * <p>
     * The bucket name of the customer S3 bucket.
     * </p>
     * 
     * @return The bucket name of the customer S3 bucket.
     */
    public final String bucketName() {
        return bucketName;
    }

    /**
     * <p>
     * The object key preview for the customer S3 location.
     * </p>
     * 
     * @return The object key preview for the customer S3 location.
     */
    public final String objectKeyPrefix() {
        return objectKeyPrefix;
    }

    /**
     * <p>
     * The encryption option for the customer S3 location. Options are S3 server-side encryption with an S3 managed key
     * or Amazon Web Services managed key.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionOption}
     * will return {@link S3EncryptionOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionOptionAsString}.
     * </p>
     * 
     * @return The encryption option for the customer S3 location. Options are S3 server-side encryption with an S3
     *         managed key or Amazon Web Services managed key.
     * @see S3EncryptionOption
     */
    public final S3EncryptionOption encryptionOption() {
        return S3EncryptionOption.fromValue(encryptionOption);
    }

    /**
     * <p>
     * The encryption option for the customer S3 location. Options are S3 server-side encryption with an S3 managed key
     * or Amazon Web Services managed key.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #encryptionOption}
     * will return {@link S3EncryptionOption#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #encryptionOptionAsString}.
     * </p>
     * 
     * @return The encryption option for the customer S3 location. Options are S3 server-side encryption with an S3
     *         managed key or Amazon Web Services managed key.
     * @see S3EncryptionOption
     */
    public final String encryptionOptionAsString() {
        return encryptionOption;
    }

    /**
     * <p>
     * The KMS key ID for the customer S3 location when encrypting with an Amazon Web Services managed key.
     * </p>
     * 
     * @return The KMS key ID for the customer S3 location when encrypting with an Amazon Web Services managed key.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bucketName());
        hashCode = 31 * hashCode + Objects.hashCode(objectKeyPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionOptionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3Configuration)) {
            return false;
        }
        S3Configuration other = (S3Configuration) obj;
        return Objects.equals(bucketName(), other.bucketName()) && Objects.equals(objectKeyPrefix(), other.objectKeyPrefix())
                && Objects.equals(encryptionOptionAsString(), other.encryptionOptionAsString())
                && Objects.equals(kmsKeyId(), other.kmsKeyId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3Configuration").add("BucketName", bucketName()).add("ObjectKeyPrefix", objectKeyPrefix())
                .add("EncryptionOption", encryptionOptionAsString()).add("KmsKeyId", kmsKeyId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BucketName":
            return Optional.ofNullable(clazz.cast(bucketName()));
        case "ObjectKeyPrefix":
            return Optional.ofNullable(clazz.cast(objectKeyPrefix()));
        case "EncryptionOption":
            return Optional.ofNullable(clazz.cast(encryptionOptionAsString()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<S3Configuration, T> g) {
        return obj -> g.apply((S3Configuration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3Configuration> {
        /**
         * <p>
         * The bucket name of the customer S3 bucket.
         * </p>
         * 
         * @param bucketName
         *        The bucket name of the customer S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketName(String bucketName);

        /**
         * <p>
         * The object key preview for the customer S3 location.
         * </p>
         * 
         * @param objectKeyPrefix
         *        The object key preview for the customer S3 location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder objectKeyPrefix(String objectKeyPrefix);

        /**
         * <p>
         * The encryption option for the customer S3 location. Options are S3 server-side encryption with an S3 managed
         * key or Amazon Web Services managed key.
         * </p>
         * 
         * @param encryptionOption
         *        The encryption option for the customer S3 location. Options are S3 server-side encryption with an S3
         *        managed key or Amazon Web Services managed key.
         * @see S3EncryptionOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3EncryptionOption
         */
        Builder encryptionOption(String encryptionOption);

        /**
         * <p>
         * The encryption option for the customer S3 location. Options are S3 server-side encryption with an S3 managed
         * key or Amazon Web Services managed key.
         * </p>
         * 
         * @param encryptionOption
         *        The encryption option for the customer S3 location. Options are S3 server-side encryption with an S3
         *        managed key or Amazon Web Services managed key.
         * @see S3EncryptionOption
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see S3EncryptionOption
         */
        Builder encryptionOption(S3EncryptionOption encryptionOption);

        /**
         * <p>
         * The KMS key ID for the customer S3 location when encrypting with an Amazon Web Services managed key.
         * </p>
         * 
         * @param kmsKeyId
         *        The KMS key ID for the customer S3 location when encrypting with an Amazon Web Services managed key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);
    }

    static final class BuilderImpl implements Builder {
        private String bucketName;

        private String objectKeyPrefix;

        private String encryptionOption;

        private String kmsKeyId;

        private BuilderImpl() {
        }

        private BuilderImpl(S3Configuration model) {
            bucketName(model.bucketName);
            objectKeyPrefix(model.objectKeyPrefix);
            encryptionOption(model.encryptionOption);
            kmsKeyId(model.kmsKeyId);
        }

        public final String getBucketName() {
            return bucketName;
        }

        public final void setBucketName(String bucketName) {
            this.bucketName = bucketName;
        }

        @Override
        public final Builder bucketName(String bucketName) {
            this.bucketName = bucketName;
            return this;
        }

        public final String getObjectKeyPrefix() {
            return objectKeyPrefix;
        }

        public final void setObjectKeyPrefix(String objectKeyPrefix) {
            this.objectKeyPrefix = objectKeyPrefix;
        }

        @Override
        public final Builder objectKeyPrefix(String objectKeyPrefix) {
            this.objectKeyPrefix = objectKeyPrefix;
            return this;
        }

        public final String getEncryptionOption() {
            return encryptionOption;
        }

        public final void setEncryptionOption(String encryptionOption) {
            this.encryptionOption = encryptionOption;
        }

        @Override
        public final Builder encryptionOption(String encryptionOption) {
            this.encryptionOption = encryptionOption;
            return this;
        }

        @Override
        public final Builder encryptionOption(S3EncryptionOption encryptionOption) {
            this.encryptionOption(encryptionOption == null ? null : encryptionOption.toString());
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        @Override
        public S3Configuration build() {
            return new S3Configuration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
