// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
package software.amazon.encryption.s3.internal;

import static software.amazon.encryption.s3.internal.ApiNameVersion.API_NAME_INTERCEPTOR;

import java.security.SecureRandom;
import java.util.concurrent.CompletableFuture;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import software.amazon.awssdk.core.async.AsyncRequestBody;
import software.amazon.awssdk.services.s3.S3AsyncClient;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.PutObjectResponse;
import software.amazon.encryption.s3.S3EncryptionClientException;
import software.amazon.encryption.s3.algorithms.AlgorithmSuite;
import software.amazon.encryption.s3.materials.CryptographicMaterialsManager;
import software.amazon.encryption.s3.materials.EncryptionMaterials;
import software.amazon.encryption.s3.materials.EncryptionMaterialsRequest;

public class PutEncryptedObjectPipeline {

    final private S3AsyncClient _s3AsyncClient;
    final private CryptographicMaterialsManager _cryptoMaterialsManager;
    final private AsyncContentEncryptionStrategy _asyncContentEncryptionStrategy;
    final private ContentMetadataEncodingStrategy _contentMetadataEncodingStrategy;
    final private AlgorithmSuite _encryptionAlgorithm;

    public static Builder builder() {
        return new Builder();
    }

    private PutEncryptedObjectPipeline(Builder builder) {
        this._s3AsyncClient = builder._s3AsyncClient;
        this._cryptoMaterialsManager = builder._cryptoMaterialsManager;
        this._asyncContentEncryptionStrategy = builder._asyncContentEncryptionStrategy;
        this._contentMetadataEncodingStrategy = builder._contentMetadataEncodingStrategy;
        this._encryptionAlgorithm = builder._encryptionAlgorithm;
    }

    public CompletableFuture<PutObjectResponse> putObject(PutObjectRequest request, AsyncRequestBody requestBody) {
        final Long contentLength;
        if (request.contentLength() != null) {
            if (requestBody.contentLength().isPresent() && !request.contentLength().equals(requestBody.contentLength().get())) {
                // if the contentLength values do not match, throw an exception, since we don't know which is correct
                throw new S3EncryptionClientException("The contentLength provided in the request object MUST match the " +
                        "contentLength in the request body");
            } else if (!requestBody.contentLength().isPresent()) {
                // no contentLength in request body, use the one in request
                contentLength = request.contentLength();
            } else {
                // only remaining case is when the values match, so either works here
                contentLength = request.contentLength();
            }
        } else {
            contentLength = requestBody.contentLength().orElseThrow(() -> new S3EncryptionClientException("Unbounded streams are currently not supported."));
        }

        if (contentLength > _encryptionAlgorithm.cipherMaxContentLengthBytes()) {
            throw new S3EncryptionClientException("The contentLength of the object you are attempting to encrypt exceeds" +
                    "the maximum length allowed for GCM encryption.");
        }

        EncryptionMaterialsRequest encryptionMaterialsRequest = EncryptionMaterialsRequest.builder()
                .s3Request(request)
                .plaintextLength(contentLength)
                //= specification/s3-encryption/encryption.md#content-encryption
                //# The S3EC MUST use the encryption algorithm configured during [client](./client.md) initialization.
                .encryptionAlgorithm(_encryptionAlgorithm)
                .build();

        EncryptionMaterials materials = _cryptoMaterialsManager.getEncryptionMaterials(encryptionMaterialsRequest);
        if (materials == null) {
            throw new S3EncryptionClientException("Encryption materials cannot be null during content encryption. " +
                    "This may be caused by a misconfigured custom CMM implementation, or " +
                    "a suppressed exception from CMM invocation due to a network failure.");
        }

        EncryptedContent encryptedContent = _asyncContentEncryptionStrategy.encryptContent(materials, requestBody);

        final byte[] contentIV = materials.algorithmSuite().isCommitting() ? materials.messageId() : materials.iv();
        PutObjectRequest modifiedRequest = _contentMetadataEncodingStrategy.encodeMetadata(materials, contentIV, request);
        PutObjectRequest encryptedPutRequest = modifiedRequest.toBuilder()
                .overrideConfiguration(API_NAME_INTERCEPTOR)
                .contentLength(encryptedContent.getCiphertextLength())
                .build();
        return _s3AsyncClient.putObject(encryptedPutRequest, encryptedContent.getAsyncCiphertext());
    }

    public static class Builder {
        private S3AsyncClient _s3AsyncClient;
        private CryptographicMaterialsManager _cryptoMaterialsManager;
        private SecureRandom _secureRandom;
        private AsyncContentEncryptionStrategy _asyncContentEncryptionStrategy;
        private InstructionFileConfig _instructionFileConfig;
        private ContentMetadataEncodingStrategy _contentMetadataEncodingStrategy;
        private AlgorithmSuite _encryptionAlgorithm;

        private Builder() {
        }

        /**
         * Note that this does NOT create a defensive clone of S3Client. Any modifications made to the wrapped
         * S3Client will be reflected in this Builder.
         */
        @SuppressFBWarnings(value = "EI_EXPOSE_REP2", justification = "Pass mutability into wrapping client")
        public Builder s3AsyncClient(S3AsyncClient s3AsyncClient) {
            this._s3AsyncClient = s3AsyncClient;
            return this;
        }

        public Builder cryptoMaterialsManager(CryptographicMaterialsManager cryptoMaterialsManager) {
            this._cryptoMaterialsManager = cryptoMaterialsManager;
            return this;
        }

        public Builder secureRandom(SecureRandom secureRandom) {
            this._secureRandom = secureRandom;
            return this;
        }

        public Builder instructionFileConfig(InstructionFileConfig instructionFileConfig) {
            this._instructionFileConfig = instructionFileConfig;
            return this;
        }

        public Builder encryptionAlgorithm(AlgorithmSuite encryptionAlgorithm) {
            this._encryptionAlgorithm = encryptionAlgorithm;
            return this;
        }

        public PutEncryptedObjectPipeline build() {
            // Default to AesGcm since it is the only active (non-legacy) content encryption strategy
            if (_asyncContentEncryptionStrategy == null) {
                _asyncContentEncryptionStrategy = StreamingAesGcmContentStrategy
                        .builder()
                        .secureRandom(_secureRandom)
                        .build();
            }
            if(_instructionFileConfig == null) {
                _instructionFileConfig = InstructionFileConfig.builder().build();
            }
            _contentMetadataEncodingStrategy = new ContentMetadataEncodingStrategy(_instructionFileConfig);

            return new PutEncryptedObjectPipeline(this);
        }
    }
}
