/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the "License").  You may not use this file except 
 * in compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt or 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html. 
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * HEADER in each file and include the License file at 
 * glassfish/bootstrap/legal/CDDLv1.0.txt.  If applicable, 
 * add the following below this CDDL HEADER, with the 
 * fields enclosed by brackets "[]" replaced with your 
 * own identifying information: Portions Copyright [yyyy] 
 * [name of copyright owner]
 */
// Copyright (c) 1998, 2007, Oracle. All rights reserved.  
package oracle.toplink.essentials.internal.parsing;

import java.util.Iterator;
import java.util.List;
import java.util.ArrayList;

import oracle.toplink.essentials.expressions.*;
import oracle.toplink.essentials.internal.expressions.ConstantExpression;
import oracle.toplink.essentials.internal.queryframework.ReportItem;
import oracle.toplink.essentials.queryframework.ReportQuery;

/**
 * INTERNAL
 * <p><b>Purpose</b>: Represent an EXISTS subquery.
 */
public class ExistsNode extends Node {

    /** True in case of a NOT EXISTS (...) query. */
    private boolean notIndicated = false;

    /**
     * Return a new ExistsNode.
     */
    public ExistsNode() {
        super();
    }

    /**
     * INTERNAL
     * Validate node and calculate its type.
     * Change subquery SELECT clause.
     */
    public void validate(ParseTreeContext context) {
        if (left != null) {
            
            // change SELECT clause of subquery
            SubqueryNode subqueryNode = (SubqueryNode)getLeft();
            // validate changed subquery
            subqueryNode.validate(context);

            TypeHelper typeHelper = context.getTypeHelper();
            setType(typeHelper.getBooleanType());
        }
    }

    /**
     * INTERNAL
     * Generate the TopLink expression for this node
     */
    public Expression generateExpression(GenerationContext context) {
        SubqueryNode subqueryNode = (SubqueryNode)getLeft();
        ReportQuery reportQuery = subqueryNode.getReportQuery(context);
        // Replace the SELECT clause of the exists subquery by SELECT 1 to
        // avoid problems with databases not supporting mutiple columns in the
        // subquery SELECT clause in SQL.
        // The original select clause expressions might include relationship
        // navigations which should result in FK joins in the generated SQL,
        // e.g. ... EXISTS (SELECT o.customer FROM Order o ...). Add the
        // select clause expressions as non fetch join attributes to the
        // ReportQuery representing the subquery. This make sure the FK joins
        // get generated.  
        List items = reportQuery.getItems();
        for (Iterator i = items.iterator(); i.hasNext();) {
            ReportItem item = (ReportItem)i.next();
            Expression expr = item.getAttributeExpression();
            reportQuery.addNonFetchJoinedAttribute(expr);
        }
        reportQuery.clearItems();
        Expression one = new ConstantExpression(new Integer(1), new ExpressionBuilder());
        reportQuery.addItem("one", one);
        reportQuery.dontUseDistinct();
        Expression expr = context.getBaseExpression();
        return notIndicated() ? expr.notExists(reportQuery) : 
            expr.exists(reportQuery);
    }

    /**
     * INTERNAL
     * Indicate if a NOT was found in the WHERE clause.
     * Examples: WHERE ... NOT EXISTS(...)
     */
    public void indicateNot() {
        notIndicated = true;
    }

    public boolean notIndicated() {
        return notIndicated;
    }

}
